package com.xebialabs.xlrelease.actors.sharding

import com.xebialabs.xlrelease.actors.ReleaseActor
import com.xebialabs.xlrelease.actors.extension.AddressExtension
import com.xebialabs.xlrelease.actors.sharding.ReleaseShardingMessages.ReleaseAction
import com.xebialabs.xlrelease.actors.sharding.ReleaseWatcherActorMessages._
import com.xebialabs.xlrelease.config.XlrConfig
import grizzled.slf4j.Logging
import org.apache.pekko.actor.{Actor, ActorRef, Props, Terminated}
import org.apache.pekko.cluster.pubsub.DistributedPubSubMediator.{Publish, Subscribe}

object ReleaseWatcherActorMessages {

  /**
   * Register actor for the watch.
   */
  case class WatchReleaseActor(releaseId: String, actorRef: ActorRef)

  /**
   * Cancels watch of the actor for particular release.
   */
  case class UnWatchReleaseActor(actorRef: ActorRef)

  /**
   * Share the state of [[ReleaseWatcherActor]] with its other instances.
   */
  case class ReleaseWatcherState(state: Map[ActorRef, String])

  /**
   * Notify other instances of new release watcher.
   */
  case class ReleaseWatcherCreated(actorRef: ActorRef)

  /**
    * Check that watchee was not unwatched, and if watchee is still in our state
    * then activate it
    */
  case class CheckWatcheeAndActivate(watchee: ActorRef)

}

object ReleaseWatcherActor {

  val name = "release-watcher"

  val releasesTopic = "ReleaseWatcherActor-releases-events"

  val watcherTopic = "ReleaseWatcherActor-watcher-events"

  def props(shardActor: ActorRef, mediator: ActorRef, xlrConfig: XlrConfig) = Props(new ReleaseWatcherActor(shardActor, mediator, xlrConfig))

}

/**
 * This actor is supposed to run on all the nodes of the cluster and watch release actors and reactivate them if necessary.
 * It has been done only when XLR is running in clustered mode because otherwise release actors will be restarted by the standard supervision mechanism.
 */
class ReleaseWatcherActor(shardActor: ActorRef, mediator: ActorRef, xlrConfig: XlrConfig) extends Actor with Logging {

  private var state: Map[ActorRef, String] = Map()

  private val durationsConfig = xlrConfig.durations

  override def preStart(): Unit = {
    super.preStart()
    logger.info(s"Starting ${classOf[ReleaseWatcherActor].getSimpleName} on ${AddressExtension(context.system).address} at ${self.path}")

    mediator ! Subscribe(ReleaseWatcherActor.releasesTopic, self)
    mediator ! Subscribe(ReleaseWatcherActor.watcherTopic, self)

    import context.dispatcher

    context.system.scheduler.scheduleOnce(
      durationsConfig.newWatcherPublishDelay,
      mediator,
      Publish(ReleaseWatcherActor.watcherTopic, ReleaseWatcherCreated(self))
    )
  }

  override def receive: Receive = {

    case WatchReleaseActor(releaseId, watchee) =>
      state += (watchee -> releaseId)
      context.watch(watchee)
      logger.debug(s"Registered new watchee: $watchee. Total watchees amount: [${state.size}].")

    case UnWatchReleaseActor(watchee) if !state.contains(watchee) =>
      logger.warn(s"Can not unwatch $watchee because it is not currently watched")

    case UnWatchReleaseActor(watchee) =>
      state -= watchee
      context.unwatch(watchee)
      logger.debug(s"Unregistered a watchee: $watchee. Total watchees amount: [${state.size}].")

    case ReleaseWatcherState(newState) if sender() != self =>
      logger.debug(s"Received [${newState.size}] new watchees.")
      newState.keySet.foreach(context.watch)
      state ++= newState

    case ReleaseWatcherCreated(newWatcher) =>
      logger.debug(s"Sending current state of [${state.size}] watchees to $newWatcher")
      newWatcher ! ReleaseWatcherState(state)

    case Terminated(watchee) =>
      // since ReleaseSupervisorActor is not waiting for any grace period after sending an
      // UnWatchReleaseActor message to us, and then Passivate to the Shard immediately,
      // it might happen, that two messages will come in wrong order. Terminated will come
      // first and UnWatchReleaseActor will come next.
      // This is handled in the following manner:
      // Once we get Terminated we schedule CheckWatcheeAndActivate after another grace period
      // If UnWatchReleaseActor will come after Terminated, then we'll remove the Watchee
      // from our state
      // Once scheduled CheckWatcheeAndActivate will come, we'll check if Watchee is still in
      // our state, and if not - then it means that UnWatchReleaseActor finally was sent to us
      // If Watchee is still in our state - then we should re-activate it.
      // If by some chance that release actor will be activated earlier, then we'll just send
      // another activate, which does not hurt. We are safe to remove old Watchee from our state,
      // because if release actor was reactivated, then it will be new Actor anyway, so
      // we don't need to watch old one any more
      logger.debug(s"got Terminated from $watchee, and ${if (state.contains(watchee)) { "we have this actor in our state"} else {"we do not have this actor in our state"}}")
      state.get(watchee).foreach((releaseId: String) => activateReleaseAfterDelay(watchee))

    case CheckWatcheeAndActivate(watchee) =>
      if (state.contains(watchee)) {
        logger.debug(s"Death watched termination of $watchee. Activating new actor for release ${state(watchee)}. Total watchees left: [${state.size}].")
        shardActor ! ReleaseAction(state(watchee), ReleaseActor.Activate)

        state -= watchee
        context.unwatch(watchee)
      } else {
        logger.debug(s"Death watched termination of $watchee, but later the UnWatchReleaseActor message came due to message delivery misorder, so no action is taken.")
      }

  }

  private def activateReleaseAfterDelay(watchee: ActorRef): Unit = {
    import context.dispatcher

    context.system.scheduler.scheduleOnce(
      durationsConfig.releaseActivationDelay,
      self,
      CheckWatcheeAndActivate(watchee)
    )

  }
}
