package com.xebialabs.xlrelease.principaldata;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Collections;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.ldap.core.DirContextOperations;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.ldap.authentication.AbstractLdapAuthenticator;
import org.springframework.security.ldap.authentication.BindAuthenticator;
import org.springframework.security.ldap.authentication.LdapAuthenticationProvider;
import org.springframework.security.ldap.search.LdapUserSearch;
import org.springframework.security.ldap.userdetails.LdapAuthoritiesPopulator;

import static com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank;
import static com.xebialabs.xlrelease.principaldata.LdapDataHelper.readStringAttribute;
import static com.xebialabs.xlrelease.principaldata.UserData.NOT_FOUND;


class LdapUserDataProvider implements UserDataProvider {
    private static final Logger logger = LoggerFactory.getLogger(LdapUserDataProvider.class);

    private final String mailAttribute = "mail";

    private final String displayNameAttribute = "displayName";

    private LdapUserSearch userSearch;

    private LdapAuthoritiesPopulator authoritiesPopulator;

    private String externalIdAttribute;

    public LdapUserDataProvider(LdapAuthenticationProvider authProvider, String externalIdAttribute) {
        try {
            BindAuthenticator authenticator = invokePrivateMethod(authProvider, LdapAuthenticationProvider.class, "getAuthenticator", BindAuthenticator.class);
            this.userSearch = invokePrivateMethod(authenticator, AbstractLdapAuthenticator.class, "getUserSearch", LdapUserSearch.class);
            this.authoritiesPopulator = invokePrivateMethod(authProvider, LdapAuthenticationProvider.class, "getAuthoritiesPopulator", LdapAuthoritiesPopulator.class);
            this.externalIdAttribute = externalIdAttribute;
        } catch (Exception ex) {
            logger.warn("Error constructing LdapDataProviderBase", ex);
        }
    }

    public LdapUserDataProvider(LdapUserSearch userSearch, LdapAuthoritiesPopulator authoritiesPopulator) {
        this.userSearch = userSearch;
        this.authoritiesPopulator = authoritiesPopulator;
    }

    private static <T, R> R invokePrivateMethod(Object target, Class<T> targetClass, String methodName, Class<R> returnType) throws Exception {
        Method method = targetClass.getDeclaredMethod(methodName);
        method.setAccessible(true);
        return returnType.cast(method.invoke(target));
    }

    @Override
    public UserData getUserData(String username) {
        return searchForUserData(username).map(data -> {
            String email = readStringAttribute(username, data, mailAttribute);
            String fullName = readStringAttribute(username, data, displayNameAttribute);
            String externalId = null;
            if (isNotBlank(externalIdAttribute)) {
                externalId = readStringAttribute(username, data, externalIdAttribute);
            }
            logger.info("Email: {} displayName: {} and externalId: {} have been read from LDAP", email, fullName, externalId);
            return new UserData(email, fullName, externalId);
        }).orElse(NOT_FOUND);
    }

    @Override
    public Collection<? extends GrantedAuthority> getAuthorities(final String username) {
        return searchForUserData(username)
                .map(data -> authoritiesPopulator.getGrantedAuthorities(data, username))
                .orElse(Collections.emptyList());
    }

    private Optional<DirContextOperations> searchForUserData(String username) {
        try {
            return Optional.ofNullable(userSearch.searchForUser(username));
        } catch (UsernameNotFoundException e) {
            logger.info("User with username: {} not found", username);
        } catch (Exception e) {
            logger.warn("Error accessing LDAP server", e);
        }
        return Optional.empty();
    }
}
