package com.xebialabs.deployit.plumbing;

import com.google.common.annotations.VisibleForTesting;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlrelease.api.v1.views.ExplodedDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xltype.serialization.json.JsonWriter;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.ext.MessageBodyWriter;
import jakarta.ws.rs.ext.Provider;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;
import java.lang.annotation.Annotation;

import static com.xebialabs.deployit.plumbing.DescriptorJsonWriter.descriptorProperties;
import static com.xebialabs.deployit.plumbing.DescriptorJsonWriter.writeSuperTypes;
import static com.xebialabs.deployit.plumbing.PropertyDescriptorJsonWriter.writePropertyDescriptor;
import static java.nio.charset.StandardCharsets.UTF_8;

@Component
@Provider
@Produces({MediaType.APPLICATION_JSON})
public class ExplodedDescriptorJsonWriter implements MessageBodyWriter<ExplodedDescriptor> {

    @Override
    public boolean isWriteable(final Class<?> type, final java.lang.reflect.Type genericType, final Annotation[] annotations, final MediaType mediaType) {
        return true;
    }

    @Override
    public void writeTo(final ExplodedDescriptor exploded, final Class<?> type, final java.lang.reflect.Type genericType, final Annotation[] annotations, final MediaType mediaType, final MultivaluedMap<String, Object> httpHeaders, final OutputStream entityStream) throws IOException, WebApplicationException {
        entityStream.write(toJson(exploded).getBytes(UTF_8));
    }

    @VisibleForTesting
    String toJson(ExplodedDescriptor exploded) {
        StringWriter stringWriter = new StringWriter();
        JsonWriter writer = new JsonWriter(stringWriter);
        writeJson(exploded, writer);
        return stringWriter.toString();
    }

    public void writeJson(final ExplodedDescriptor exploded, final JsonWriter writer) {
        var descriptor = exploded.getDescriptor();
        writer.object();
        descriptorProperties(writer, descriptor);
        writeExplodedPropertyDescriptors(writer, descriptor);
        writeSuperTypes(writer, descriptor);
        writer.endObject();
    }

    protected static void writeExplodedPropertyDescriptors(JsonWriter writer, Descriptor descriptor) {
        writer.key("properties");
        writer.array();
        writeExplodedProperty(
                writer, descriptor.getType(), "id", PropertyKind.STRING, true, true);
        writeExplodedProperty(
                writer, descriptor.getType(), "type", PropertyKind.STRING, true, true);
        writeExplodedProperty(
                writer, descriptor.getType(), "$createdBy", PropertyKind.STRING, false, false);
        writeExplodedProperty(
                writer, descriptor.getType(), "$createdAt", PropertyKind.DATE, false, false);
        writeExplodedProperty(
                writer, descriptor.getType(), "$lastModifiedBy", PropertyKind.STRING, false, false);
        writeExplodedProperty(
                writer, descriptor.getType(), "$lastModifiedAt", PropertyKind.DATE, false, false);
        writeExplodedProperty(
                writer, descriptor.getType(), "$scmTraceabilityDataId", PropertyKind.STRING, false, false);
        for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
            writePropertyDescriptor(writer, pd);
        }

        writer.endArray();
    }

    static void writeExplodedProperty(
            JsonWriter writer, Type rootType, String name, PropertyKind kind, boolean required, boolean readOnly) {
        writer.object();
        writer.key("name").value(name);
        writer.key("fqn").value(rootType + "." + name);
        writer.key("kind").value(kind.name());
        writer.key("required").value(required);
        writer.key("readonly").value(readOnly);
        writer.key("hidden").value(true);
        writer.key("asContainment").value(false);
        writer.key("category").value(null);
        writer.key("description").value(name);
        writer.key("label").value(name);
        writer.key("inspection").value(false);
        writer.key("nested").value(false);
        writer.key("password").value(false);
        writer.key("referencedType").value(null);
        writer.key("requiredInspection").value(false);
        writer.key("size").value(Property.Size.DEFAULT);
        writer.key("transient").value(false);
        writer.key("default").value(null);
        writer.endObject();
    }
}
