package com.xebialabs.deployit.plumbing;

import java.io.IOException;
import java.util.Date;
import java.util.UUID;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.ext.ExceptionMapper;
import jakarta.ws.rs.ext.Provider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.Exceptions;
import com.xebialabs.deployit.Sanitizer;
import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.core.rest.resteasy.PathInterceptor;

import static com.xebialabs.deployit.ClosingExceptionMapper.X_EXCEPTION_TYPE;
import static com.xebialabs.deployit.ClosingExceptionMapper.X_PATH;

@Provider
@Component
public class IOExceptionMapper implements ExceptionMapper<IOException> {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    private final boolean hideInternals;

    public IOExceptionMapper(ServerConfiguration serverConfiguration) {
        this.hideInternals = serverConfiguration.isHideInternals();
    }

    @Override
    public Response toResponse(IOException exception) {
        String content = Exceptions.getAllMessages(exception);
        Response handled = checkWhetherWeHideInternals(exception, content);
        if (handled != null) {
            return handled;
        } else {
            final Response.ResponseBuilder builder = Response.status(Response.Status.BAD_REQUEST).type(MediaType.TEXT_PLAIN);
            builder.header(X_EXCEPTION_TYPE, exception.getClass().getName());
            builder.header(X_PATH, PathInterceptor.PATH.get());
            builder.header("Date", new Date());
            builder.entity(Sanitizer.sanitize(content));
            return builder.build();
        }
    }

    private Response checkWhetherWeHideInternals(IOException e, String content) {
        if (hideInternals) {
            UUID uuid = UUID.randomUUID();
            logger.error("[UUID={}] Exception: {}", uuid, e.getClass().getName());
            logger.error("[UUID={}] Error messages: {}", uuid, content);
            return Response.status(Response.Status.BAD_REQUEST).type(MediaType.TEXT_PLAIN).entity("An internal error has occurred, please notify your system administrator with the following code: " + uuid).build();
        }
        return null;
    }
}
