package com.xebialabs.xlrelease.api.internal;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.core.api.dto.RolePermissions;
import com.xebialabs.deployit.core.rest.api.SecurityResource;
import com.xebialabs.deployit.engine.api.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.xlrelease.domain.events.GlobalRolesOrPermissionsUpdatedEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.views.RolePermissionsView;
import com.xebialabs.xlrelease.views.RolesPermissionsView;

import io.micrometer.core.annotation.Timed;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static com.xebialabs.xlrelease.security.PermissionChecker.GLOBAL_SECURITY_ALIAS;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.getGlobalPermissions;

/**
 * The permissions assigned to a security role.
 */

@Path("/roles/permissions")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class RolePermissionsResource {

    private final RoleService roleService;
    private final SecurityResource securityResource;
    private final PermissionChecker permissionChecker;
    private final XLReleaseEventBus xlReleaseEventBus;

    @Autowired
    public RolePermissionsResource(RoleService roleService,
                                   SecurityResource securityResource,
                                   PermissionChecker permissionChecker,
                                   XLReleaseEventBus xlReleaseEventBus) {
        this.roleService = roleService;
        this.securityResource = securityResource;
        this.permissionChecker = permissionChecker;
        this.xlReleaseEventBus = xlReleaseEventBus;
    }

    @GET
    @Timed
    @Path("global")
    public RolesPermissionsView getGlobalRolePermissions() {
        permissionChecker.check(EDIT_SECURITY);

        List<RolePermissions> permissions = securityResource.readRolePermissions(GLOBAL_SECURITY_ALIAS(), null, null, null);
        Map<String, List<RolePermissions>> permissionLookup = permissions.stream().collect(Collectors.groupingBy(role -> role.getRole().getId()));

        List<com.xebialabs.deployit.security.Role> roles = roleService.getRoles();
        List<RolePermissions> rolePermissions = new ArrayList<>();

        for (com.xebialabs.deployit.security.Role securityRole : roles) {
            if (permissionLookup.containsKey(securityRole.getId())) {
                rolePermissions.addAll(permissionLookup.get(securityRole.getId()));
            } else {
                rolePermissions.add(new RolePermissions(new Role(securityRole.getId(), securityRole.getName()), new ArrayList<>()));
            }
        }

        return new RolesPermissionsView(rolePermissions, getGlobalPermissions());
    }

    @PUT
    @Timed
    @Path("global")
    public void setGlobalRolePermissions(List<RolePermissionsView> rolePermissionsViews) {
        this.permissionChecker.check(EDIT_SECURITY);

        List<RolePermissions> permissions = rolePermissionsViews.stream().map(RolePermissionsView::toRolePermissions).collect(Collectors.toList());
        securityResource.writeRolePermissions(GLOBAL_SECURITY_ALIAS(), permissions);
        xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent.apply());
    }
}
