package com.xebialabs.xlrelease.api.internal;

import java.util.List;
import java.util.stream.Collectors;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.domain.Team;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.TeamService;
import com.xebialabs.xlrelease.service.TeamUpdateOperation;
import com.xebialabs.xlrelease.service.TeamUpdateOperations;
import com.xebialabs.xlrelease.views.TeamView;
import com.xebialabs.xlrelease.views.converters.TeamMemberViewConverter;
import com.xebialabs.xlrelease.views.teams.TeamUpdateRequest;
import com.xebialabs.xlrelease.views.teams.TeamUpdateRequests;

import io.micrometer.core.annotation.Timed;
import scala.collection.immutable.Seq;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.repository.Ids.isReleaseId;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

/**
 * The teams defined on a release.
 */
@Path("/teams")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class TeamResource {
    private final Logger logger = LoggerFactory.getLogger(TeamResource.class);

    private final PermissionChecker permissions;
    private final ReleaseActorService releaseActorService;
    private final TeamService teamService;
    private final TeamMemberViewConverter teamMemberViewConverter;

    @Autowired
    public TeamResource(PermissionChecker permissions,
                        ReleaseActorService releaseActorService,
                        TeamService teamService,
                        TeamMemberViewConverter teamMemberViewConverter) {
        this.permissions = permissions;
        this.releaseActorService = releaseActorService;
        this.teamService = teamService;
        this.teamMemberViewConverter = teamMemberViewConverter;
    }

    @GET
    @Timed
    @Path("{teamContainerId:.*(Release|Folder)[^/-]*}")
    public List<TeamView> getTeams(@PathParam("teamContainerId") @IdParam String teamContainerId) {
        permissions.checkViewTeams(teamContainerId);
        return teamService.getEffectiveTeamViews(teamContainerId, teamMemberViewConverter);
    }

    @DELETE
    @Timed
    @Path("/{teamContainerId:.*(Release|Folder)[^/-]*}")
    public void deleteAllOwnTeams(@PathParam("teamContainerId") @IdParam String teamContainerId) {
        permissions.checkDeleteOwnTeams(teamContainerId);
        if (isReleaseId(teamContainerId)) {
            releaseActorService.updateTeams(teamContainerId, emptyList());
        } else {
            teamService.deleteTeamsFromPlatform(teamContainerId);
        }
    }

    @POST
    @Timed
    @Path("{teamContainerId:.*(Release|Folder)[^/-]*}")
    public List<TeamView> replaceTeams(@PathParam("teamContainerId") @IdParam String teamContainerId, List<TeamView> updatedListOfTeams) {
        // replaceTeams method can only be removed if release-permissions-controller.js is updated to use updateTeamOperations instead of replaceTeams
        permissions.checkViewTeams(teamContainerId);
        var teamNames = updatedListOfTeams.stream().map(TeamView::getTeamName).collect(Collectors.joining(",", "[", "]"));
        checkArgument(isReleaseAdminTeamPresent(updatedListOfTeams),
                "Cannot save teams %s in %s, missing 'Release Admin' team", teamNames, teamContainerId);

        List<Team> submittedTeams = updatedListOfTeams.stream().map(TeamView::toTeam).collect(toList());
        permissions.checkEditTeamsAgainstExisting(teamContainerId, submittedTeams);

        if (isReleaseId(teamContainerId)) {
            releaseActorService.updateTeams(teamContainerId, submittedTeams);
        } else {
            teamService.saveTeamsToPlatform(teamContainerId, submittedTeams);
        }
        // since this method updates ALL teams it is cheaper to call dedicated method to fetch all TeamViews
        // than to call a method that uses TeamMemberViewConverter for each individual member
        return teamService.getTeamViews(teamContainerId);
    }

    @PUT
    @Timed
    @Path("update/{teamContainerId:.*(Folder)[^/-]*}")
    public Response updateTeamOperations(@PathParam("teamContainerId") @IdParam String teamContainerId, TeamUpdateRequests updateRequests) {
        permissions.checkViewTeams(teamContainerId);
        var requests = updateRequests.requests();
        TeamUpdateRequest.validate(teamContainerId, requests);
        Seq<TeamUpdateOperation> operations = TeamUpdateOperations.reduce(requests);
        permissions.checkEditTeams(teamContainerId, operations);
        try {
            teamService.executeTeamUpdateOperations(operations);
            return Response.ok().build();
        } catch (Exception e) {
            String errorMsg = "Unable to execute team update operations";
            throw new DeployitException(e, errorMsg);
        }
    }

    private boolean isReleaseAdminTeamPresent(List<TeamView> teamViewList) {
        return teamViewList.stream().anyMatch(team -> Team.RELEASE_ADMIN_TEAMNAME.equals(team.getTeamName()));
    }
}
