package com.xebialabs.xlrelease.domain.tasks.createrelease;

import com.xebialabs.xlrelease.domain.Changes;
import com.xebialabs.xlrelease.domain.CreateReleaseTask;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective;
import com.xebialabs.xlrelease.domain.tasks.task.DefaultTaskUpdater;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.service.CiIdService;
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.Set;

import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TASK_CONFIGURATION;
import static com.xebialabs.xlrelease.risk.domain.RiskProfile.RISK_PROFILE;
import static com.xebialabs.xlrelease.variable.VariablePersistenceHelper.fixUpVariableIds;

@Component
public class CreateReleaseTaskUpdater extends DefaultTaskUpdater {

    private CiIdService ciIdService;

    @Autowired
    public CreateReleaseTaskUpdater(final CiIdService ciIdService) {
        this.ciIdService = ciIdService;
    }

    @Override
    public Class<? extends Task> getTaskClass() {
        return CreateReleaseTask.class;
    }

    @Override
    public Changes update(final Task original, final Task updated, final Set<TaskUpdateDirective> updateDirectives) {
        Changes changes = super.update(original, updated, updateDirectives);
        if (hasDirectiveToUpdateAllProperties(original, updateDirectives)) {
            updateCreateReleaseTaskConfiguration((CreateReleaseTask) original, (CreateReleaseTask) updated, changes);
        } else if (updateDirectives.contains(UPDATE_TASK_CONFIGURATION)) {
            changes.update(original);
            updateCreateReleaseTaskConfiguration((CreateReleaseTask) original, (CreateReleaseTask) updated, changes);
        }

        return changes;
    }

    private void updateCreateReleaseTaskConfiguration(final CreateReleaseTask original, final CreateReleaseTask updated, final Changes changes) {
        original.setNewReleaseTitle(updated.getNewReleaseTitle());
        original.setTemplateId(updated.getTemplateId());
        original.setFolderId(updated.getFolderId());
        original.setCreatedReleaseId(updated.getCreatedReleaseId());
        original.setStartRelease(updated.getStartRelease());
        original.setReleaseTags(updated.getReleaseTags());

        if (original.hasProperty(RISK_PROFILE)) {
            original.setProperty(RISK_PROFILE, updated.getProperty(RISK_PROFILE));
        }

        updateTemplateVariables(original, updated, changes);

        taskConcurrencyService.updateLastModifiedDetails(original);
    }

    private void updateTemplateVariables(final CreateReleaseTask original, final CreateReleaseTask updated, final Changes changes) {
        original.getTemplateVariables().stream()
                .filter(v -> !updated.getTemplateVariables().contains(v))
                .forEach(v -> changes.remove(v.getId()));

        Map<String, Variable> originalTemplateVariableMap = original.getTemplateVariablesByKeys();

        updated.getTemplateVariables().stream()
                .filter(Variable::isPassword)
                .forEach(variable -> {
                    Variable origVar = originalTemplateVariableMap.get(variable.getKey());
                    Object origValue = origVar != null ? origVar.getValue() : null;
                    variable.setUntypedValue(PasswordVerificationUtils.replacePasswordIfNeeded(origValue, variable.getValue()));
                });

        original.setTemplateVariables(updated.getTemplateVariables());
        fixUpVariableIds(original.getId(), original.getTemplateVariables(), ciIdService);
        changes.updateAll(original.getTemplateVariables());
    }

}
