package com.xebialabs.xlrelease.repository

import com.fasterxml.jackson.annotation.JsonSubTypes.Type
import com.fasterxml.jackson.annotation._
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.module.scala.DefaultScalaModule

private[repository] object ReleaseJsonParser {

  implicit class IdExtension(id: String) {
    def normalized: String = {
      Ids.normalizeId(Ids.getFolderlessId(id))
    }
  }

  lazy val objectMapper = {
    val mapper = new ObjectMapper()
    mapper.registerModule(DefaultScalaModule)
    mapper
  }

  lazy val releaseLevelObjectMapper = objectMapper.copy().addMixIn(classOf[ReleaseData], classOf[IgnorePhases])
  lazy val phaseLevelObjectMapper = objectMapper.copy().addMixIn(classOf[PhaseData], classOf[IgnoreTasks])

  def mapper(level: PlanItemLevel) = level match {
    case ReleaseLevel => releaseLevelObjectMapper
    case PhaseLevel => phaseLevelObjectMapper
    case TaskLevel => objectMapper
  }

  def parse(releaseJson: String, level: PlanItemLevel): ReleaseData = {
    mapper(level).readValue(releaseJson, classOf[ReleaseData])
  }


  trait IgnorePhases {
    @JsonIgnore
    def phases: List[PhaseData]
  }

  trait IgnoreTasks {
    @JsonIgnore
    def tasks: List[TaskData]
  }

  sealed trait PlanItemLevel

  case object ReleaseLevel extends PlanItemLevel

  case object PhaseLevel extends PlanItemLevel

  case object TaskLevel extends PlanItemLevel

  @JsonIgnoreProperties(ignoreUnknown = true)
  sealed trait PlanItemData {
    def id: String

    @JsonProperty("type")
    def itemType: String

    def title: String

    def status: String

    def startDate: String

    def endDate: String

    def scheduledStartDate: String

    def dueDate: String

    def plannedDuration: Integer
  }

  @JsonTypeInfo(use = JsonTypeInfo.Id.NAME, include = JsonTypeInfo.As.PROPERTY, property = "type", visible = true, defaultImpl = classOf[TaskData])
  @JsonSubTypes(Array(
    new Type(value = classOf[ParallelTaskGroupData], name = "xlrelease.ParallelGroup"),
    new Type(value = classOf[SequentialTaskGroupData], name = "xlrelease.SequentialGroup")
  ))
  sealed trait BaseTaskData extends PlanItemData {
  }

  trait TaskContainerData {
    def tasks: List[BaseTaskData]

    def getTask(taskId: String): BaseTaskData = {
      val normalizedTaskId = taskId.normalized
      val container = tasks.find(baseTaskData => normalizedTaskId.startsWith(baseTaskData.id.normalized)).orNull
      container match {
        case t: TaskData if normalizedTaskId == t.id => t
        case t: TaskData => throw new IllegalStateException(s"TaskData id [${t.id.normalized}] is not equal requested task id [$normalizedTaskId]")
        case pg: ParallelTaskGroupData => pg.getTask(normalizedTaskId)
        case sg: SequentialTaskGroupData => sg.getTask(normalizedTaskId)
      }
    }

    def allTasks: List[BaseTaskData] = tasks.flatMap {
      case t: TaskData => List(t)
      case pg: ParallelTaskGroupData => pg.allTasks
      case sg: SequentialTaskGroupData => sg.allTasks
    }
  }

  trait TaskGroupData extends TaskContainerData { self: BaseTaskData =>
    override def getTask(taskId: String): BaseTaskData = {
      if (self.id.normalized == taskId.normalized) {
        self
      } else {
        super.getTask(taskId)
      }
    }
  }

  case class ParallelTaskGroupData(
                                    id: String,
                                    itemType: String,
                                    title: String,
                                    status: String,
                                    startDate: String,
                                    endDate: String,
                                    scheduledStartDate: String,
                                    dueDate: String,
                                    tasks: List[BaseTaskData],
                                    plannedDuration: Integer
                                  ) extends BaseTaskData with TaskGroupData {

  }

  case class SequentialTaskGroupData(
                                      id: String,
                                      itemType: String,
                                      title: String,
                                      status: String,
                                      startDate: String,
                                      endDate: String,
                                      scheduledStartDate: String,
                                      dueDate: String,
                                      tasks: List[BaseTaskData],
                                      plannedDuration: Integer
                                    ) extends BaseTaskData with TaskGroupData

  case class TaskData(
                       id: String,
                       itemType: String,
                       title: String,
                       status: String,
                       startDate: String,
                       endDate: String,
                       scheduledStartDate: String,
                       dueDate: String,
                       plannedDuration: Integer
                     ) extends BaseTaskData

  case class PhaseData(
                        id: String,
                        itemType: String,
                        title: String,
                        status: String,
                        color: String,
                        startDate: String,
                        endDate: String,
                        scheduledStartDate: String,
                        dueDate: String,
                        tasks: List[BaseTaskData],
                        plannedDuration: Integer
                      ) extends PlanItemData with TaskContainerData {
  }

  case class ReleaseData(
                          id: String,
                          itemType: String,
                          title: String,
                          status: String,
                          startDate: String,
                          endDate: String,
                          scheduledStartDate: String,
                          dueDate: String,
                          realFlagStatus: String,
                          flagStatus: String,
                          flagComment: String,
                          phases: List[PhaseData],
                          variables: List[ReleaseVariable],
                          plannedDuration: Integer,
                          tags: List[String]
                        ) extends PlanItemData {

    def getPhase(phaseId: String): PhaseData = this.phases.find(phaseData => phaseData.id.normalized == phaseId.normalized).get

    def getTask(taskId: String): BaseTaskData = {
      val phaseData = getPhase(Ids.phaseIdFrom(taskId))
      phaseData.getTask(taskId)
    }

    def allTasks(): List[BaseTaskData] = phases.flatMap(_.allTasks)

    def allStringVariables(): List[StringVariable] = {
      variables.collect { case v: StringVariable => v }
    }
  }

  @JsonIgnoreProperties(ignoreUnknown = true)
  @JsonTypeInfo(use = JsonTypeInfo.Id.NAME, include = JsonTypeInfo.As.PROPERTY, property = "type", visible = true, defaultImpl = classOf[NonStringVariable])
  @JsonSubTypes(Array(
    new Type(value = classOf[StringVariable], name = "xlrelease.StringVariable")
  ))
  trait ReleaseVariable {
    def id: String

    @JsonProperty("type")
    def itemType: String

    def key: String
  }

  case class StringVariable(
                             id: String,
                             itemType: String,
                             key: String,
                             value: String
                           ) extends ReleaseVariable {

  }

  case class NonStringVariable(
                                id: String,
                                itemType: String,
                                key: String
                              ) extends ReleaseVariable
}



