package com.xebialabs.xlrelease.scheduler.logs

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.repository.Ids.getFolderlessId
import com.xebialabs.xlrelease.runner.domain.JobId
import com.xebialabs.xlrelease.scheduler.storage.spring.StorageConfiguration.URI_SCHEME_LOCAL_STORAGE
import com.xebialabs.xlrelease.storage.domain.JobEntryRef
import com.xebialabs.xlrelease.storage.service.StorageService
import grizzled.slf4j.Logging
import org.apache.commons.codec.digest.DigestUtils
import org.apache.commons.io.IOUtils
import org.springframework.util.StringUtils.hasText

import java.io.OutputStream
import java.net.URI
import java.nio.file.Paths
import scala.util.{Failure, Try, Using}

private[logs] case class TaskExecutionLog(taskId: String, executionId: String) extends Logging {
  require(hasText(taskId), "taskId is null or empty")
  require(hasText(executionId), "executionId is null or empty")

  private def fetchChunks(storageService: StorageService, jobUri: URI, lastJob: JobId, lastChunk: Long): List[JobEntryRef] = {
    val jobEntryRef = entryRef(jobUri)
    val chunks = storageService.listFiles(jobEntryRef)
    chunks.takeWhile { chunkUri =>
      val path = Paths.get(chunkUri)
      val jobId = path.subpath(path.getNameCount - 2, path.getNameCount - 1).toString.toLong
      val chunkId = path.subpath(path.getNameCount - 1, path.getNameCount).toString.toLong
      (jobId < lastJob) || ((jobId == lastJob) && chunkId <= lastChunk)
    }.map(chunkUri => {
      val chunkRef = entryRef(chunkUri)
      chunkRef
    })
  }

  private def fetch(storageService: StorageService, lastJob: JobId, chunk: Long): List[JobEntryRef] = {
    val jobDirs = storageService.listDirectories(executionEntryRef)
    if (null == jobDirs || jobDirs.isEmpty) {
      throw new NotFoundException("Could not find log entries for task [%s] with executionId [%s]", taskId, executionId)
    }
    jobDirs.takeWhile { jobDirUri =>
      val path = Paths.get(jobDirUri)
      val lastEntry = path.subpath(path.getNameCount - 1, path.getNameCount).toString.toLong
      lastEntry <= lastJob
    }.flatMap(jobUri => fetchChunks(storageService, jobUri, lastJob, chunk))
  }

  def fetch(storageService: StorageService, outputStream: OutputStream, jobId: JobId, chunk: Long): Unit = {
    for {
      jobEntryRef <- fetch(storageService, jobId, chunk)
      is <- Try(storageService.get(jobEntryRef)).recoverWith {
        case e =>
          logger.warn(s"Unable to fetch job entry ref: $jobEntryRef", e)
          Failure(e)
      }
    } Try {
      Using.resource(is) { content => IOUtils.copy(content, outputStream) }
    }
  }

  import TaskExecutionLog.TaskIdHash

  private def executionEntryRef: JobEntryRef = JobEntryRef(
    new URI(URI_SCHEME_LOCAL_STORAGE, null, s"/jobs/${taskId.hash()}/$executionId", null)
  )

  private def entryRef(chunkUri: URI): JobEntryRef = {
    val uri = new URI(URI_SCHEME_LOCAL_STORAGE, null, chunkUri.getPath, null)
    JobEntryRef(uri)
  }
}

object TaskExecutionLog {
  implicit class TaskIdHash(taskId: String) {
    def hash(): String = DigestUtils.sha256Hex(getFolderlessId(taskId))
  }
}
