package com.xebialabs.xlrelease.script

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.actors.initializer.SubscriptionInitializer
import com.xebialabs.xlrelease.domain.distributed.events.{DistributedConfigurationCreatedEvent, DistributedConfigurationUpdatedEvent}
import com.xebialabs.xlrelease.events.{EventBus, Subscribe}
import com.xebialabs.xlrelease.features.settings.ScriptEngineFeatureSettings
import com.xebialabs.xlrelease.script.ScriptEngineReloadScheduler.settingsType
import com.xebialabs.xlrelease.service.ConfigurationService
import grizzled.slf4j.Logging
import jakarta.annotation.PreDestroy
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.scheduling.TaskScheduler
import org.springframework.stereotype.Component

import java.time.temporal.ChronoUnit
import java.time.{Duration, Instant}
import java.util.concurrent.ScheduledFuture
import scala.jdk.CollectionConverters._

object ScriptEngineReloadScheduler {
  lazy val settingsType: String = Type.valueOf(classOf[ScriptEngineFeatureSettings]).toString
}

@Component
class ScriptEngineReloadScheduler @Autowired()(taskScheduler: TaskScheduler,
                                               configurationService: ConfigurationService,
                                               eventBus: EventBus,
                                               scriptExecutors: java.util.List[Jsr223ScriptExecutor]) extends Logging with SubscriptionInitializer {

  private var runningSchedule: Option[ScheduledFuture[_]] = None
  private val scheduleLock = new Object

  override def initialize(): Unit = {
    eventBus.register(this)
    scheduleReloadIfNecessary()
  }

  @PreDestroy
  def unregister(): Unit = {
    eventBus.deregister(this)
    runningSchedule.foreach(_.cancel(false))
    logger.info("Script engine scheduler stopped and de-registered from event bus")
  }

  @Subscribe
  def onFeatureCreate(event: DistributedConfigurationCreatedEvent): Unit = if (event.configurationType == settingsType) {
    scheduleReloadIfNecessary()
  }

  @Subscribe
  def onFeatureUpdate(event: DistributedConfigurationUpdatedEvent): Unit = if (event.configurationType == settingsType) {
    scheduleReloadIfNecessary()
  }

  private def scheduleReloadIfNecessary(): Unit = try {
    val scriptEngineSettings = configurationService.getFeatureSettings(settingsType).asInstanceOf[ScriptEngineFeatureSettings]

    scheduleLock.synchronized {
      runningSchedule.foreach(_.cancel(false))
      if (scriptEngineSettings.enableEngineReload) {
        logger.info(s"Scheduling script engine reload with ${scriptEngineSettings.reloadInterval} min interval")
        val reloadInterval = scriptEngineSettings.reloadInterval.longValue()
        runningSchedule = Some(taskScheduler.scheduleAtFixedRate(new Runnable {
          override def run(): Unit = {
            logger.info("Reloading script engines")
            scriptExecutors.asScala.foreach(_.reloadScriptEngines())
          }
        }, Instant.now().plus(reloadInterval, ChronoUnit.MINUTES), Duration.ofMinutes(reloadInterval)))
      } else if (runningSchedule.isDefined) {
        logger.info(s"Disabled script engine reload")
      }
    }
  } catch {
    case e: Exception => logger.error("Error scheduling engine reload", e)
  }
}
