package com.xebialabs.xlrelease.security.filter;

import java.io.IOException;
import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.security.authentication.AuthenticationDetailsSource;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.authentication.WebAuthenticationDetailsSource;
import org.springframework.security.web.authentication.www.BasicAuthenticationConverter;
import org.springframework.util.Assert;
import org.springframework.web.filter.OncePerRequestFilter;

import com.xebialabs.deployit.security.authentication.PersonalAuthenticationToken;

import static com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank;

/**
 * filter to get personal access token from request header or basic auth header
 */
public class PATAuthenticationFilter extends OncePerRequestFilter {

    private final AuthenticationEntryPoint authenticationEntryPoint;

    private final AuthenticationManager authenticationManager;

    private final String header;

    private final AuthenticationDetailsSource<HttpServletRequest, ?> authenticationDetailsSource = new WebAuthenticationDetailsSource();

    private final BasicAuthenticationConverter authenticationConverter = new BasicAuthenticationConverter();

    public PATAuthenticationFilter(AuthenticationManager authenticationManager,
                                   AuthenticationEntryPoint authenticationEntryPoint,
                                   String header) {
        Assert.notNull(authenticationManager, "authenticationManager cannot be null");
        Assert.notNull(authenticationEntryPoint, "authenticationEntryPoint cannot be null");
        Assert.hasText(header, "header cannot be empty");
        this.authenticationManager = authenticationManager;
        this.authenticationEntryPoint = authenticationEntryPoint;
        this.header = header;
    }

    @Override
    protected void doFilterInternal(final HttpServletRequest request, final HttpServletResponse response, final FilterChain filterChain) throws ServletException, IOException {
        String token = this.getTokenFromHeader(request);
        if (token == null) {
            //try to resolve from basic auth header
            token = this.getTokenFromAuthorization(request);
        }
        if (token == null) {
            logger.trace("Did not process request since did not find personal access token");
            filterChain.doFilter(request, response);
            return;
        }

        try {
            PersonalAuthenticationToken authenticationToken = new PersonalAuthenticationToken(token);
            authenticationToken.setDetails(this.authenticationDetailsSource.buildDetails(request));

            Authentication authenticationResult = this.authenticationManager.authenticate(authenticationToken);
            SecurityContext context = SecurityContextHolder.createEmptyContext();
            context.setAuthentication(authenticationResult);
            SecurityContextHolder.setContext(context);
            logger.debug(String.format("Set SecurityContextHolder to %s", authenticationResult));
            setTokenExpiryHeader(response, authenticationResult);
            filterChain.doFilter(request, response);
        } catch (AuthenticationException ex) {
            SecurityContextHolder.clearContext();
            logger.trace("Failed to process authentication request", ex);
            this.authenticationEntryPoint.commence(request, response, ex);
        }
    }

    private String getTokenFromHeader(HttpServletRequest request) {
        return request.getHeader(header);
    }

    private String getTokenFromAuthorization(HttpServletRequest request) {
        UsernamePasswordAuthenticationToken token = this.authenticationConverter.convert(request);
        if (null == token) {
            logger.trace("Did not find username and password in Basic Authorization header");
            return null;
        } else if (isNotBlank(token.getPrincipal().toString())) {
            logger.trace(String.format("Found '%s' in Basic Authorization header", token.getPrincipal().toString()));
            return null;
        } else {
            logger.trace("Found personal access token in Basic Authorization header");
            return token.getCredentials().toString();
        }
    }

    private void setTokenExpiryHeader(HttpServletResponse response, Authentication authentication) {
        if (authentication instanceof PersonalAuthenticationToken pat && null != pat.getExpiryDate()) {
            response.setDateHeader("Release-Personal-Token-Expiration", pat.getExpiryDate().getTime());
        }
    }
}
