package com.xebialabs.xlrelease.service;

import java.util.*;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.xlrelease.domain.Changes;
import com.xebialabs.xlrelease.domain.Changes.VariablesChanges;
import com.xebialabs.xlrelease.domain.CustomScriptTask;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.events.ReleaseVariablesUpdateOperation;
import com.xebialabs.xlrelease.events.TaskStatusLineOperation;
import com.xebialabs.xlrelease.events.XLReleaseOperation;
import com.xebialabs.xlrelease.repository.CiCloneHelper;
import com.xebialabs.xlrelease.script.DefaultScriptService.BaseScriptTaskResults;
import com.xebialabs.xlrelease.script.DefaultScriptService.CustomScriptTaskResults;
import com.xebialabs.xlrelease.script.DefaultScriptService.ScriptTaskResults;
import com.xebialabs.xlrelease.script.ScriptVariables;
import com.xebialabs.xlrelease.utils.PythonScriptCiHelper;
import com.xebialabs.xlrelease.variable.VariableHelper;


import static com.google.common.base.Strings.isNullOrEmpty;
import static com.xebialabs.xlrelease.domain.CustomScriptTask.PYTHON_SCRIPT_PREFIX;
import static com.xebialabs.xlrelease.variable.VariableHelper.containsOnlyVariable;
import static com.xebialabs.xlrelease.variable.VariablePersistenceHelper.fixUpVariableIds;
import static java.lang.String.format;
import static org.springframework.util.StringUtils.hasText;

@Service
public class ScriptResultsService {

    public static final String OUTPUT_TOO_LONG_MESSAGE = "This value was too long and has been cut off at the end. ";
    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    private final ScriptVariables scriptVariables;
    private final CiIdService ciIdService;


    @Autowired
    public ScriptResultsService(ScriptVariables scriptVariables, CiIdService ciIdService) {
        this.scriptVariables = scriptVariables;
        this.ciIdService = ciIdService;
    }

    public Changes resolveScriptTaskResults(Task task, BaseScriptTaskResults baseScriptTaskResults) {
        // TODO remove this smelly instanceof
        if (task instanceof CustomScriptTask && baseScriptTaskResults instanceof CustomScriptTaskResults) {
            return customScriptTask((CustomScriptTask) task, (CustomScriptTaskResults) baseScriptTaskResults);
        } else if (baseScriptTaskResults instanceof ScriptTaskResults) {
            return task(task, (ScriptTaskResults) baseScriptTaskResults);
        } else {
            throw new IllegalArgumentException("oh snap! :(");
        }
    }

    private Changes task(Task task, ScriptTaskResults scriptTaskResults) {
        VariablesChanges releaseVariablesChanges = scriptTaskResults.getReleaseVariablesChanges();
        VariablesChanges folderVariablesChanges = scriptTaskResults.getFolderVariablesChanges();
        VariablesChanges globalVariablesChanges = scriptTaskResults.getGlobalVariablesChanges();
        // we need to save the variables here, as we are gonna freeze them after this
        scriptVariables.processReleaseVariablesChanges(task.getRelease(), releaseVariablesChanges);
        scriptVariables.processFolderVariablesChanges(folderVariablesChanges);
        scriptVariables.processGlobalVariablesChanges(globalVariablesChanges);

        List<XLReleaseOperation> allOperationsExceptReleaseVariables = new ArrayList<>();
        allOperationsExceptReleaseVariables.addAll(globalVariablesChanges.getOperations());
        allOperationsExceptReleaseVariables.addAll(folderVariablesChanges.getOperations());

        Changes changes = new Changes();
        changes.addOperations(allOperationsExceptReleaseVariables);
        return changes;
    }

    private Changes customScriptTask(CustomScriptTask task, CustomScriptTaskResults results) {

        task.setNextScriptPath(results.getNextScriptPath());
        task.setStatusLine(results.getStatusLine());
        task.setInterval(results.getInterval());

        if (!task.isStillExecutingScript(results.getScriptExecutionId())) {
            logger.debug("Will not save script results of task: '{}' : it has been aborted.", task.getId());
            return new Changes();
        }

        Changes changes = processCustomScriptTaskResults(task, results.getOutputVariables());
        changes.update(task);
        if (hasText(task.getStatusLine())) {
            changes.addOperation(new TaskStatusLineOperation(task, task.getStatusLine()));
        }
        if (task.getNextScriptPath() != null) {
            changes.addPostAction(new ExecuteNextCustomScriptPath(task));
        }
        return changes;
    }

    /**
     * This is a non-persisting method which only changes/creates objects in memory
     */
    private Changes processCustomScriptTaskResults(CustomScriptTask task, Map<String, Object> outputAttributes) {
        setTransitionalProperties(task, outputAttributes);
        return setOutputPropertiesAndVariables(task, outputAttributes);
    }

    private void setTransitionalProperties(CustomScriptTask task, Map<String, Object> outputAttributes) {
        task.getPythonScript().getTransitionalProperties().forEach(propertyDescriptor -> {
            String propertyName = propertyDescriptor.getName();
            Object value = outputAttributes.get(propertyName);
            if (value == null) {
                logger.debug("Python script task {} did not return a value for property {}", task.getId(), propertyDescriptor.getName());
            } else {
                task.getPythonScript().setProperty(propertyName, value);
            }
        });
    }

    private Changes setOutputPropertiesAndVariables(CustomScriptTask task, Map<String, Object> outputAttributes) {
        final Changes changes = new Changes();
        final Release release = task.getRelease();
        final Collection<PropertyDescriptor> outputProperties = task.getPythonScript().getOutputProperties();
        final Map<String, Object> newVariableValues = new HashMap<>();
        final Map<String, Object> newPasswordVariableValues = new HashMap<>();
        final String taskId = task.getId();

        for (PropertyDescriptor propertyDescriptor : outputProperties) {
            String propertyName = propertyDescriptor.getName();
            String fqPropertyName = PYTHON_SCRIPT_PREFIX + propertyName;
            String variableName = null;
            Object propertyValue = task.getPythonScript().getProperty(propertyName);

            // In pre-5.0 a string output property value contains a variable name to put the resulting value to
            if (propertyValue instanceof String && containsOnlyVariable((String) propertyValue)) {
                variableName = (String) propertyValue;
            }

            // In 5.0+ mapping from output properties to target variables is stored in a separate property: variableMapping
            if (task.getVariableMapping().containsKey(fqPropertyName)) {
                variableName = task.getVariableMapping().get(fqPropertyName);
                if (!task.hasNextScriptToExecute()) {
                    task.getVariableMapping().remove(fqPropertyName);
                }
            }

            Object value = outputAttributes.get(propertyName);

            if (task.isKeepPreviousOutputPropertiesOnRetry() && !PythonScriptCiHelper.isEmpty(task.getPythonScript().getProperty(propertyName))) {
                value = task.getPythonScript().getProperty(propertyName);
                if (propertyDescriptor.getKind() == PropertyKind.STRING) {
                    // look into CustomScriptTaskUpdater:53
                    value = VariableHelper.withoutVariableSyntax((String) value);
                }
            }

            if (value == null) {
                logger.debug("Python script task {} did not return a value for property {}", taskId, propertyDescriptor.getName());
                continue;
            }

            if (propertyDescriptor.getKind() == PropertyKind.STRING) {
                value = truncateValue(task, propertyDescriptor.getLabel(), value.toString(), changes);
            }

            task.getPythonScript().setProperty(propertyName, value);

            if (isNullOrEmpty(variableName)) {
                logger.debug("Missing output variable on Python script task {} for property {}", taskId, propertyDescriptor.getName());
                continue;
            }

            if (propertyDescriptor.isPassword()) {
                newPasswordVariableValues.put(variableName, value);
            } else {
                newVariableValues.put(variableName, value);
            }
        }
        // TODO check if we can get rid of this somehow - customers have releases with quite a lot of tasks and a lot of variables we don't want to copy
        List<Variable> oldVariables = CiCloneHelper.cloneCis(release.getVariables());
        release.setVariableValues(newVariableValues);
        release.setPasswordVariableValues(newPasswordVariableValues);
        fixUpVariableIds(release.getId(), release.getVariables(), ciIdService);

        //let's add the operation, maybe nothing is changed but that will be calculated by VariablesDiff
        changes.addOperation(new ReleaseVariablesUpdateOperation(oldVariables, release.getVariables(), true));

        return changes;
    }

    private String truncateValue(CustomScriptTask task, String propertyLabel, String value, Changes changes) {
        int maxOutputPropertySize = task.getPythonScript().getMaxOutputPropertySize();
        if (value.length() > maxOutputPropertySize) {
            value = OUTPUT_TOO_LONG_MESSAGE.concat(value.substring(0, maxOutputPropertySize));
            logger.warn("The content of the output property '{}' from task '{}' has been truncated to {} characters",
                    propertyLabel, task.getName(), maxOutputPropertySize);
            changes.addComment(task, format("*Warning*: the value for property '%s' has been truncated because it was too large", propertyLabel));
        }
        return value;
    }

}
