package com.xebialabs.xlrelease.service;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Optional;
import jakarta.activation.FileTypeMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.io.StreamWrappingOverthereFile;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.configuration.CustomLogoContentType;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.events.TemplateLogoUpdatedEvent;
import com.xebialabs.xlrelease.domain.utils.IdUtils;
import com.xebialabs.xlrelease.events.EventBus;
import com.xebialabs.xlrelease.repository.*;
import com.xebialabs.xlrelease.utils.FileContentValidation;

import io.micrometer.core.annotation.Timed;

import static java.lang.String.format;
import static scala.jdk.javaapi.OptionConverters.toJava;

@Service
public class UploadService {

    private static final Logger logger = LoggerFactory.getLogger(UploadService.class);
    public static final String NEWLINES = "[\n\r]";
    private final AttachmentService attachmentService;
    private final ReleaseRepository releaseRepository;
    private final AttachmentRepository attachmentRepository;
    private final ReleaseActorService releaseActorService;
    private final TemplateMetadataRepository templateMetadataRepository;
    private final EventBus eventBus;

    @Autowired
    public UploadService(AttachmentService attachmentService,
                         ReleaseRepository releaseRepository,
                         AttachmentRepository attachmentRepository,
                         ReleaseActorService releaseActorService,
                         TemplateMetadataRepository templateMetadataRepository,
                         EventBus eventBus) {
        this.attachmentService = attachmentService;
        this.releaseRepository = releaseRepository;
        this.attachmentRepository = attachmentRepository;
        this.releaseActorService = releaseActorService;
        this.templateMetadataRepository = templateMetadataRepository;
        this.eventBus = eventBus;
    }

    @Timed
    public Attachment addAttachment(String ciId, String fileName, byte[] fileByteArray) {
        return addAttachment(ciId, fileName, FileTypeMap.getDefaultFileTypeMap().getContentType(fileName), new ByteArrayInputStream(fileByteArray));
    }

    @Timed
    public Attachment addAttachment(String ciId, String fileName, String contentType, InputStream inputStream) {
        logger.debug("Attaching {} to {}", fileName.replaceAll(NEWLINES, "_"), ciId.replaceAll(NEWLINES, "_"));
        Release release = releaseRepository.findById(Ids.releaseIdFrom(ciId));
        Task task = null;
        if (!Ids.isReleaseId(ciId)) {
            task = release.getTask(ciId);
        }
        try (InputStream validatedInputStream = FileContentValidation.apply(fileName, inputStream).get()) {
            Attachment attachment = attachmentService.attachToRelease(release, fileName, contentType, validatedInputStream);
            if (task != null) {
                task.getAttachments().add(attachment);
            }
            attachmentRepository.insertAttachment(release.getId(), attachment);
            if (null != task) {
                return releaseActorService.createAttachmentOnTask(task.getId(), attachment);
            } else {
                return releaseActorService.createAttachmentOnRelease(release.getId(), attachment);
            }
        } catch (IOException e) {
            throw new DeployitException(e, "Unable to attach '%s'", fileName);
        }
    }

    @Timed
    public TemplateLogo addLogo(String ciId, String filename, String contentType, InputStream content) throws IOException {
        if (!CustomLogoContentType.getAllTypes().contains(contentType)) {
            throw new IllegalArgumentException(format("Content type %s not allowed. Allowed content types are: %s", contentType, CustomLogoContentType.getBrowserTypes().toString()));
        }
        try (InputStream validatedInputStream = FileContentValidation.apply(filename, content).get()) {
            logger.debug("Attaching logo to {}", ciId);
            Optional<Integer> releaseUid = toJava(releaseRepository.getUid(ciId));

            if (releaseUid.isPresent()) {
                TemplateLogo logo = createLogo(ciId, filename, contentType, validatedInputStream);
                TemplateLogo createdLogo = templateMetadataRepository.createOrUpdateLogo(releaseUid.get(), logo);
                eventBus.publish(new TemplateLogoUpdatedEvent(ciId, createdLogo));
                return createdLogo;
            } else {
                throw new NotFoundException("Repository entity [%s] not found", ciId);
            }
        }
    }

    private TemplateLogo createLogo(String releaseId, String filename, String contentType, InputStream bytes) {
        AttachmentSizeLimiter limiter = new AttachmentSizeLimiter(TemplateLogo.getDefaultMaxLogoSize(), bytes);
        StreamWrappingOverthereFile file = new StreamWrappingOverthereFile(filename, limiter);
        String logoId = IdUtils.getUniqueId(Type.valueOf(TemplateLogo.class), releaseId);

        TemplateLogo templateLogo = new TemplateLogo(file, contentType);
        templateLogo.setId(logoId);
        templateLogo.getExportFilename();

        String releaseTitle = releaseRepository.getTitle(releaseId);
        templateLogo.setProperty(BaseArtifact.PARENT_TITLE_PROPERTY_NAME, releaseTitle);
        templateLogo.getPortableFilename();

        return templateLogo;
    }
}
