package com.xebialabs.xlrelease.support.report

import com.xebialabs.deployit.booter.local.TypePluginMappingManager
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.plugin.manager.rest.api.PluginSource
import com.xebialabs.plugin.manager.service.PluginService
import com.xebialabs.xlplatform.support.report.ReportDataProvider
import com.xebialabs.xlrelease.config.ArchivingSettingsManager
import com.xebialabs.xlrelease.customscripts.ScriptTypes
import com.xebialabs.xlrelease.domain.ReleaseKind
import com.xebialabs.xlrelease.support.report.repository.TaskTypeCategory.{Container, Regular, Unknown}
import com.xebialabs.xlrelease.support.report.repository.{DataStatisticsRepository, PluginType, TaskTypeCategory, TaskTypeUsage}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.io.File
import java.util
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._

@Component
class AnalyticsDataStatisticsProvider @Autowired()(val repository: DataStatisticsRepository,
                                                   val archivingSettingsManager: ArchivingSettingsManager,
                                                   val scriptTypes: ScriptTypes,
                                                   val pluginService: PluginService) extends ReportDataProvider {

  val name = "analyticsdata"

  //noinspection ScalaStyle
  override def collectStatistics(): Map[String, Any] = {
    val statistics = new AnalyticsUsageStatistics()
    statistics.workflowTop25TaskTypes = repository.countTopTaskTypeOccurrences(25, Option(ReleaseKind.WORKFLOW)).asJava
    statistics.averageTasksPerWorkflow = repository.averageTasksPerRelease(Option(ReleaseKind.WORKFLOW))
    statistics.maxTasksInWorkflow = repository.findMaxTasksInRelease(Option(ReleaseKind.WORKFLOW))

    collectTaskTypeStatistics(statistics)

    Map[String, Any]("analyticsdata" -> statistics)
  }

  private def collectTaskTypeStatistics(statistics: AnalyticsUsageStatistics): AnalyticsUsageStatistics = {
    logger.debug("Collecting Task Type Usage statistics - START")
    val taskUsages = repository.findTaskTypeUsageInLastDays(180)
    val categorizedTaskUsages = categorizeTaskUsage(taskUsages)

    statistics.coreTasksLast180Days = categorizedTaskUsages.filter(_.pluginType == PluginType.BuiltIn).map(_.print()).asJava
    statistics.extensionTasksLast180Days = categorizedTaskUsages.filter(_.pluginType == PluginType.Extension).map(_.print()).asJava
    statistics.pluginTasksLast180Days = categorizedTaskUsages
      .filter(usage => usage.pluginType == PluginType.Custom || usage.pluginType == PluginType.Official)
      .map(_.print()).asJava
    logger.debug("Collecting Task Type Usage statistics - END")
    statistics
  }

  private def categorizeTaskUsage(taskTypeUsages: List[TaskTypeUsage]): List[TaskTypeUsage] = {
    logger.debug("Categorize Task Usages based on task type")
    taskTypeUsages.map { usage =>
      val taskCategory = getTaskCategory(usage.taskType)
      val pluginUrl = TypePluginMappingManager.getPluginUrl(usage.taskType)

      val updatedUsage = Option(pluginUrl).map { url =>
        val installedPlugins = pluginService.pluginManager.listInstalled()
        installedPlugins.find(id => url.contains(id.pluginId.toString())) match {
          case Some(id) if id.source == PluginSource.LOCAL =>
            usage.copy(pluginType = PluginType.Custom, pluginName = id.name, pluginVersion = id.version.map(_.id).getOrElse(""))
          case Some(id) if id.source == PluginSource.XLR_OFFICIAL =>
            usage.copy(pluginType = PluginType.Official, pluginName = id.name, pluginVersion = id.version.map(_.id).getOrElse(""))
          //        NormalizePath in XlrPluginClassLoader.scala Hence the forward slash check as well
          case None if url.contains(s"ext${File.separator}") || url.contains(s"ext/") =>
            usage.copy(pluginType = PluginType.Extension)
          case _ =>
            usage.copy(pluginType = PluginType.BuiltIn)
        }
      }.getOrElse {
        usage.copy(pluginType = if (taskCategory == Unknown) PluginType.Custom else PluginType.BuiltIn)
      }.copy(taskCategory = taskCategory, pluginUrl = pluginUrl)

      updatedUsage
    }
  }

  private def getTaskCategory(taskType: String): TaskTypeCategory = Type.valueOf(taskType) match {
    case t: Type if t.exists() && isContainerTask(t) => Container
    case t: Type if t.exists() => Regular
    case _ => Unknown
  }

  private def isContainerTask(targetType: Type): Boolean = scriptTypes.getContainerTaskTypes.contains(targetType)

}

class AnalyticsUsageStatistics {
  @BeanProperty
  var workflowTop25TaskTypes: util.List[String] = new util.ArrayList[String]()

  @BeanProperty
  var averageTasksPerWorkflow: Float = 0

  @BeanProperty
  var maxTasksInWorkflow: Int = 0

  @BeanProperty
  var pluginTasksLast180Days: util.List[String] = new util.ArrayList[String]()

  @BeanProperty
  var coreTasksLast180Days: util.List[String] = new util.ArrayList[String]()

  @BeanProperty
  var extensionTasksLast180Days: util.List[String] = new util.ArrayList[String]()
}