package com.xebialabs.xlrelease.security

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.{PermissionDeniedException, Permissions}
import com.xebialabs.xlrelease.domain.{Release, Task, Team}
import com.xebialabs.xlrelease.repository.Ids.{isReleaseId, releaseIdFrom}
import com.xebialabs.xlrelease.repository.query.TaskBasicData
import com.xebialabs.xlrelease.repository.{Ids, ReleaseInformation}
import com.xebialabs.xlrelease.security.PermissionChecker.{exceptionForPermissions, isPermissionApplicableTo, isWorkflowRelatedPermission}
import com.xebialabs.xlrelease.security.XLReleasePermissions._
import grizzled.slf4j.Logging
import org.springframework.security.core.Authentication

import scala.annotation.varargs
import scala.collection.mutable

//scalastyle:off number.of.methods
private[security] case class ReleasePermissionContext(env: PermissionContextEnv) extends PermissionContext with Logging {

  private val releaseInformations: mutable.Map[String, Option[ReleaseInformation]] = mutable.Map()

  private val globalPermissions: mutable.Map[Permission, Boolean] = mutable.Map()

  type TaskId = String

  private val taskDataLookup: mutable.Map[TaskId, TaskBasicData] = mutable.Map()

  private def releaseInformation(releaseId: String): Option[ReleaseInformation] = {
    val result = releaseInformations.getOrElseUpdate(releaseId, {
      logger.debug(s"Release information requested for releaseId: $releaseId.")
      if (releaseInformations.size > 0) {
        logger.error(s"Unexpected release information requested for releaseId: $releaseId. Current entries: ${releaseInformations.keys}")
      }
      env.getReleaseInformation(releaseId)
    })
    result
  }

  override def hasGlobalPermission(permission: Permission): Boolean = {
    globalPermissions.getOrElseUpdate(permission, {
      logger.debug(s"Requested check for global permission $permission")
      env.hasGlobalPermission(permission)
    })
  }

  private def getTaskData(taskId: TaskId): TaskBasicData = {
    taskDataLookup.getOrElseUpdate(taskId, env.getTaskData(taskId))
  }

  lazy val isCurrentUserAdmin: Boolean = env.isCurrentUserAdmin()

  override def checkView(releaseId: String): Unit = {
    if (!hasGlobalPermission(XLReleasePermissions.AUDIT_ALL)) {
      if (!isWorkflowExecutionOwner(releaseId)) {
        check(getViewPermission(releaseId), releaseId)
      }
    }
  }

  override def checkEdit(releaseId: String): Unit = {
    val releaseInfo = releaseInformation(releaseId).getOrElse(throw PermissionDeniedException.withMessage(s"You cannot edit $releaseId"))
    val permissionToCheck: Permission = if (releaseInfo.isTemplate()) EDIT_TEMPLATE else EDIT_RELEASE
    check(permissionToCheck, releaseId)
  }

  override def checkEditTask(releaseId: String): Unit = {
    val releaseInfo = releaseInformation(releaseId).getOrElse(throw PermissionDeniedException.withMessage(s"You cannot edit task on $releaseId"))
    val permissionToCheck: Permission = if (releaseInfo.isTemplate()) EDIT_TEMPLATE else EDIT_RELEASE_TASK
    check(permissionToCheck, releaseId)
  }

  override def checkLockTaskPermission(releaseId: String): Unit = {
    val releaseInfo = releaseInformation(releaseId).getOrElse(throw PermissionDeniedException.withMessage(s"You cannot lock task on $releaseId"))
    val permissionToCheck: Permission = if (releaseInfo.isTemplate()) LOCK_TEMPLATE_TASK else LOCK_RELEASE_TASK
    check(permissionToCheck, releaseId)
  }

  override def checkLockTaskPermissionForAssignment(): Unit = {
    if(!isCurrentUserAdmin) {
      throw PermissionDeniedException.withMessage(s"Only Release Admins can change the assignment of a locked task")
    }
  }

  override def checkEditPreconditionPermission(releaseId: String): Unit = {
    val releaseInfo = releaseInformation(releaseId).getOrElse(throw PermissionDeniedException.withMessage(s"You cannot edit precondition on $releaseId"))
    if (releaseInfo.isTemplate()) {
      check(EDIT_TEMPLATE_PRECONDITION, releaseId)
    } else {
      checkAny(releaseId, EDIT_RELEASE_PRECONDITION, EDIT_RELEASE_TASK)
    }
  }

  override def checkEditFailureHandlerPermission(releaseId: String): Unit = {
    val releaseInfo = releaseInformation(releaseId).getOrElse(throw PermissionDeniedException.withMessage(s"You cannot edit failure handler on $releaseId"))
    if (releaseInfo.isTemplate()) {
      check(EDIT_TEMPLATE_FAILURE_HANDLER, releaseId)
    } else {
      checkAny(releaseId, EDIT_RELEASE_FAILURE_HANDLER, EDIT_RELEASE_TASK)
    }
  }

  override def canEditTask(releaseId: String): Boolean = {
    val releaseInfo = releaseInformation(releaseId).getOrElse(throw PermissionDeniedException.withMessage(s"You cannot edit task on $releaseId"))
    val permissionToCheck: Permission = if (releaseInfo.isTemplate()) EDIT_TEMPLATE else EDIT_RELEASE_TASK
    hasPermission(permissionToCheck, releaseId)
  }

  override def canViewRelease(releaseId: String): Boolean = {
    hasGlobalPermission(AUDIT_ALL) ||
      isWorkflowExecutionOwner(releaseId) ||
      hasCiPermission(getViewPermission(releaseId), releaseId)
  }

  private def isWorkflowExecutionOwner(releaseId: String): Boolean = {
    releaseInformation(releaseId).fold(false)(_.isWorkflowExecution() && isCurrentUserOwner(releaseId))
  }

  private def isCurrentUserOwner(releaseId: String): Boolean = {
    releaseInformation(releaseId).fold(false) {
      _.owner.exists(_.equalsIgnoreCase(Permissions.getAuthenticatedUserName)) || isCurrentUserAdmin
    }
  }

  private def isWorkflowExecution(releaseId: String): Boolean = {
    releaseInformation(releaseId).fold(false)(_.isWorkflowExecution())
  }

  //noinspection ScalaStyle
  def check(permission: Permission, ciId: String): Unit = {
    // TODO add cache for invocation of this method
    // check if release is workflow and current authenticated user is the owner of the workflow or admin
    val checkOwner = isWorkflowRelatedPermission(permission) && isWorkflowExecution(ciId)
    if (checkOwner) {
      if (!isWorkflowExecutionOwner(ciId)) {
        throw PermissionDeniedException.withMessage(s"You are not the owner of the workflow execution $ciId")
      }
    } else {
      if (!hasPermission(permission, ciId)) {
        throw PermissionDeniedException.forPermission(permission, ciId)
      }
    }
  }

  private[security] def hasPermission(permission: Permission, ciId: String): Boolean = {
    hasGlobalPermission(permission) || hasCiPermission(permission, ciId)
  }

  private[security] def hasCiPermission(permission: Permission, containerId: String): Boolean = {
    val hasRequiredPermission = if (!isPermissionApplicableTo(permission, containerId)) {
      false
    } else {
      releaseInformation(containerId) match {
        case Some(r) =>
          if (r.isArchived) {
            env.hasArchivedReleasePermission(permission, containerId)
          } else {
            if (isWorkflowRelatedPermission(permission) && r.isWorkflowExecution()) {
              isWorkflowExecutionOwner(containerId)
            } else {
              env.hasEffectiveSecuredCiPermission(permission, containerId)
            }
          }
        case None =>
          env.hasEffectiveSecuredCiPermission(permission, containerId)
      }
    }
    hasRequiredPermission
  }

  @varargs
  private[security] def checkAny(ciId: String, permissions: Permission*): Unit = {
    // TODO add cache for invocation of these methods
    for (permission <- permissions) {
      if (hasPermission(permission, ciId)) {
        return
      }
    }
    throw exceptionForPermissions(ciId, permissions)
  }


  private def getViewPermission(releaseId: String): Permission = {
    releaseInformation(releaseId).map(getViewPermission).getOrElse(VIEW_RELEASE)
  }

  private def getViewPermission(releaseInformation: ReleaseInformation): Permission = {
    if (releaseInformation.isTemplate()) {
      VIEW_TEMPLATE
    } else if (releaseInformation.isWorkflowExecution()) {
      VIEW_WORKFLOW_EXECUTION
    } else {
      VIEW_RELEASE
    }
  }

  override def checkAbort(releaseId: String): Unit = {
    //enough to check only in live database for abort operation
    if (isWorkflow(releaseId)) {
      if (!isWorkflowExecutionOwner(releaseId) && !hasPermission(ABORT_WORKFLOW_EXECUTION, releaseId)) {
        throw PermissionDeniedException.withMessage(
          s"You are not the owner of the workflow execution or do not have $ABORT_WORKFLOW_EXECUTION permission on $releaseId"
        )
      }
    } else if (!hasPermission(ABORT_RELEASE, releaseId)) {
      throw PermissionDeniedException.forPermission(ABORT_RELEASE, releaseId)
    }
  }

  private def isWorkflow(releaseId: String): Boolean = {
    releaseInformation(releaseId).fold(false)(_.isWorkflow())
  }

  override def checkEditAttachment(ciId: String): Unit = {
    if (Ids.isTaskId(ciId)) {
      checkIsAllowedToEditAttachmentsOnTask(ciId)
    } else {
      checkEdit(releaseIdFrom(ciId))
    }
  }

  override def checkIsAllowedToEditAttachmentsOnTask(taskId: String): Unit = {
    try {
      val releaseId = releaseIdFrom(taskId)
      if (!hasPermissionToUpdateTask(taskId) && !hasPermission(EDIT_RELEASE_TASK_ATTACHMENT, releaseId)) {
        throw PermissionDeniedException.forNodeAndPrivilege(taskId, "'Task owner', 'Member of task team', 'edit task' or 'edit task attachments'")
      }
    } catch {
      case e: NotFoundException =>
        throw PermissionDeniedException.withMessage(s"You cannot upload attachments on archived task [$taskId]")
    }
  }

  @throws[NotFoundException]("if taskId is not found in repository database")
  private def owns(taskId: String): Boolean = {
    val taskData = getTaskData(taskId) // will blow up with NotFoundException
    if(taskData.owner != null) {
      taskData.owner.equalsIgnoreCase(Permissions.getAuthenticatedUserName)
    } else {
    false
    }
  }

  private[security] def hasPermissionToUpdateTask(taskId: String): Boolean = {
    val releaseId = releaseIdFrom(taskId)
    if (isCurrentUserAdmin || canEditTask(releaseId) || owns(taskId) ) {
      true
    } else {
      Option(getTaskData(taskId).team).exists { team =>
        env.isMemberOrRoleOf(releaseIdFrom(taskId), team)
      }
    }
  }

  override def checkEditTaskConfigurationFacet(releaseId: String): Unit = {
    if (!(canEditTask(releaseId) || canEditTaskConfigurationFacet(releaseId))) {
      throw PermissionDeniedException.withMessage(s"You are not allowed to create facets on tasks of release [$releaseId]")
    }
  }

  private def canEditTaskConfigurationFacet(releaseId: TaskId): Boolean = {
    hasPermission(EDIT_RELEASE_TASK_CONFIGURATION_FACET, releaseId)
  }

  override def checkEditDate(ciId: String): Unit = {
    val releaseId = releaseIdFrom(ciId)
    if (isReleaseId(ciId) || Ids.isPhaseId(ciId)) {
      checkEdit(releaseId)
    } else {
      val releaseInfo = releaseInformation(releaseId).getOrElse(
        throw PermissionDeniedException.withMessage(s"You do not have permission to edit dates on $ciId")
      )
      val permissionsToCheck = if (releaseInfo.isTemplate()) List(EDIT_TEMPLATE) else List(EDIT_RELEASE_TASK, EDIT_TASK_DATES)
      checkAny(releaseId, permissionsToCheck: _*)
    }
  }

  override def checkEditBlackoutPermission(releaseId: TaskId): Unit = {
    val releaseInfo = releaseInformation(releaseId).getOrElse(
      throw PermissionDeniedException.withMessage(s"You do not have permission to edit blackout on $releaseId")
    )
    if (releaseInfo.isTemplate()) {
      check(EDIT_TEMPLATE, releaseId)
    } else {
      checkAny(releaseId, EDIT_RELEASE_TASK, EDIT_BLACKOUT)
    }

  }

  override def checkReassignTaskPermission(releaseId: String): Unit = {
    val relaseInfo = releaseInformation(releaseId).getOrElse(
      throw PermissionDeniedException.withMessage(s"You do not have permission to reassign task on $releaseId")
    )
    if (relaseInfo.isTemplate()) {
      check(EDIT_TEMPLATE, releaseId)
    } else {
      checkAny(releaseId, EDIT_RELEASE_TASK, REASSIGN_RELEASE_TASK)
    }
  }

  override def checkReassignTaskToUser(taskId: TaskId, newUser: TaskId): Unit = {
    try {
      val releaseId = releaseIdFrom(taskId)
      checkReassignTaskPermission(releaseId)
      val task = getTaskData(taskId)
      if (task.locked) {
        checkLockTaskPermissionForAssignment()
      }
    } catch {
      case e: PermissionDeniedException =>
        if (!areUsersInTheSameTaskTeam(taskId, newUser)) {
          throw e
        }
    }
  }

  override def areUsersInTheSameTaskTeam(task: Task, newUser: String): Boolean = {
    areUsersInTheSameTaskTeam(task.getId, newUser)
  }

  private def areUsersInTheSameTaskTeam(taskId: String, newUser: String): Boolean = {
    // REL-2554 members of a task's team can always assign the task to one another
    val currentAuthentication: Authentication = Permissions.getAuthentication
    val task = getTaskData(taskId)
    if (task.team != null) {
      val releaseId = releaseIdFrom(taskId)
      val teamOptional: Option[Team] = env.findTeamByName(releaseId, task.team)
      teamOptional.exists(team => isUserInTeam(currentAuthentication.getName, team) && (newUser == null || isUserInTeam(newUser, team)))
    } else {
      false
    }
  }

  private def isUserInTeam(user: String, team: Team): Boolean = {
    team.hasMember(user) || team.hasAnyRole(env.getRolesFor(user)) || env.isExternalUserPrincipalIsFromTheGroupAssigned(user, team)
  }

  override def isAllowedToWorkOnTask(taskId: TaskId): Boolean = {
    val task = getTaskData(taskId)
    task.isUpdatable && hasPermissionToUpdateTask(taskId)
  }

  override def checkRelevantTaskTransitionPermission(taskId: String): Unit = {
    checkTaskIsUpdatable(taskId)
    val releaseId = releaseIdFrom(taskId)
    val task = getTaskData(taskId)
    if (task.isPlanned) {
      if (!hasAdvanceTaskTransitionPermission(releaseId)) {
        throw PermissionDeniedException.withMessage(s"You cannot make advance task transition for task=[$taskId]. You need to have permission or be release owner")
      }
    } else {
      if (!hasTaskTransitionPermission(taskId)) {
        throw PermissionDeniedException.withMessage(s"You cannot make task transition for task=[$taskId]. You need to have permission or be either release owner or task owner")
      }
    }
  }

  private[security] def isTaskUpdateable(taskId: TaskId): Boolean = {
    val task = getTaskData(taskId)
    task.isUpdatable
  }

  private[security] def checkTaskIsUpdatable(taskId: String): Unit = {
    if (!isTaskUpdateable(taskId)) {
      // TODO should this be a permission denied exception?
      throw new IllegalArgumentException("You can not work with a defunct or done in advance task")
    }
  }

  private def hasAdvanceTaskTransitionPermission(releaseId: String): Boolean = {
    isCurrentUserAdmin || canEditTask(releaseId) || hasCiPermission(ADVANCE_TASK_TRANSITION, releaseId)
  }

  private[security] def hasTaskTransitionPermission(taskId: String): Boolean = {
    val allowReleaseOwnerTaskTransition = env.xlrConfig.isReleaseOwnerTaskTransitionAllowed
    val releaseId = releaseIdFrom(taskId)
    (allowReleaseOwnerTaskTransition && isCurrentUserOwner(releaseId)) || hasPermissionToUpdateTask(taskId) || hasCiPermission(TASK_TRANSITION, releaseId)
  }

  override def checkHasPermissionsToUpdateTask(task: Task): Unit = {
    checkHasPermissionsToUpdateTask(task.getId)
  }

  private def checkHasPermissionsToUpdateTask(taskId: String): Unit = {
    if (!hasPermissionToUpdateTask(taskId)) {
      throw PermissionDeniedException.forNodeAndPrivilege(taskId, "'Task owner', 'Member of task team' or 'edit task'")
    }
  }

  override def checkIsAllowedToWorkOnTask(taskId: String): Unit = {
    checkTaskIsUpdatable(taskId)
    checkHasPermissionsToUpdateTask(taskId)
  }

  override def checkForLockTaskPermissionForTransition(task: Task): Unit = {
    val isLockedTask = task.isLocked
    if (isLockedTask && !(isCurrentUserAdmin || hasLockTaskTransitionPermissions(task: Task))) {
      throw PermissionDeniedException.withMessage(s"You cannot make task transition for a locked task=[${task.getId}]. You need to have locked task assigned to yourself or be part of the assigned team of the locked task")
    }
  }

  private def hasLockTaskTransitionPermissions(task: Task): Boolean = {
    var canTransitionLockedTask = false

    if(isCurrentUserAdmin) {
      canTransitionLockedTask = true
    }

    if(!canTransitionLockedTask) {
      val release: Release = task.getRelease
      val currentAuthentication: Authentication = Permissions.getAuthentication

      if(!canTransitionLockedTask && task.getOwner != null) {
        canTransitionLockedTask = owns(task.getId)
      }
      if (!canTransitionLockedTask && task.hasTeam) {
        val team: Option[Team] = env.findTeamByName(release.getId, task.getTeam)
        canTransitionLockedTask = team != null && isUserInTeam(currentAuthentication.getName, team.get)
      }
    }
    canTransitionLockedTask
  }

  override def checkIsAllowedToCommentOnTask(taskId: String): Unit = {
    val taskReleaseId = releaseIdFrom(taskId)
    val releaseInfo = releaseInformation(taskReleaseId).getOrElse(throw PermissionDeniedException.withMessage(s"You cannot comment task $taskId"))
    if (releaseInfo.isArchived) {
      throw PermissionDeniedException.withMessage(s"You cannot comment on archived task [$taskId]")
    }
    if (!isCurrentUserAdmin) {
      checkView(taskReleaseId)
    }
  }


  private[security] def checkCanOverrideBlackout(taskId: String): Unit = {
    val releaseId = releaseIdFrom(taskId)
    if (!hasCiPermission(EDIT_BLACKOUT, releaseId) && !hasCiPermission(EDIT_RELEASE_TASK, releaseId)) {
      throw PermissionDeniedException.forNodeAndPrivilege(taskId, "'Edit task blackout'")
    }
  }

  override def checkTaskTransitionPermission(taskId: TaskId): Unit = {
    checkTaskIsUpdatable(taskId)
    if (!hasTaskTransitionPermission(taskId)) {
      throw PermissionDeniedException.withMessage(s"You cannot make task transition for task=[$taskId]. You need to have permission or be either release owner or task owner")
    }
  }

  override def checkAdvanceTaskTransitionPermission(taskId: String): Unit = {
    checkTaskIsUpdatable(taskId)
    val task = getTaskData(taskId)
    val releaseId = releaseIdFrom(taskId)
    if (task.isPlanned && !hasAdvanceTaskTransitionPermission(releaseId)) {
      throw PermissionDeniedException.withMessage(s"You cannot make advance task transition for task=[$taskId]. You need to have permission or be release owner")
    }
  }

}
