package com.xebialabs.deployit.plumbing;

import java.io.Closeable;
import java.io.IOException;
import java.io.Writer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.overthere.OverthereExecutionOutputHandler;
import com.xebialabs.xlrelease.utils.SensitiveValueScrubber;

public class ExecutionOutputWriter extends Writer {

    private static final Logger logger = LoggerFactory.getLogger(ExecutionOutputWriter.class);

	private final SensitiveValueScrubber scrubber;
	protected Writer out;
    protected OverthereExecutionOutputHandler[] outputHandlers;

    public ExecutionOutputWriter(final SensitiveValueScrubber scrubber, Writer out, OverthereExecutionOutputHandler... outputHandlers) {
        super(out);
	    this.scrubber = scrubber;
	    this.out = out;
        this.outputHandlers = outputHandlers;
    }

    @Override
    public void write(final char[] cbuf, final int off, final int len) throws IOException {
	    if(len == 1) {
		    out.write(cbuf, off, len);
		    notifyChar(cbuf[off]);
	    } else {
		    final String line = String.valueOf(cbuf, off, len);
		    String cleanLine = scrubber.scrubValues(line);
		    out.write(cleanLine);
		    notifyLine(cleanLine);
        }
    }

    private void notifyChar(final char c) {
        for (OverthereExecutionOutputHandler outputHandler : outputHandlers) {
            outputHandler.handleChar(c);
        }
    }

    protected void notifyLine(final String line) {
        for (OverthereExecutionOutputHandler outputHandler : outputHandlers) {
            outputHandler.handleLine(line);
        }
    }

    @Override
    public void flush() throws IOException {
        out.flush();
    }

    @Override
    public void close() throws IOException {
        closeHandlers();
        out.close();
    }

    protected void closeHandlers() throws IOException {
        logger.debug("Closing {} output handlers", outputHandlers.length);
        for (OverthereExecutionOutputHandler outputHandler : outputHandlers) {
            if (outputHandler instanceof Closeable) {
                ((Closeable) outputHandler).close();
            }
        }
    }

    @Override
    public String toString() {
        return out.toString();
    }

    public Writer getWriter() {
        return out;
    }
}
