package com.xebialabs.deployit.plumbing.authorization;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authorization.AuthorizationDecision;
import org.springframework.security.authorization.AuthorizationManager;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;

import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.security.authentication.RunnerAuthenticationToken;
import com.xebialabs.xlrelease.service.UserLastActiveActorService;
import com.xebialabs.xlrelease.service.UserProfileService;

import static com.xebialabs.xlrelease.security.XLReleasePermissions.ADMIN_USERNAME;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.isAdmin;

public class LoginAuthorizationManager<T> implements AuthorizationManager<T> {

    private static final Logger logger = LoggerFactory.getLogger(LoginAuthorizationManager.class);

    private final UserProfileService userProfileService;
    private final UserLastActiveActorService userLastActiveActorService;

    public LoginAuthorizationManager(final UserProfileService userProfileService,
                                     final UserLastActiveActorService userLastActiveActorService) {
        this.userProfileService = userProfileService;
        this.userLastActiveActorService = userLastActiveActorService;
    }

    @Override
    public AuthorizationDecision check(final Supplier<Authentication> authentication, final T object) {
        boolean granted = isGranted(authentication.get());
        return new AuthorizationDecision(granted);
    }

    private boolean isGranted(Authentication authentication) {
        return authentication != null && authentication.isAuthenticated() && isAuthorized(authentication);
    }

    private boolean isAuthorized(Authentication authentication) {
        String username = authentication.getName();
        logger.debug("Checking if login allowed for [{}]", username);

        if (authentication instanceof RunnerAuthenticationToken) {
            return true;
        }

        if (isAdmin(username)) {
            userLastActiveActorService.updateLastActive(ADMIN_USERNAME);
            return true;
        }

        UserProfile userProfile = userProfileService.findByUsername(username);

        if (userProfile != null && userProfile.isLoginAllowed()) {
            logger.debug("User [{}] is authorized for login", username);
            onAccessGranted(authentication, userProfile);
            return true;
        }

        logger.error("User [{}] is not authorized for login", username);
        return false;
    }

    private void onAccessGranted(Authentication authentication, UserProfile userProfile) {
        // Copy of LoginPermissionVoter.onAccessGranted in 23.1 or older version

        userLastActiveActorService.updateLastActive(userProfile.getCanonicalId());

        // TODO: this does not belong here
        // TODO: https://digitalai.atlassian.net/browse/REL-1768
        // TODO: There really isn't an appropriate place to put this... you need this role to be returned by all AuthenticationProviders in the system,
        //  but that can't be forced without using aspect loadtime weaving or some other hacky way.
        var newAuthentication = addAuthenticatedUserAuthority(authentication);

        var newCtxt = SecurityContextHolder.createEmptyContext();
        newCtxt.setAuthentication(newAuthentication);
        SecurityContextHolder.setContext(newCtxt);
    }

    public static Authentication addAuthenticatedUserAuthority(Authentication authentication) {
        var oldAuthorities = authentication.getAuthorities();
        var authenticatedUserAuthority = new SimpleGrantedAuthority(XLReleasePermissions.AUTHENTICATED_USER);
        Collection<? extends GrantedAuthority> newAuthorities;
        if (!oldAuthorities.contains(authenticatedUserAuthority)) {
            Collection<GrantedAuthority> modifiableAuthorities = new HashSet<>();
            modifiableAuthorities.add(authenticatedUserAuthority);
            modifiableAuthorities.addAll(oldAuthorities);
            newAuthorities = Collections.unmodifiableCollection(modifiableAuthorities);
        } else {
            return authentication;
        }
        try {
            Field authorities = findAuthoritiesField(authentication.getClass());
            authorities.setAccessible(true);
            authorities.set(authentication, newAuthorities);
        } catch (Exception ex) {
            logger.error("Unable to set authenticated-user authority into authentication", ex);
        }

        return authentication;
    }

    private static Field findAuthoritiesField(Class<?> clazz) throws NoSuchFieldException {
        while (clazz != null) {
            try {
                return clazz.getDeclaredField("authorities");
            } catch (NoSuchFieldException e) {
                clazz = clazz.getSuperclass();
            }
        }
        throw new NoSuchFieldException("Field 'authorities' not found in the class hierarchy.");
    }
}
