package com.xebialabs.xlrelease.repository.query

import grizzled.slf4j.Logging
import org.springframework.data.domain.{Page, PageRequest, Pageable, ReleasePageImpl}
import org.springframework.jdbc.core.RowMapper
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate

import scala.jdk.CollectionConverters._

case class PageableQueryResult[R](results: java.util.List[R], totalCount: Long)

trait Query[R] {
  def execute(): R
}

trait PageableQuery[R] {
  def execute(): Page[R]
}

class SqlQuery[R](namedTemplate: NamedParameterJdbcTemplate, val sqlQuery: String, val params: Map[String, Any], val rowMapper: RowMapper[R])
  extends Query[R] with Logging {
  override def execute(): R = {
    val NL = System.lineSeparator()
    logger.debug(s"Executing object query:$NL$sqlQuery${NL}with parameters:$NL${params.mkString("'", ",", ",")}")
    namedTemplate.queryForObject[R](sqlQuery, params.asJava, rowMapper)
  }
}

class SqlListQuery[T](namedTemplate: NamedParameterJdbcTemplate, val sqlQuery: String, val params: Map[String, Any], val rowMapper: RowMapper[T])
  extends Query[java.util.List[T]] with Logging {
  override def execute(): java.util.List[T] = {
    val NL = System.lineSeparator()
    logger.debug(s"Executing list query:$NL$sqlQuery${NL}with parameters:$NL${params.mkString("'", ",", "'")}")
    namedTemplate.query(sqlQuery, params.asJava, rowMapper)
  }
}

class SqlPageableQuery[R](namedTemplate: NamedParameterJdbcTemplate,
                          pageable: Pageable,
                          val resultsQuery: SqlListQuery[R],
                          val totalCountQuery: SqlQuery[Long]
                         ) extends PageableQuery[R] {
  override def execute(): Page[R] = {
    val queryPageable = if (pageable == null) {
      PageRequest.of(0, Int.MaxValue)
    } else {
      pageable
    }
    val results = resultsQuery.execute()
    val totalCount = totalCountQuery.execute()
    new ReleasePageImpl[R](results, queryPageable, totalCount)
  }
}
