package com.xebialabs.xlrelease.scheduler.logs

import com.xebialabs.xlrelease.scheduler.logs.ExecutionLogWatchActor._
import com.xebialabs.xlrelease.service.SseService
import com.xebialabs.xlrelease.storage.domain.JobEntryRef
import com.xebialabs.xlrelease.storage.service.StorageService
import com.xebialabs.xlrelease.support.pekko.spring.SpringActor
import org.apache.commons.io.IOUtils
import org.apache.pekko.actor.{Actor, NoSerializationVerificationNeeded, ReceiveTimeout}
import org.apache.pekko.event.LoggingReceive
import org.apache.pekko.event.slf4j.SLF4JLogging

import java.net.URI
import java.time.{Duration, Instant}
import scala.concurrent.duration.DurationInt
import scala.util.Using

@SpringActor
class ExecutionLogWatchActor(taskExecutionLogService: TaskExecutionLogService, storageService: StorageService, sseService: SseService)
  extends Actor with SLF4JLogging {
  var executionId: String = _
  var taskId: String = _
  var lastEventTimestamp: Instant = Instant.now()

  context.setReceiveTimeout(5.seconds)

  override def receive: Receive = LoggingReceive {
    case m: WatcherMsg => handleWatchMsgs(m)
    case ReceiveTimeout => self ! Check(executionId)
  }

  override def postStop(): Unit = {
    if (executionId != null) {
      sseService.sendEvent(executionId, END_EVENT, "")
      sseService.unsubscribeAllUsers(executionId)
    }
  }

  //noinspection ScalaStyle
  private def handleWatchMsgs(watcherMsg: WatcherMsg): Unit = watcherMsg match {
    case StartWatch(taskId, executionId, username) =>
      this.executionId = executionId
      this.taskId = taskId
      sseService.subscribeTopicToUser(executionId, username)
      streamCurrentState(executionId)
      streamLastChunk(executionId)
      self ! Check(executionId)
    case StopWatching(executionId, Some(username)) =>
      sseService.unsubscribeTopicToUser(executionId, username)
      self ! Check(executionId)
    case StopWatching(executionId, None) =>
      sseService.sendEvent(executionId, END_EVENT, "")
      sseService.unsubscribeAllUsers(executionId)
      self ! Check(executionId)
    case Pong(_) =>
      this.lastEventTimestamp = Instant.now()
    case Check(executionId) =>
      val hasUsersFollowingExecution = sseService.hasActiveUsers(executionId)
      // should unsubscribe... any msg that arrives will be sent to dead letter
      if (!hasUsersFollowingExecution) {
        context.stop(self)
      } else {
        val maybeExecutionEntry = taskExecutionLogService.getTaskExecutionEntry(taskId, executionId)
        val executionIsCompleted = maybeExecutionEntry.exists(_.endDate.isDefined)
        if (executionIsCompleted) {
          self ! StopWatching(executionId, None)
        } else {
          // if there was no new entry for certain amount of time ... send ping so we would close closed sinks
          val durationSinceLastEvent = Duration.between(lastEventTimestamp, Instant.now()).toSeconds
          if (durationSinceLastEvent > 10 && durationSinceLastEvent < 60) {
            sseService.sendEvent(executionId, PING_SSE_EVENT, new String(":\n\n"))
          } else if (durationSinceLastEvent >= 60) {
            self ! StopWatching(executionId, None)
          }
        }
      }
    case NewEntry(executionId, newEntryUris) =>
      this.lastEventTimestamp = Instant.now()
      newEntryUris.sorted.foreach(uri => sendLogs(executionId, uri))
  }

  private def streamCurrentState(executionId: String): Unit = {
    val (status, jobId, chunk) = taskExecutionLogService.getTaskExecutionEntry(taskId, executionId) match {
      case Some(row) => (if (row.endDate.isEmpty) "in_progress" else "closed", row.lastJob, row.lastChunk)
      case None => ("unknown", -1, -1)
    }
    val payload = s"$status, $jobId, $chunk"
    sseService.sendEvent(executionId, EXECUTION_LOG_STATUS_EVENT, new String(payload))
  }

  private def streamLastChunk(executionId: String): Unit = {
    val maybeRow = taskExecutionLogService.getTaskExecutionEntry(taskId, executionId)
    maybeRow.foreach { row =>
      // row -> log entry ref
      val taskIdHash = row.taskIdHash
      val executionId = row.executionId
      val jobId = row.lastJob
      val chunk = row.lastChunk
      val uriPath = s"/jobs/$taskIdHash/$executionId/$jobId/$chunk"
      val uriScheme = storageService.defaultStorageType()
      val entryUri = URI.create(s"$uriScheme://$uriPath")
      sendLogs(executionId, entryUri)
    }
  }

  private def sendLogs(executionId: String, entryUri: URI): Unit = {
    val payload = Using.resource(storageService.get(JobEntryRef(entryUri)))(IOUtils.toByteArray)
    sseService.sendEvent(executionId, LOG_CHUNK_ENTRY_CREATED_EVENT, new String(payload))
  }
}

object ExecutionLogWatchActor {

  final val LOG_CHUNK_ENTRY_CREATED_EVENT: String = "execution-log-chunk-entry-created"

  final val PING_SSE_EVENT: String = "execution-ping"
  final val END_EVENT: String = "execution-end"

  final val EXECUTION_LOG_STATUS_EVENT: String = "execution-status" // in_progress or closed, last job_id and last_chunk?

  sealed trait WatcherMsg {
    def executionId: String
  }

  case class StartWatch(taskId: String, executionId: String, username: String) extends WatcherMsg with NoSerializationVerificationNeeded

  case class StopWatching(executionId: String, username: Option[String]) extends WatcherMsg

  case class Check(executionId: String) extends WatcherMsg

  case class Pong(executionId: String) extends WatcherMsg

  case class NewEntry(executionId: String, newEntryUris: List[URI]) extends WatcherMsg
}
