package com.xebialabs.xlrelease.security.authentication;

import java.util.Collection;
import java.util.Optional;
import org.springframework.ldap.core.DirContextOperations;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.ldap.userdetails.LdapUserDetailsMapper;

/**
 * LdapUserDetailsMapper implementation to get original username from LDAP by using proper attribute name
 */
public class UsernameAttributeLdapContextMapper extends LdapUserDetailsMapper {

    /* This attribute is used to retrieve the actual user name in LDAP.
        For Active Directory, it's "samAccountName".
        Note: this property doesn't exist in other LDAP implementations. "cn" might be used instead.
     */
    private String ldapUsernameAttribute = "samAccountName";

    @Override
    public UserDetails mapUserFromContext(final DirContextOperations ctx,
                                          final String username,
                                          final Collection<? extends GrantedAuthority> authorities) {
        return Optional
                .ofNullable(ctx.getStringAttribute(getLdapUsernameAttribute()))
                .map(originalUsername -> super.mapUserFromContext(ctx, originalUsername, authorities))
                .orElseThrow(() -> new IllegalStateException(String.format("LDAP attribute not found: %s", getLdapUsernameAttribute())));
    }

    /**
     * @param ldapUsernameAttribute Sets the name of the LDAP attribute to use for the username after authentication.
     */
    public void setLdapUsernameAttribute(String ldapUsernameAttribute) {
        this.ldapUsernameAttribute = ldapUsernameAttribute;
    }

    public String getLdapUsernameAttribute() {
        return this.ldapUsernameAttribute;
    }

}
