package com.xebialabs.xlrelease.spring.configuration;

import java.util.EnumSet;
import jakarta.servlet.DispatcherType;
import javax.sql.DataSource;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.convert.ConversionService;
import org.springframework.core.convert.support.DefaultConversionService;
import org.springframework.core.convert.support.GenericConversionService;
import org.springframework.core.serializer.support.DeserializingConverter;
import org.springframework.core.serializer.support.SerializingConverter;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;
import org.springframework.security.core.session.SessionRegistry;
import org.springframework.session.jdbc.JdbcIndexedSessionRepository;
import org.springframework.session.jdbc.config.annotation.SpringSessionDataSource;
import org.springframework.session.jdbc.config.annotation.web.http.JdbcHttpSessionConfiguration;
import org.springframework.session.security.SpringSessionBackedSessionRegistry;
import org.springframework.session.web.http.SessionRepositoryFilter;
import org.springframework.transaction.PlatformTransactionManager;

import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.security.SpringSessionConverterFactory;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.repository.SpringSessionRepository;

@Configuration
@ConditionalOnProperty(value = "xl.server.session.storage.enabled", havingValue = "true")
public class SpringSessionConfiguration extends JdbcHttpSessionConfiguration {

    private final SpringSessionConverterFactory converterFactory;
    private final String cleanupCron;
    private ClassLoader classLoader;

    public SpringSessionConfiguration(@Autowired XlrConfig xlrConfig,
                                      @Autowired ServerConfiguration serverConfiguration,
                                      @Autowired(required = false) SpringSessionConverterFactory converterFactory) {
        this.converterFactory = converterFactory;

        //If timeout is set to 0, set it as 30 days default for session persistence to purge data
        int sessionTimeoutSeconds = serverConfiguration.getClientSessionTimeoutMinutes() * 60;
        int maxInactiveInterval = sessionTimeoutSeconds == 0 ? 2592000 : sessionTimeoutSeconds;
        super.setMaxInactiveIntervalInSeconds(maxInactiveInterval);
        this.cleanupCron = xlrConfig.server_session_storage_cleanupCron();
        super.setCleanupCron(cleanupCron);
    }

    @Override
    @Autowired
    public void setDataSource(@SpringSessionDataSource ObjectProvider<DataSource> springSessionDataSource,
                              @Qualifier("xlrRepositoryDataSourceProxy") ObjectProvider<DataSource> dataSource
    ) {
        super.setDataSource(springSessionDataSource, dataSource);
    }

    @Override
    @Autowired
    public void setTransactionManager(@Qualifier("xlrRepositoryTransactionManager") PlatformTransactionManager transactionManager) {
        super.setTransactionManager(transactionManager);
    }

    @Override
    public void setBeanClassLoader(ClassLoader classLoader) {
        super.setBeanClassLoader(classLoader);
        this.classLoader = classLoader;
    }

    @Bean
    @Override
    public JdbcIndexedSessionRepository sessionRepository() {
        JdbcIndexedSessionRepository repo = super.sessionRepository();
        ConversionService conversionService = createConversionServiceWithBeanClassLoader();
        repo.setConversionService(conversionService);
        return repo;
    }

    @Bean
    public SessionRegistry sessionRegistry() {
        return new SpringSessionBackedSessionRegistry<>(sessionRepository());
    }


    @Bean
    public FilterRegistrationBean<SessionRepositoryFilter<?>> sessionRepositoryFilterRegistration(SessionRepositoryFilter<?> filter) {
        FilterRegistrationBean<SessionRepositoryFilter<?>> registration = new FilterRegistrationBean<>(filter);
        registration.setDispatcherTypes(EnumSet.of(DispatcherType.ASYNC, DispatcherType.ERROR, DispatcherType.REQUEST));
        registration.setOrder(SessionRepositoryFilter.DEFAULT_ORDER);
        return registration;
    }

    private GenericConversionService createConversionServiceWithBeanClassLoader() {
        DefaultConversionService conversionService = new DefaultConversionService();
        if (this.converterFactory != null) {
            conversionService.addConverter(this.converterFactory.buildSerializer(classLoader));
            conversionService.addConverter(this.converterFactory.buildDeserializer(classLoader));
        } else {
            conversionService.addConverter(Object.class, byte[].class,
                    new SerializingConverter());
            conversionService.addConverter(byte[].class, Object.class,
                    new DeserializingConverter(this.classLoader));
        }
        return conversionService;
    }

    /**
     * Configuration of scheduled job for cleaning up expired sessions.
     */
    @EnableScheduling
    @Configuration(proxyBeanMethods = false)
    class EmptyPrincipalCleanupConfiguration implements SchedulingConfigurer {

        private final SpringSessionRepository springSessionRepository;

        EmptyPrincipalCleanupConfiguration(SpringSessionRepository springSessionRepository) {
            this.springSessionRepository = springSessionRepository;
        }

        @Override
        public void configureTasks(ScheduledTaskRegistrar taskRegistrar) {
            long delayDuration = 5 * 60 * 1000L;
            taskRegistrar.addFixedDelayTask(this.springSessionRepository::cleanUpSessionsWithEmptyPrincipal, delayDuration);
        }

    }

}
