package com.xebialabs.xlrelease.utils

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind._
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.config.XlrConfig

import java.util
import java.util.regex.Pattern
import java.util.{Optional, List => JList, Set => JSet}
import scala.jdk.CollectionConverters._

object PasswordVerificationUtils {

  private val acceptEncryptedSecrets = XlrConfig.getInstance.security.acceptEncryptedSecrets.enabled
  private val passwordEncrypter = PasswordEncrypter.getInstance()
  private val maskPattern = Pattern.compile("^[*]+$")

  val PASSWORD_MASK: String = "********"

  /**
   * Checks whether the value is encoded and encrypted or not.
   *
   * @param value the value
   * @return true, if the value is encoded and encrypted or not.
   */
  def isValueEncrypted(value: String): Boolean = !isNullOrEmpty(value) && passwordEncrypter.isEncodedAndDecryptable(value)

  /**
   * Checks whether the value is masked or not.
   * If value only contains `<b>*</b>`, it will be considered as masked. i.e. `<b>***</b>` or `<b>*******</b>`.
   *
   * @param value the value
   * @return true, if the value is masked.
   */
  def isMaskedPassword(value: String): Boolean = !isNullOrEmpty(value) && maskPattern.matcher(value).matches()

  /**
   * Returns original value if updated value is masked.
   *
   * @param original the original value
   * @param updated  the updated value
   * @return replaced value
   * @throws IllegalArgumentException If given updated value is not same as original and it is encrypted
   */
  def replacePasswordIfNeeded(original: String, updated: String): String = {
    // original != updated is added to allow users to pass original encrypted value in script task
    if (!acceptEncryptedSecrets && original != updated && isValueEncrypted(updated)) {
      throw new IllegalArgumentException("Could not accept encrypted password")
    }
    if (isMaskedPassword(updated)) original else updated
  }

  /**
   * Returns original value if updated value is masked.
   *
   * @param original the original value
   * @param updated  the updated value
   * @return replaced value
   * @throws IllegalArgumentException If given updated value is not same as original and it is encrypted.
   */
  def replacePasswordIfNeeded(original: AnyRef, updated: AnyRef): String = {
    val originalValue = if (null == original) null else original.asInstanceOf[String]
    val updatedValue = if (null == updated) null else updated.asInstanceOf[String]
    replacePasswordIfNeeded(originalValue, updatedValue)
  }

  /**
   * Replaces password properties in updated CI with original if it's masked.
   * If original CI is <i>None</i>, password properties will be replaced with <i>null</i> if it is masked.
   *
   * @param original the original CI
   * @param updated  the updated CI
   * @throws IllegalArgumentException If in given updated CI any password property value is not same as original and it is encrypted.
   */
  def replacePasswordPropertiesInCiIfNeededJava(original: Optional[ConfigurationItem], updated: ConfigurationItem): Unit = {
    val ci = if (original.isPresent) Some(original.get()) else None
    replacePasswordPropertiesInCiIfNeeded(ci, updated)
  }

  /**
   * Replaces password properties in updated CI with original if it's masked.
   * If original CI is <i>None</i>, password properties will be replaced with <i>null</i> if it is masked.
   *
   * It will ignore all external references except direct nestedCi references.
   *
   * @param original the original CI
   * @param updated  the updated CI
   * @throws IllegalArgumentException If in given updated CI any password property value is not same as original and it is encrypted.
   */
  def replacePasswordPropertiesInCiIfNeeded(original: Option[ConfigurationItem], updated: ConfigurationItem): Unit = {
    if (null != updated) {
      val externalReferences = CiHelper.getExternalReferences(updated).asScala.map(_.getId).toList

      val originalNestedCis = original match {
        case Some(ci) => Some(CiHelper.getNestedCis(ci).asScala)
        case None => None
      }
      val updatedNestedCis = CiHelper.getNestedCis(updated).asScala

      updatedNestedCis
        .filterNot(nestedCi => null != nestedCi.getId && externalReferences.contains(nestedCi.getId))
        .foreach(nestedCi => replacePasswordPropertiesIfNeeded(originalNestedCis match {
          case Some(cis) =>
            if (isNullOrEmpty(nestedCi.getId)) {
              None
            } else {
              cis.find(t => t.getType == nestedCi.getType && t.getId == nestedCi.getId)
            }
          case None => None
        }, nestedCi, externalReferences))
      replacePasswordInDirectCiRefs(original, updated, externalReferences)
    }
  }

  private def replacePasswordInDirectCiRefs(original: Option[ConfigurationItem], updated: ConfigurationItem, externalReferences: List[String]): Unit = {
    updated.getType.getDescriptor.getPropertyDescriptors.asScala foreach {
      case pd if pd.getKind == CI =>
        val references: List[String] = if (pd.isNested) List.empty else externalReferences
        replacePasswordPropertiesIfNeeded(original match {
          case Some(configurationItem) => Some(configurationItem.getProperty(pd.getName).asInstanceOf[ConfigurationItem])
          case None => None
        }, updated.getProperty(pd.getName).asInstanceOf[ConfigurationItem], references)
      case pd if pd.getKind == SET_OF_CI =>
        val updatedCis = updated.getProperty(pd.getName).asInstanceOf[JSet[ConfigurationItem]]
        replacePasswordPropertiesIfNeeded(original match {
          case Some(configurationItem) => Some(configurationItem.getProperty(pd.getName).asInstanceOf[JSet[ConfigurationItem]].asScalaMap)
          case None => None
        }, updatedCis.asScalaMap, externalReferences)
        replacePasswordPropertiesIfNeeded(updatedCis.getNewCiList, externalReferences)
      case pd if pd.getKind == LIST_OF_CI =>
        val updatedCis = updated.getProperty(pd.getName).asInstanceOf[JList[ConfigurationItem]]
        replacePasswordPropertiesIfNeeded(original match {
          case Some(configurationItem) => Some(configurationItem.getProperty(pd.getName).asInstanceOf[JList[ConfigurationItem]].asScalaMap)
          case None => None
        }, updatedCis.asScalaMap, externalReferences)
        replacePasswordPropertiesIfNeeded(updatedCis.getNewCiList, externalReferences)
      case _ => //do nothing
    }
  }

  private def replacePasswordPropertiesIfNeeded(updated: List[ConfigurationItem], externalReferences: List[String]): Unit = {
    updated.foreach(ci => replacePasswordPropertiesIfNeeded(None, ci, externalReferences))
  }

  private def replacePasswordPropertiesIfNeeded(original: Option[Map[String, ConfigurationItem]], updated: Map[String, ConfigurationItem], externalReferences: List[String]): Unit = {
    updated.foreach {
      case (name, ci) => replacePasswordPropertiesIfNeeded(original match {
        case Some(configurationItem) => configurationItem.get(name)
        case None => None
      }, ci, externalReferences)
    }
  }

  private def replacePasswordPropertiesIfNeeded(original: Option[ConfigurationItem], updated: ConfigurationItem, externalReferences: List[String]): Unit = {
    def isStringPassword(pd: PropertyDescriptor) = {
      pd.isPassword && pd.getKind == STRING
    }

    if (null != updated && (null == updated.getId || !externalReferences.contains(updated.getId))) {
      for (pd <- updated.getType.getDescriptor.getPropertyDescriptors.asScala) {
        // List, Set or Map of passwords is not supported yet
        if (isStringPassword(pd)) {
          updated.setProperty(pd.getName, replacePasswordIfNeeded(original match {
            case Some(ci) =>
              val originalPd = ci.getType.getDescriptor.getPropertyDescriptor(pd.getName)
              if (originalPd != null && isStringPassword(originalPd)) ci.getProperty(pd.getName).asInstanceOf[String] else null
            case None => null
          }, updated.getProperty(pd.getName)))
        }
      }
    }
  }

  private implicit class ConfigurationItemExtension(cis: util.Collection[ConfigurationItem]) {
    def asScalaMap: Map[String, ConfigurationItem] = {
      if (cis == null) {
        Map.empty[String, ConfigurationItem]
      } else {
        cis.asScala.filterNot(c => isNullOrEmpty(c.getId)).map(ci => ci.getName -> ci).toMap
      }
    }

    def getNewCiList: List[ConfigurationItem] = {
      if (cis == null) {
        List.empty[ConfigurationItem]
      } else {
        cis.asScala.filter(c => isNullOrEmpty(c.getId)).toList
      }
    }
  }

}
