package com.xebialabs.xlrelease.security

import com.google.common.base.Strings
import com.xebialabs.deployit.checks.Checks
import com.xebialabs.deployit.security.Permissions.getAuthentication
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.authentication.AuthenticationFailureException
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.permission.PlatformPermissions.{ADMIN, EDIT_SECURITY}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.domain.{Release, ReleaseKind, Task, Team}
import com.xebialabs.xlrelease.principaldata.PrincipalDataProvider
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.query.TaskBasicData
import com.xebialabs.xlrelease.security.PermissionChecker.{GLOBAL_SECURITY_ALIAS, exceptionForPermissions, isPermissionApplicableTo, isWorkflowRelatedPermission}
import com.xebialabs.xlrelease.security.XLReleasePermissions._
import com.xebialabs.xlrelease.security.XlrPermissionLabels._
import com.xebialabs.xlrelease.security.{PermissionChecker => XlrPermissionChecker}
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.service.{ArchivingService, CalendarService, TeamService}
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils

import java.util
import java.util.function.{Consumer, Supplier}
import java.util.{Date, List => JList}
import scala.annotation.varargs
import scala.jdk.CollectionConverters._
import scala.jdk.StreamConverters._

//noinspection ScalaStyle
@Component("permissionCheckerWithCache")
class PermissionCheckerWithCache(permissionEnforcer: PermissionEnforcer,
                                 releaseRepository: ReleaseRepository,
                                 taskRepository: TaskRepository,
                                 val roleService: RoleService,
                                 val teamService: TeamService,
                                 calendarService: CalendarService,
                                 securedCis: SecuredCis,
                                 archivingService: ArchivingService,
                                 xlrConfig: XlrConfig,
                                 principalDataProvider: PrincipalDataProvider)
  extends XlrPermissionChecker with PermissionCheckerSharedLogic {

  def check(permission: Permission): Unit = {
    if (!hasGlobalPermission(permission)) {
      throw PermissionDeniedException.forPermission(permission, null.asInstanceOf[String])
    }
  }

  def check(permission: Permission, ciId: String): Unit = {
    context().check(permission, ciId)
  }

  def check(permission: Permission, release: Release): Unit = {
    // check if release is workflow and current authenticated user is the owner of the workflow or admin
    if (isWorkflowRelatedPermission(permission) && isWorkflowExecution(release)) {
      if (!isWorkflowExecutionOwner(release)) {
        throw PermissionDeniedException.withMessage(s"You are not the owner of the workflow execution ${release.getId}")
      }
    } else if (!hasPermission(permission, release)) {
      throw PermissionDeniedException.forPermission(permission, release.getId)
    }
  }

  @varargs
  def checkAny(ciId: String, permissions: Permission*): Unit = {
    for (permission <- permissions) {
      if (hasPermission(permission, ciId)) {
        return
      }
    }
    throw exceptionForPermissions(ciId, permissions)
  }

  @varargs
  def checkAny(permissions: Permission*): Unit = {
    for (permission <- permissions) {
      if (hasGlobalPermission(permission)) return
    }
    throw exceptionForPermissions(permissions: _*)
  }


  def checkView(releaseId: String): Unit = {
    context().checkView(releaseId)
  }

  def checkView(release: Release): Unit = {
    if (!hasGlobalPermission(AUDIT_ALL) && !isWorkflowExecutionOwner(release)) {
      check(getViewPermission(release), release)
    }
  }

  def canViewRelease(releaseId: String): Boolean = {
    context().canViewRelease(releaseId)
  }

  private def isWorkflowExecutionOwner(release: Release): Boolean = {
    if (isWorkflowExecution(release)) {
      release.hasOwner(Permissions.getAuthenticatedUserName) || isCurrentUserAdmin
    } else {
      false
    }
  }

  private def isWorkflowExecution(release: Release): Boolean = {
    !release.isTemplate && release.isWorkflow
  }

  def checkEditAttachment(ciId: String): Unit = {
    context().checkEditAttachment(ciId)
  }

  def checkEdit(releaseId: String): Unit = {
    context().checkEdit(releaseId)
  }

  def checkAbort(releaseId: String): Unit = {
    context().checkAbort(releaseId)
  }

  def checkEditTask(releaseId: String): Unit = {
    context().checkEditTask(releaseId)
  }

  def checkEditDate(ciId: String): Unit = {
    context().checkEditDate(ciId)
  }

  def canEditTask(releaseId: String): Boolean = {
    context().canEditTask(releaseId)
  }

  def checkEditVariable(release: Release, variable: Variable): Unit = {
    val permissionToCheck: Permission = if (release.isTemplate) EDIT_TEMPLATE else EDIT_RELEASE
    val pc: ReleasePermissionContext = context().asInstanceOf[ReleasePermissionContext]
    val releaseId = release.getId
    if (pc.hasPermission(permissionToCheck, releaseId)) {
      return
    }
    if (pc.hasPermission(EDIT_RELEASE_TASK_CONFIGURATION, releaseId)) {
      return
    }
    // previous semantic that was here: if user can edit ANY(exists) tasks that refer to the variable
    val canEditAllTasksThatReferToVariable = release.getAllTasks.asScala.filter(_.getReferencedVariables.contains(variable)).exists(task =>
      pc.hasPermissionToUpdateTask(task.getId)
    )
    if (canEditAllTasksThatReferToVariable) {
      return
    }
    throw PermissionDeniedException.withMessage(s"You do not have edit variable permission on variable [${variable.getId}]")
  }

  def checkEditVariable(release: Release, task: Task, variable: Variable): Unit = {
    val permissionToCheck: Permission = if (release.isTemplate) EDIT_TEMPLATE else EDIT_RELEASE
    val pc: ReleasePermissionContext = context().asInstanceOf[ReleasePermissionContext]
    val releaseId = release.getId
    if (pc.hasPermission(permissionToCheck, releaseId)) {
      return
    }
    val taskId = task.getId
    if (task.getReferencedVariables.contains(variable) && (pc.hasPermissionToUpdateTask(taskId) || pc.hasPermission(EDIT_RELEASE_TASK_CONFIGURATION, releaseId))) {
      return
    }
    throw PermissionDeniedException.withMessage(s"You do not have edit variable permission on variable [${variable.getId}]")
  }

  def checkEditTaskConfigurationFacet(releaseId: String): Unit = {
    context().checkEditTaskConfigurationFacet(releaseId: String)
  }

  def checkDeleteTasks(ids: JList[String]): Unit = {
    val releaseId: String = checkTasksFromSingleRelease(ids).orNull
    if (releaseId != null) {
      checkEdit(releaseId)
    }
  }

  private def checkTasksFromSingleRelease(ids: JList[String]): Option[String] = {
    if (!ids.isEmpty) {
      val iterator: util.Iterator[String] = ids.iterator
      val releaseId: String = releaseIdFrom(iterator.next)
      while (iterator.hasNext) {
        if (!(releaseIdFrom(iterator.next) == releaseId)) {
          throw new IllegalArgumentException("You can not work with a set of tasks from different releases or templates")
        }
      }
      return Some(releaseId)
    }
    None
  }

  def checkReassignTasks(ids: JList[String], newUser: String): Unit = {
    val releaseId: String = checkTasksFromSingleRelease(ids).orNull
    if (releaseId != null) {
      val pc = context()
      pc.checkReassignTaskPermission(releaseId)
      ids.forEach(id => pc.checkReassignTaskToUser(id, newUser))
    }
  }

  def checkReassignTaskToUser(taskId: String, newUser: String): Unit = {
    context().checkReassignTaskToUser(taskId, newUser)
  }

  def areUsersInTheSameTaskTeam(task: Task, newUser: String): Boolean = {
    context().areUsersInTheSameTaskTeam(task, newUser)
  }

  def checkEditBlackoutPermission(release: Release): Unit = {
    checkEditBlackoutPermission(release.getId())
  }

  def checkEditBlackoutPermission(releaseId: String): Unit = {
    context().checkEditBlackoutPermission(releaseId)
  }

  def checkEditFailureHandlerPermission(release: Release): Unit = {
    if (release.isTemplate) {
      check(EDIT_TEMPLATE_FAILURE_HANDLER, release)
    } else {
      checkAny(release, EDIT_RELEASE_FAILURE_HANDLER, EDIT_RELEASE_TASK)
    }
  }

  def checkEditFailureHandlerPermission(releaseId: String): Unit = {
    context().checkEditFailureHandlerPermission(releaseId)
  }

  def checkEditPreconditionPermission(release: Release): Unit = {
    if (release.isTemplate) {
      check(EDIT_TEMPLATE_PRECONDITION, release)
    } else {
      checkAny(release, EDIT_RELEASE_PRECONDITION, EDIT_RELEASE_TASK)
    }
  }

  def checkEditPreconditionPermission(releaseId: String): Unit = {
    context().checkEditPreconditionPermission(releaseId)
  }

  def checkReassignTaskPermission(releaseId: String): Unit = {
    context().checkReassignTaskPermission(releaseId)
  }

  def checkReassignTaskPermission(release: Release): Unit = {
    checkReassignTaskPermission(release.getId)
  }

  def checkReopenTaskInRelease(releaseId: String): Unit = {
    check(VIEW_RELEASE, releaseId)
    checkAny(releaseId, EDIT_RELEASE_TASK, TASK_TRANSITION)
  }

  def checkReopenTasksInRelease(taskIds: JList[String]): Unit = {
    val releaseId: String = checkTasksFromSingleRelease(taskIds).orNull
    checkReopenTaskInRelease(releaseId)
  }

  /**
   * Checks whether user can add, or edit comments on task.
   * Throws {@link PermissionDeniedException} if that is not allowed.
   */
  def checkIsAllowedToCommentOnTask(taskId: String): Unit = {
    context().checkIsAllowedToCommentOnTask(taskId)
  }

  /**
   * Checks whether user can add or delete attachments on task.
   * Throws {@link PermissionDeniedException} if that is not allowed.
   */
  def checkIsAllowedToEditAttachmentsOnTask(taskId: String): Unit = {
    context().checkIsAllowedToEditAttachmentsOnTask(taskId)
  }

  def filterAllowedToCommentOnTasks(taskIds: JList[String]): JList[String] = filterTasksSilently(taskIds, this.checkIsAllowedToCommentOnTask)

  def checkIsAllowedToStartTask(taskId: String): Unit = {
    // TODO probably we do not need any decorators nor references here - just task
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    val pc: ReleasePermissionContext = context().asInstanceOf[ReleasePermissionContext]
    pc.checkTaskIsUpdatable(taskId)
    if (task.isPostponedDueToBlackout) {
      pc.checkCanOverrideBlackout(taskId)
    } else {
      if (task.isDelayDuringBlackout && calendarService.isInBlackout(new Date)) {
        pc.checkCanOverrideBlackout(taskId)
      }
      pc.checkTaskTransitionPermission(taskId)
    }
  }

  def checkIsAllowedToWorkOnTask(taskId: String): Unit = {
    context().checkIsAllowedToWorkOnTask(taskId)
  }

  def checkTaskTransitionPermission(taskId: String): Unit = {
    context().checkTaskTransitionPermission(taskId)
  }

  def checkAdvanceTaskTransitionPermission(taskId: String): Unit = {
    context().checkAdvanceTaskTransitionPermission(taskId)
  }

  def checkRelevantTaskTransitionPermission(taskId: String): Unit = {
    context().checkRelevantTaskTransitionPermission(taskId)
  }

  def filterTasksWithTaskTransitionPermission(taskIds: JList[String]): JList[String] = {
    checkTasksFromSingleRelease(taskIds) match {
      case Some(_) =>
        val pc: ReleasePermissionContext = context().asInstanceOf[ReleasePermissionContext]
        taskIds.asScala.filter(taskId => pc.hasTaskTransitionPermission(taskId) && pc.isTaskUpdateable(taskId)).toList.asJava
      case None => new util.ArrayList[String]
    }
  }

  def isAllowedToWorkOnTask(taskId: String): Boolean = {
    context().isAllowedToWorkOnTask(taskId)
  }

  // TODO check usage
  def filterStartableReleases(releaseIds: JList[String]): JList[String] = filterSilently(releaseIds, (id: String) => check(START_RELEASE, id))

  // TODO check usage
  def filterAbortableReleases(releaseIds: JList[String]): JList[String] = filterSilently(releaseIds, (id: String) => checkAbort(id))

  // TODO check usage
  def filter(items: JList[Release], permission: Permission): JList[Release] = items.asScala.filter(hasPermission(permission, _)).toList.asJava


  private def checkPermission(permission: Permission, targetId: String): Unit = {
    if (!hasCiPermission(permission, targetId)) {
      throw PermissionDeniedException.forNodeAndPrivilege(targetId, s"'${permission.label()}'")
    }
  }

  // TODO check usage
  def checkIsAllowedToCreateReleaseFromTemplate(templateId: String, targetFolderId: String): Unit = {
    if (isNullId(targetFolderId)) {
      throw new IllegalArgumentException("targetFolderId can not be null")
    } else if (!isFolderId(targetFolderId) && !ROOT_FOLDER_ID.equals(targetFolderId)) {
      throw new IllegalArgumentException(s"targetFolderId['$targetFolderId'] is not a valid folder id")
    }

    checkAuthenticated()
    if (isCurrentUserAdmin) {
      return
    }
    checkPermission(VIEW_TEMPLATE, templateId)
    val sourceFolderId = findFolderId(templateId)
    val isRootTemplate = !isInFolder(templateId)
    val isWorkflow = Ids.isReleaseId(templateId) && releaseRepository.isWorkflow(templateId)

    if (isRootTemplate) {
      if (ROOT_FOLDER_ID.equals(targetFolderId)) {
        if (isWorkflow) {
          if (!hasPermission(START_WORKFLOW_EXECUTION, templateId)) {
            throw PermissionDeniedException.forPermission(START_WORKFLOW_EXECUTION, templateId)
          }
        } else {
          val hasCreateReleasePermission = hasGlobalPermission(CREATE_RELEASE) || hasPermission(CREATE_RELEASE_FROM_TEMPLATE, templateId)
          if (!hasCreateReleasePermission) {
            throw PermissionDeniedException.forNodeAndPrivilege(templateId, s"'${CREATE_RELEASE.label()}' (global or on the template)")
          }
        }
      } else {
        if (isWorkflow) {
          checkPermission(START_WORKFLOW_EXECUTION, targetFolderId)
        } else {
          checkPermission(CREATE_RELEASE_FROM_TEMPLATE, targetFolderId)
          checkPermission(CREATE_RELEASE_IN_ANOTHER_FOLDER, templateId)
        }
      }
    } else {
      if (isWorkflow) {
        checkPermission(START_WORKFLOW_EXECUTION, targetFolderId)
      } else {
        checkPermission(CREATE_RELEASE_FROM_TEMPLATE, targetFolderId)
        if (targetFolderId != sourceFolderId) {
          checkPermission(CREATE_RELEASE_IN_ANOTHER_FOLDER, sourceFolderId)
        }
      }
    }
  }

  /**
   * Used to check create release permission when release is created without template
   */
  def checkIsAllowedToCreateReleaseInFolder(folderId: String, releaseKind: ReleaseKind): Unit = {
    checkAuthenticated()
    if (isCurrentUserAdmin) {
      return
    }
    val isValidFolder = !Strings.isNullOrEmpty(folderId) && isFolderId(folderId) && !isRoot(folderId)
    releaseKind match {
      case ReleaseKind.RELEASE =>
        if (isValidFolder) {
          check(CREATE_RELEASE_FROM_TEMPLATE, folderId)
        } else {
          check(CREATE_RELEASE)
        }
      case ReleaseKind.WORKFLOW =>
        if (isValidFolder) {
          check(START_WORKFLOW_EXECUTION, folderId)
        } else {
          throw PermissionDeniedException.withMessage(s"You cannot start workflow on folder [$folderId]")
        }
    }
  }

  def checkIsAllowedToRegisterRunner(): Unit = {
    checkAuthenticated()
    if (isCurrentUserAdmin) {
      return
    }
    check(XLReleasePermissions.RUNNER_REGISTRATION)
  }

  def checkEditSecurity(releaseId: String): Unit = {
    if (hasGlobalPermission(EDIT_SECURITY)) {
      return
    }
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(releaseId)) EDIT_TEMPLATE_SECURITY else EDIT_RELEASE_SECURITY
    check(permissionToCheck, releaseId)
  }

  private def hasCiPermission(permission: Permission, release: Release): Boolean = {
    hasCiPermission(permission, release.getId)
  }

  private[security] def hasEffectiveSecuredCiPermission(permission: Permission, containerId: String): Boolean = {
    val securedUid = securedCis.getEffectiveSecuredCi(containerId).getSecurityUid
    permissionEnforcer.hasLoggedInUserPermission(permission, securedUid)
  }


  private def hasCiPermission(permission: Permission, containerId: String): Boolean = {
    context().hasCiPermission(permission, containerId)
  }

  def hasGlobalPermission(permission: Permission): Boolean = {
    context().hasGlobalPermission(permission)
  }

  def hasPermission(permission: Permission, ciId: String): Boolean = {
    context().hasCiPermission(permission, ciId)
  }

  def hasPermission(permission: Permission, release: Release): Boolean = hasGlobalPermission(permission) || hasCiPermission(permission, release)

  def checkHasPermissionsToUpdateTask(task: Task): Unit = {
    context().checkHasPermissionsToUpdateTask(task)
  }

  def checkViewTask(task: Task): Unit = {
    if (!hasGlobalPermission(AUDIT_ALL) && !hasViewTaskPermissions(task, Permissions.getAuthenticatedUserName, null)) {
      throw new PermissionDeniedException(s"You are not allowed to view task [${task.getId}]")
    }
  }

  def checkViewFolder(containerId: String): Unit = {
    if (containerId != ROOT_FOLDER_ID && !hasGlobalPermission(AUDIT_ALL)) {
      check(VIEW_FOLDER, containerId)
    }
  }

  /**
   * A user can see a task only if he meets at least one of those 4 conditions:
   * - he has AUDIT_ALL global permission
   * - he has VIEW_PERMISSION on the release or template the task belongs to
   * - he is the owner of the task
   * - he belongs to a team assigned to the task
   */
  private[security] def hasViewTaskPermissions(task: Task, username: String, userRoles: JList[Role]): Boolean = {
    val releaseId: String = releaseIdFrom(task.getId)
    val release = task.getRelease
    (hasGlobalPermission(AUDIT_ALL)
      || hasPermission(getViewPermission(release), releaseId)
      || task.hasOwner(username)
      || isMemberOrRoleOf(releaseId, task.getTeam, userRoles))
  }

  def isMemberOrRoleOf(releaseId: String, teamName: String): Boolean = isMemberOrRoleOf(releaseId, teamName, null)

  private def isMemberOrRoleOf(releaseId: String, teamName: String, prefetchedUserRoles: JList[Role]): Boolean = {
    if (!StringUtils.hasText(teamName)) {
      return false
    }
    val teamsToCheck: Seq[Team] = teamService.findTeamsByNames(releaseId, util.Arrays.asList(teamName, Team.RELEASE_ADMIN_TEAMNAME)).toScala(Seq)
    val currentUser: String = Permissions.getAuthenticatedUserName
    if (teamsToCheck.exists(_.hasMember(currentUser))) {
      return true
    }
    val roles: JList[Role] = if (prefetchedUserRoles == null) roleService.getRolesFor(getAuthentication) else prefetchedUserRoles
    teamsToCheck.exists(_.hasAnyRole(roles))
  }

  private def getViewPermission(release: Release): Permission = {
    if (release.isTemplate) {
      VIEW_TEMPLATE
    } else if (release.isWorkflow) {
      VIEW_WORKFLOW_EXECUTION
    } else {
      VIEW_RELEASE
    }
  }

  private def checkAuthenticated(): Unit = {
    if (isNotAuthenticated) {
      throw new AuthenticationFailureException("Authentication is missing. Did you specify 'Run automated tasks as user' property of the release?")
    }
  }

  def isNotAuthenticated: Boolean = getAuthentication == null

  def isCurrentUserAdmin: Boolean = permissionEnforcer.isCurrentUserAdmin


  private def filterTasksSilently(taskIds: JList[String], checker: Consumer[String]): JList[String] = {
    checkTasksFromSingleRelease(taskIds) match {
      case Some(_) => filterSilently(taskIds, checker)
      case None => new util.ArrayList[String]
    }
  }

  def filterSilently(ids: JList[String], checker: Consumer[String]): JList[String] = {
    def filterId(id: String): Boolean = {
      try {
        checker.accept(id)
        true
      } catch {
        case _: Exception => false
      }
    }

    ids.asScala.filter(filterId).toList.asJava
  }

  def checkCopyTask(releaseId: String): Unit = {
    val pc = context()
    pc.checkView(releaseId)
    pc.checkEdit(releaseId)
    pc.checkEditTask(releaseId)
  }

  def copyPhase(releaseId: String): Unit = {
    val pc = context()
    pc.checkView(releaseId)
    pc.checkEdit(releaseId)
  }

  def checkLockTaskPermission(releaseId: String): Unit = {
    context().checkLockTaskPermission(releaseId)
  }

  def checkViewTeams(teamContainerId: String): Unit = {
    if (!hasGlobalPermission(AUDIT_ALL)) {
      if (isReleaseId(teamContainerId)) {
        checkEdit(teamContainerId)
      } else {
        if (isFolderId(teamContainerId)) {
          checkAny(teamContainerId, VIEW_FOLDER_SECURITY, EDIT_FOLDER_SECURITY, EDIT_FOLDER_TEAMS,
            EDIT_FOLDER_NOTIFICATIONS,
            EDIT_DELIVERY_PATTERN,
            EDIT_RELEASE_DELIVERY,
            VIEW_DELIVERY_PATTERN,
            VIEW_RELEASE_DELIVERY)
        }
        else {
          throw new Checks.IncorrectArgumentException("[%s] is not a release or a folder", teamContainerId)
        }
      }
    }
  }



  def checkEditNotification(folderId: String): Unit = if (folderId == null) check(ADMIN) else check(EDIT_FOLDER_NOTIFICATIONS, folderId)

  @varargs
  def checkAny(release: Release, permissions: Permission*): Unit = {
    for (permission <- permissions) {
      if (hasPermission(permission, release)) {
        return
      }
    }
    throw exceptionForPermissions(release.getId, permissions)
  }

  private lazy val permissionContextEnv = new PermissionContextEnv {

    override def getReleaseInformation(releaseId: String): Option[ReleaseInformation] = {
      if (isReleaseId(releaseId)) {
        releaseRepository.getReleaseInformation(releaseId)
          .orElse(archivingService.getReleaseInformation(releaseId))
          .orElse(None)
      } else {
        None
      }
    }

    override def hasGlobalPermission(permission: Permission): Boolean = {
      checkAuthenticated()
      isPermissionApplicableTo(permission, GLOBAL_SECURITY_ALIAS) && permissionEnforcer.hasLoggedInUserPermission(permission)
    }

    private def hasArchivedReleasePermission(permission: Permission, release: Supplier[Release]): Boolean = {
      if (hasGlobalPermission(ADMIN)) {
        return true
      }
      val archivedRelease = release.get()
      // check if user is workflow owner for archived release
      if (isWorkflowRelatedPermission(permission) && isWorkflowExecution(archivedRelease)) {
        isWorkflowExecutionOwner(archivedRelease)
      } else {
        val auth: Authentication = getAuthentication
        val userRoles: JList[Role] = roleService.getRolesFor(auth)
        val userPrincipals: util.Collection[String] = Permissions.authenticationToPrincipals(auth)
        archivedRelease.getPermissions(userPrincipals, userRoles).contains(permission.getPermissionName)
      }
    }

    override def hasArchivedReleasePermission(permission: Permission, containerId: String): Boolean = {
      hasArchivedReleasePermission(permission, () => archivingService.getRelease(containerId))
    }

    override def isCurrentUserAdmin(): Boolean = {
      PermissionCheckerWithCache.this.isCurrentUserAdmin
    }

    override def hasEffectiveSecuredCiPermission(permission: Permission, containerId: String): Boolean = {
      PermissionCheckerWithCache.this.hasEffectiveSecuredCiPermission(permission, containerId)
    }

    override def getTaskData(taskId: String): TaskBasicData = {
      taskRepository.findTaskBasicData(taskId)
    }

    override def isMemberOrRoleOf(releaseId: String, teamName: String): Boolean = {
      PermissionCheckerWithCache.this.isMemberOrRoleOf(releaseId, teamName)
    }

    override def findTeamByName(containerId: String, teamName: String): Option[Team] = {
      import scala.jdk.OptionConverters._
      teamService.findTeamByName(containerId: String, teamName: String).toScala
    }

    override def getRolesFor(principal: String): java.util.List[Role] = {
      roleService.getRolesFor(principal)
    }

    override def isExternalUserPrincipalIsFromTheGroupAssigned(user: String, team: Team): Boolean = {
      val authorities = principalDataProvider.getAuthorities(user).asScala
      val principalsOfUser: Set[String] = authorities.map(_.getAuthority).toSet ++ Set(user)
      team.getRoles.asScala.map(role => roleService.getRoleForRoleName(role).getPrincipals.asScala).flatMap(_.toList).toSet.exists(principalsOfUser.contains)
    }

    val xlrConfig: XlrConfig = PermissionCheckerWithCache.this.xlrConfig
  }

  def context(): PermissionContext = {
    ReleasePermissionContext(permissionContextEnv)
  }
}


