package com.xebialabs.xlrelease.security

import com.google.common.base.Strings
import com.xebialabs.deployit.checks.Checks
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.Permissions.getAuthentication
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.authentication.AuthenticationFailureException
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.permission.PlatformPermissions.{ADMIN, EDIT_SECURITY}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.domain.{Release, ReleaseKind, Task, Team}
import com.xebialabs.xlrelease.principaldata.PrincipalDataProvider
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.{Ids, ReleaseRepository, SecuredCis, TaskRepository}
import com.xebialabs.xlrelease.security.PermissionChecker.exceptionForPermissions
import com.xebialabs.xlrelease.security.XLReleasePermissions._
import com.xebialabs.xlrelease.security.XlrPermissionLabels._
import com.xebialabs.xlrelease.security.{PermissionChecker => XlrPermissionChecker}
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.service.{ArchivingService, CalendarService, TeamService}
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils

import java.util
import java.util.function.{Consumer, Supplier}
import java.util.{Date, Optional, List => JList}
import scala.annotation.varargs
import scala.jdk.CollectionConverters._
import scala.jdk.StreamConverters._

//noinspection ScalaStyle
@Component("permissionCheckerWithoutCache")
@Deprecated(since = "23.3.x", forRemoval = true)
class PermissionCheckerWithoutCache(permissionEnforcer: PermissionEnforcer,
                                    releaseRepository: ReleaseRepository,
                                    taskRepository: TaskRepository,
                                    val roleService: RoleService,
                                    val teamService: TeamService,
                                    calendarService: CalendarService,
                                    securedCis: SecuredCis,
                                    archivingService: ArchivingService,
                                    xlrConfig: XlrConfig,
                                    principalDataProvider: PrincipalDataProvider)
  extends XlrPermissionChecker with PermissionContext with PermissionCheckerSharedLogic {

  def check(permission: Permission): Unit = {
    if (!hasGlobalPermission(permission)) {
      throw PermissionDeniedException.forPermission(permission, null.asInstanceOf[String])
    }
  }

  def check(permission: Permission, ciId: String): Unit = {
    // check if release is workflow and current authenticated user is the owner of the workflow or admin
    if (isWorkflowRelatedPermission(permission) && isWorkflowExecution(ciId)._1) {
      if (!isWorkflowExecutionOwner(ciId)) {
        throw PermissionDeniedException.withMessage(s"You are not the owner of the workflow execution $ciId")
      }
    } else if (!hasPermission(permission, ciId)) {
      throw PermissionDeniedException.forPermission(permission, ciId)
    }
  }

  def check(permission: Permission, release: Release): Unit = {
    // check if release is workflow and current authenticated user is the owner of the workflow or admin
    if (isWorkflowRelatedPermission(permission) && isWorkflowExecution(release)) {
      if (!isWorkflowExecutionOwner(release)) {
        throw PermissionDeniedException.withMessage(s"You are not the owner of the workflow execution ${release.getId}")
      }
    } else if (!hasPermission(permission, release)) {
      throw PermissionDeniedException.forPermission(permission, release.getId)
    }
  }

  @varargs
  def checkAny(ciId: String, permissions: Permission*): Unit = {
    for (permission <- permissions) {
      if (hasPermission(permission, ciId)) {
        return
      }
    }
    throw exceptionForPermissions(ciId, permissions)
  }

  @varargs
  def checkAny(permissions: Permission*): Unit = {
    for (permission <- permissions) {
      if (hasGlobalPermission(permission)) return
    }
    throw exceptionForPermissions(permissions: _*)
  }


  def checkView(releaseId: String): Unit = {
    if (!hasGlobalPermission(AUDIT_ALL) && !isWorkflowExecutionOwner(releaseId)) {
      check(getViewPermission(releaseId), releaseId)
    }
  }

  def checkView(release: Release): Unit = {
    if (!hasGlobalPermission(AUDIT_ALL) && !isWorkflowExecutionOwner(release)) {
      check(getViewPermission(release), release)
    }
  }

  def canViewRelease(releaseId: String): Boolean = {
    hasGlobalPermission(AUDIT_ALL) || isWorkflowExecutionOwner(releaseId) || hasCiPermission(getViewPermission(releaseId), releaseId)
  }

  private def canViewRelease(release: Release): Boolean = {
    hasGlobalPermission(AUDIT_ALL) || isWorkflowExecutionOwner(release) || hasCiPermission(getViewPermission(release), release)
  }

  private def isWorkflowExecutionOwner(release: Release): Boolean = {
    if (isWorkflowExecution(release)) {
      release.hasOwner(Permissions.getAuthenticatedUserName) || isCurrentUserAdmin
    } else {
      false
    }
  }

  private def isWorkflowExecutionOwner(ciId: String): Boolean = {
    def isCurrentUserOwner(owner: String): Boolean = {
      (owner != null && owner.equalsIgnoreCase(Permissions.getAuthenticatedUserName)) || isCurrentUserAdmin
    }

    if (Ids.isReleaseId(ciId)) {
      val (isWorkflow, isArchived) = isWorkflowExecution(ciId)
      if (isWorkflow) {
        if (isArchived) {
          isCurrentUserOwner(archivingService.getReleaseOwner(ciId))
        } else {
          isCurrentUserOwner(releaseRepository.getOwner(ciId))
        }
      } else {
        false
      }
    } else {
      false
    }
  }

  private def isWorkflowExecution(release: Release): Boolean = {
    !release.isTemplate && release.isWorkflow
  }

  private def isWorkflowExecution(ciId: String): (Boolean, Boolean) = {
    if (Ids.isReleaseId(ciId)) {
      if (!releaseRepository.exists(ciId) && archivingService.exists(ciId)) {
        (archivingService.isWorkflow(ciId), true)
      } else {
        (!releaseRepository.isTemplate(ciId) && releaseRepository.isWorkflow(ciId), false)
      }
    } else {
      (false, false)
    }
  }

  private def isWorkflowRelatedPermission(permission: Permission): Boolean = {
    Seq(EDIT_RELEASE, START_RELEASE, RESTART_PHASE).contains(permission)
  }

  def checkEditAttachment(ciId: String): Unit = {
    if (Ids.isTaskId(ciId)) {
      checkIsAllowedToEditAttachmentsOnTask(ciId)
    } else {
      checkEdit(releaseIdFrom(ciId))
    }
  }

  def checkEdit(releaseId: String): Unit = {
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(releaseId)) EDIT_TEMPLATE else EDIT_RELEASE
    check(permissionToCheck, releaseId)
  }

  def checkAbort(releaseId: String): Unit = {
    //enough to check only in live database for abort operation
    if (releaseRepository.isWorkflow(releaseId)) {
      if (!isWorkflowExecutionOwner(releaseId) && !hasPermission(ABORT_WORKFLOW_EXECUTION, releaseId)) {
        throw PermissionDeniedException.withMessage(s"You are not the owner of the workflow execution or do not have $ABORT_WORKFLOW_EXECUTION permission on $releaseId")
      }
    } else if (!hasPermission(ABORT_RELEASE, releaseId)) {
      throw PermissionDeniedException.forPermission(ABORT_RELEASE, releaseId)
    }
  }

  def checkEditTask(releaseId: String): Unit = {
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(releaseId)) EDIT_TEMPLATE else EDIT_RELEASE_TASK
    check(permissionToCheck, releaseId)
  }

  def checkEditDate(ciId: String): Unit = {
    if (isReleaseId(ciId) || Ids.isPhaseId(ciId)) {
      checkEdit(releaseIdFrom(ciId))
    } else {
      val releaseId = releaseIdFrom(ciId)
      val permissionsToCheck = if (releaseRepository.isTemplate(releaseId)) List(EDIT_TEMPLATE) else List(EDIT_RELEASE_TASK, EDIT_TASK_DATES)
      checkAny(releaseId, permissionsToCheck: _*)
    }
  }

  def canEditTask(releaseId: String): Boolean = {
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(releaseId)) EDIT_TEMPLATE else EDIT_RELEASE_TASK
    hasPermission(permissionToCheck, releaseId)
  }

  override def checkEditVariable(release: Release, variable: Variable): Unit = {
    val permissionToCheck: Permission = if (release.isTemplate) EDIT_TEMPLATE else EDIT_RELEASE
    if (hasPermission(permissionToCheck, release)) {
      return
    }
    if (canEditTaskInputOutputProperties(release)) {
      return
    }
    val canUpdate = release.getAllTasks.asScala.filter(_.getReferencedVariables.contains(variable)).exists(hasPermissionToUpdateTask(_, release))
    if (canUpdate) {
      return
    }
    throw PermissionDeniedException.withMessage(s"You do not have edit variable permission on variable [${variable.getId}]")
  }

  override def checkEditVariable(release: Release, task: Task, variable: Variable): Unit = {
    val permissionToCheck: Permission = if (release.isTemplate) EDIT_TEMPLATE else EDIT_RELEASE
    if (hasPermission(permissionToCheck, release)) {
      return
    }
    if (task.getReferencedVariables.contains(variable) && (hasPermissionToUpdateTask(task, release) || canEditTaskInputOutputProperties(release))) {
      return
    }
    throw PermissionDeniedException.withMessage(s"You do not have edit variable permission on variable [${variable.getId}]")
  }


  def checkEditTaskConfigurationFacet(releaseId: String): Unit = {
    if (!(canEditTask(releaseId) || canEditTaskConfigurationFacet(releaseId))) {
      throw PermissionDeniedException.withMessage(s"You are not allowed to create facets on tasks of release [$releaseId]")
    }
  }

  def checkDeleteTasks(ids: JList[String]): Unit = {
    val releaseId: String = checkTasksFromSingleRelease(ids).orNull
    if (releaseId != null) {
      checkEdit(releaseId)
    }
  }

  private def checkTasksFromSingleRelease(ids: JList[String]): Option[String] = {
    if (!ids.isEmpty) {
      val iterator: util.Iterator[String] = ids.iterator
      val releaseId: String = releaseIdFrom(iterator.next)
      while (iterator.hasNext) {
        if (!(releaseIdFrom(iterator.next) == releaseId)) {
          throw new IllegalArgumentException("You can not work with a set of tasks from different releases or templates")
        }
      }
      return Some(releaseId)
    }
    None
  }

  def checkReassignTasks(ids: JList[String], newUser: String): Unit = {
    val releaseId: String = checkTasksFromSingleRelease(ids).orNull
    if (releaseId != null) {
      val release: Release = releaseRepository.findById(releaseId, ResolveOptions.WITH_DECORATORS)
      checkReassignTaskPermission(release)
      ids.forEach(id => this.checkReassignTaskToUser(id, newUser))
    }
  }

  def checkReassignTaskToUser(taskId: String, newUser: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    checkReassignTaskToUser(task, newUser)
  }

  private[security] def checkReassignTaskToUser(task: Task, newUser: String): Unit = {
    try {
      checkReassignTaskPermission(task.getRelease)
    } catch {
      case e: PermissionDeniedException =>
        if (!areUsersInTheSameTaskTeam(task, newUser)) {
          throw e
        }
    }
  }

  def areUsersInTheSameTaskTeam(task: Task, newUser: String): Boolean = {
    // REL-2554 members of a task's team can always assign the task to one another
    val currentAuthentication: Authentication = Permissions.getAuthentication
    if (task.hasTeam) {
      val teamOptional: Optional[Team] = teamService.findTeamByName(task.getRelease.getId, task.getTeam)
      if (teamOptional.isPresent) {
        val team: Team = teamOptional.get
        return isUserInTeam(currentAuthentication, team) && (newUser == null || isUserInTeam(newUser, team))
      }
    }
    false
  }

  def checkEditBlackoutPermission(release: Release): Unit = {
    if (release.isTemplate) {
      check(EDIT_TEMPLATE, release)
    } else {
      checkAny(release.getId, EDIT_RELEASE_TASK, EDIT_BLACKOUT)
    }
  }

  def checkEditBlackoutPermission(releaseId: String): Unit = {
    if (releaseRepository.isTemplate(releaseId)) {
      check(EDIT_TEMPLATE, releaseId)
    } else {
      checkAny(releaseId, EDIT_RELEASE_TASK, EDIT_BLACKOUT)
    }
  }


  def checkEditFailureHandlerPermission(release: Release): Unit = {
    checkEditFailureHandlerPermission(release.getId, release.isTemplate)
  }

  def checkEditFailureHandlerPermission(releaseId: String): Unit = {
    checkEditFailureHandlerPermission(releaseId, releaseRepository.isTemplate(releaseId))
  }

  private def checkEditFailureHandlerPermission(releaseId: String, isTemplate: Boolean): Unit = {
    if (isTemplate) {
      check(EDIT_TEMPLATE_FAILURE_HANDLER, releaseId)
    } else {
      checkAny(releaseId, EDIT_RELEASE_FAILURE_HANDLER, EDIT_RELEASE_TASK)
    }
  }

  def checkEditPreconditionPermission(release: Release): Unit = {
    checkEditPreconditionPermission(release.getId, release.isTemplate)
  }

  def checkEditPreconditionPermission(releaseId: String): Unit = {
    checkEditPreconditionPermission(releaseId, releaseRepository.isTemplate(releaseId))
  }

  private def checkEditPreconditionPermission(releaseId: String, isTemplate: Boolean): Unit = {
    if (isTemplate) {
      check(EDIT_TEMPLATE_PRECONDITION, releaseId)
    } else {
      checkAny(releaseId, EDIT_RELEASE_PRECONDITION, EDIT_RELEASE_TASK)
    }
  }

  private def isUserInTeam(authentication: Authentication, team: Team): Boolean = isUserInTeam(authentication.getName, team)

  private def isUserInTeam(user: String, team: Team): Boolean =
    team.hasMember(user) || team.hasAnyRole(roleService.getRolesFor(user)) || isExternalUserPrincipalIsFromTheGroupAssigned(user, team)

  private def isExternalUserPrincipalIsFromTheGroupAssigned(user: String, team: Team): Boolean = {
    val authorities = principalDataProvider.getAuthorities(user).asScala
    val principalsOfUser: Set[String] = authorities.map(_.getAuthority).toSet ++ Set(user)
    team.getRoles.asScala.map(role => roleService.getRoleForRoleName(role).getPrincipals.asScala).flatMap(_.toList).toSet.exists(principalsOfUser.contains)
  }

  def checkReassignTaskPermission(releaseId: String): Unit = {
    checkEditOrReassignTask(releaseId, releaseRepository.isTemplate(releaseId))
  }

  def checkReassignTaskPermission(release: Release): Unit = {
    checkEditOrReassignTask(release.getId, release.isTemplate)
  }

  private def checkEditOrReassignTask(releaseId: String, isTemplate: Boolean): Unit = {
    if (isTemplate) {
      check(EDIT_TEMPLATE, releaseId)
    } else {
      checkAny(releaseId, EDIT_RELEASE_TASK, REASSIGN_RELEASE_TASK)
    }
  }

  def checkReopenTaskInRelease(releaseId: String): Unit = {
    check(VIEW_RELEASE, releaseId)
    checkAny(releaseId, EDIT_RELEASE_TASK, TASK_TRANSITION)
  }

  def checkReopenTasksInRelease(taskIds: JList[String]): Unit = {
    val releaseId: String = checkTasksFromSingleRelease(taskIds).orNull
    checkReopenTaskInRelease(releaseId)
  }

  /**
   * Checks whether user can add, or edit comments on task.
   * Throws {@link PermissionDeniedException} if that is not allowed.
   */
  def checkIsAllowedToCommentOnTask(taskId: String): Unit = {
    try {
      if (!isCurrentUserAdmin) {
        checkView(releaseIdFrom(taskId))
      }
    } catch {
      case e: NotFoundException =>
        if (archivingService.exists(taskId)) {
          throw PermissionDeniedException.withMessage(s"You cannot comment on archived task [$taskId]")
        } else {
          throw e
        }
    }
  }

  /**
   * Checks whether user can add or delete attachments on task.
   * Throws {@link PermissionDeniedException} if that is not allowed.
   */
  def checkIsAllowedToEditAttachmentsOnTask(taskId: String): Unit = {
    try {
      val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
      if (!hasPermissionToUpdateTask(task, task.getRelease) && !hasPermission(EDIT_RELEASE_TASK_ATTACHMENT, task.getRelease)) {
        throw PermissionDeniedException.forNodeAndPrivilege(task.getId, "'Task owner', 'Member of task team', 'edit task' or 'edit task attachments'")
      }
    } catch {
      case e: NotFoundException =>
        if (taskRepository.exists(taskId)) {
          throw PermissionDeniedException.withMessage(s"You cannot upload attachments on archived task [$taskId]")
        } else {
          throw e
        }
    }
  }

  def filterAllowedToCommentOnTasks(taskIds: JList[String]): JList[String] = filterTasksSilently(taskIds, this.checkIsAllowedToCommentOnTask)

  def checkIsAllowedToStartTask(taskId: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    checkTaskIsUpdatable(task)
    if (task.isPostponedDueToBlackout) {
      canOverrideBlackout(task)
    } else {
      if (task.isDelayDuringBlackout && calendarService.isInBlackout(new Date)) {
        canOverrideBlackout(task)
      }
      checkTaskTransitionPermission(taskId)
    }
  }

  private def canOverrideBlackout(task: Task): Unit = {
    if (!hasCiPermission(EDIT_BLACKOUT, task.getRelease) && !hasCiPermission(EDIT_RELEASE_TASK, task.getRelease)) {
      throw PermissionDeniedException.forNodeAndPrivilege(task.getId, "'Edit task blackout'")
    }
  }

  def checkIsAllowedToWorkOnTask(taskId: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    checkTaskIsUpdatable(task)
    checkHasPermissionsToUpdateTask(task)
  }

  def checkTaskTransitionPermission(taskId: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    checkTaskIsUpdatable(task)
    if (!hasTaskTransitionPermission(task)) {
      throw PermissionDeniedException.withMessage(s"You cannot make task transition for task=[$taskId]. You need to have permission or be either release owner or task owner")
    }
  }

  def checkAdvanceTaskTransitionPermission(taskId: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    checkTaskIsUpdatable(task)
    if (task.isPlanned && !hasAdvanceTaskTransitionPermission(task)) {
      throw PermissionDeniedException.withMessage(s"You cannot make advance task transition for task=[$taskId]. You need to have permission or be release owner")
    }
  }

  def checkRelevantTaskTransitionPermission(taskId: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    checkTaskIsUpdatable(task)
    if (task.isPlanned) {
      if (!hasAdvanceTaskTransitionPermission(task)) {
        throw PermissionDeniedException.withMessage(s"You cannot make advance task transition for task=[$taskId]. You need to have permission or be release owner")
      }
    } else {
      if (!hasTaskTransitionPermission(task)) {
        throw PermissionDeniedException.withMessage(s"You cannot make task transition for task=[$taskId]. You need to have permission or be either release owner or task owner")
      }
    }
  }

  def filterTasksWithTaskTransitionPermission(taskIds: JList[String]): JList[String] = {
    def filterTasks(id: String): Boolean = {
      val task: Task = taskRepository.findById(id, ResolveOptions.WITH_DECORATORS)
      hasTaskTransitionPermission(task) && task.isUpdatable
    }

    checkTasksFromSingleRelease(taskIds) match {
      case Some(_) => taskIds.asScala.filter(filterTasks).toList.asJava
      case None => new util.ArrayList[String]
    }
  }

  private def hasTaskTransitionPermission(task: Task): Boolean = {
    val release: Release = task.getRelease
    val allowReleaseOwnerTaskTransition = xlrConfig.isReleaseOwnerTaskTransitionAllowed
    (allowReleaseOwnerTaskTransition && release.hasOwner(Permissions.getAuthenticatedUserName)) || hasPermissionToUpdateTask(task, null) || hasCiPermission(TASK_TRANSITION, release)
  }

  private def hasAdvanceTaskTransitionPermission(task: Task): Boolean = {
    val release: Release = task.getRelease
    isCurrentUserAdmin || canEditTask(release.getId) || hasCiPermission(ADVANCE_TASK_TRANSITION, release)
  }

  def isAllowedToWorkOnTask(taskId: String): Boolean = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    task.isUpdatable && hasPermissionToUpdateTask(task, null)
  }

  def filterStartableReleases(releaseIds: JList[String]): JList[String] = filterSilently(releaseIds, (id: String) => check(START_RELEASE, id))

  def filterAbortableReleases(releaseIds: JList[String]): JList[String] = filterSilently(releaseIds, (id: String) => checkAbort(id))

  private def checkTaskIsUpdatable(task: Task): Unit = {
    if (!task.isUpdatable) {
      throw new IllegalArgumentException("You can not work with a defunct or done in advance task")
    }
  }

  def filter(items: JList[Release], permission: Permission): JList[Release] = items.asScala.filter(hasPermission(permission, _)).toList.asJava


  private def checkPermission(permission: Permission, targetId: String): Unit = {
    if (!hasCiPermission(permission, targetId)) {
      throw PermissionDeniedException.forNodeAndPrivilege(targetId, s"'${permission.label()}'")
    }
  }

  def checkIsAllowedToCreateReleaseFromTemplate(templateId: String, targetFolderId: String): Unit = {
    if (isNullId(targetFolderId)) {
      throw new IllegalArgumentException("targetFolderId can not be null")
    } else if (!isFolderId(targetFolderId) && !ROOT_FOLDER_ID.equals(targetFolderId)) {
      throw new IllegalArgumentException(s"targetFolderId['$targetFolderId'] is not a valid folder id")
    }

    checkAuthenticated()
    if (isCurrentUserAdmin) {
      return
    }
    checkPermission(VIEW_TEMPLATE, templateId)
    val sourceFolderId = findFolderId(templateId)
    val isRootTemplate = !isInFolder(templateId)
    val isWorkflow = Ids.isReleaseId(templateId) && releaseRepository.isWorkflow(templateId)

    if (isRootTemplate) {
      if (ROOT_FOLDER_ID.equals(targetFolderId)) {
        if (isWorkflow) {
          if (!hasPermission(START_WORKFLOW_EXECUTION, templateId)) {
            throw PermissionDeniedException.forPermission(START_WORKFLOW_EXECUTION, templateId)
          }
        } else {
          val hasCreateReleasePermission = hasGlobalPermission(CREATE_RELEASE) || hasPermission(CREATE_RELEASE_FROM_TEMPLATE, templateId)
          if (!hasCreateReleasePermission) {
            throw PermissionDeniedException.forNodeAndPrivilege(templateId, s"'${CREATE_RELEASE.label()}' (global or on the template)")
          }
        }
      } else {
        if (isWorkflow) {
          checkPermission(START_WORKFLOW_EXECUTION, targetFolderId)
        } else {
          checkPermission(CREATE_RELEASE_FROM_TEMPLATE, targetFolderId)
          checkPermission(CREATE_RELEASE_IN_ANOTHER_FOLDER, templateId)
        }
      }
    } else {
      if (isWorkflow) {
        checkPermission(START_WORKFLOW_EXECUTION, targetFolderId)
      } else {
        checkPermission(CREATE_RELEASE_FROM_TEMPLATE, targetFolderId)
        if (targetFolderId != sourceFolderId) {
          checkPermission(CREATE_RELEASE_IN_ANOTHER_FOLDER, sourceFolderId)
        }
      }
    }
  }

  /**
   * Used to check create release permission when release is created without template
   */
  def checkIsAllowedToCreateReleaseInFolder(folderId: String, releaseKind: ReleaseKind): Unit = {
    checkAuthenticated()
    if (isCurrentUserAdmin) {
      return
    }
    val isValidFolder = !Strings.isNullOrEmpty(folderId) && isFolderId(folderId) && !isRoot(folderId)
    releaseKind match {
      case ReleaseKind.RELEASE =>
        if (isValidFolder) {
          check(CREATE_RELEASE_FROM_TEMPLATE, folderId)
        } else {
          check(CREATE_RELEASE)
        }
      case ReleaseKind.WORKFLOW =>
        if (isValidFolder) {
          check(START_WORKFLOW_EXECUTION, folderId)
        } else {
          throw PermissionDeniedException.withMessage(s"You cannot start workflow on folder [$folderId]")
        }
    }

  }

  def checkIsAllowedToRegisterRunner(): Unit = {
    checkAuthenticated()
    if (isCurrentUserAdmin) {
      return
    }
    check(XLReleasePermissions.RUNNER_REGISTRATION)
  }

  def checkEditSecurity(releaseId: String): Unit = {
    if (hasGlobalPermission(EDIT_SECURITY)) {
      return
    }
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(releaseId)) EDIT_TEMPLATE_SECURITY else EDIT_RELEASE_SECURITY
    check(permissionToCheck, releaseId)
  }

  private def hasCiPermission(permission: Permission, release: Release): Boolean = {
    val hasRequiredPermission = if (!isPermissionApplicableTo(permission, release.getId)) {
      false
    } else if (release.isArchived) {
      hasArchivedReleasePermission(permission, () => release)
    } else if (isWorkflowRelatedPermission(permission) && isWorkflowExecution(release)) {
      isWorkflowExecutionOwner(release)
    } else {
      permissionEnforcer.hasLoggedInUserPermission(permission, securedCis.getEffectiveSecuredCi(release.getId).getSecurityUid)
    }
    hasRequiredPermission
  }

  def hasCiPermission(permission: Permission, containerId: String): Boolean = {
    val hasRequiredPermission = if (!isPermissionApplicableTo(permission, containerId)) {
      false
    } else if (isReleaseId(containerId) && !releaseRepository.exists(containerId) && archivingService.exists(containerId)) {
      hasArchivedReleasePermission(permission, () => archivingService.getRelease(containerId))
    } else if (isWorkflowRelatedPermission(permission) && isWorkflowExecution(containerId)._1) {
      isWorkflowExecutionOwner(containerId)
    } else {
      permissionEnforcer.hasLoggedInUserPermission(permission, securedCis.getEffectiveSecuredCi(containerId).getSecurityUid)
    }
    hasRequiredPermission
  }

  private def hasArchivedReleasePermission(permission: Permission, release: Supplier[Release]): Boolean = {
    if (hasGlobalPermission(ADMIN)) {
      return true
    }
    val archivedRelease = release.get()
    // check if user is workflow owner for archived release
    if (isWorkflowRelatedPermission(permission) && isWorkflowExecution(archivedRelease)) {
      isWorkflowExecutionOwner(archivedRelease)
    } else {
      val auth: Authentication = getAuthentication
      val userRoles: JList[Role] = roleService.getRolesFor(auth)
      val userPrincipals: util.Collection[String] = Permissions.authenticationToPrincipals(auth)
      archivedRelease.getPermissions(userPrincipals, userRoles).contains(permission.getPermissionName)
    }
  }

  def hasGlobalPermission(permission: Permission): Boolean = {
    checkAuthenticated()
    isPermissionApplicableTo(permission, XlrPermissionChecker.GLOBAL_SECURITY_ALIAS) && permissionEnforcer.hasLoggedInUserPermission(permission)
  }

  def hasPermission(permission: Permission, ciId: String): Boolean = hasGlobalPermission(permission) || hasCiPermission(permission, ciId)

  def hasPermission(permission: Permission, release: Release): Boolean = hasGlobalPermission(permission) || hasCiPermission(permission, release)

  def checkHasPermissionsToUpdateTask(task: Task): Unit = {
    if (!hasPermissionToUpdateTask(task, null)) {
      throw PermissionDeniedException.forNodeAndPrivilege(task.getId, "'Task owner', 'Member of task team' or 'edit task'")
    }
  }

  private def hasPermissionToUpdateTask(task: Task, release: Release): Boolean = {
    if (isCurrentUserAdmin || owns(task)) {
      return true
    }

    val resolvedRelease = if (release == null) releaseRepository.findById(releaseIdFrom(task.getId), ResolveOptions.WITH_DECORATORS) else release
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(resolvedRelease.getId)) EDIT_TEMPLATE else EDIT_RELEASE_TASK

    if (hasCiPermission(permissionToCheck, resolvedRelease)) {
      return true
    }
    if (task.hasTeam) {
      return isMemberOrRoleOf(releaseIdFrom(task.getId), task.getTeam)
    }
    false
  }

  private def canEditTaskInputOutputProperties(release: Release): Boolean = hasPermission(EDIT_RELEASE_TASK_CONFIGURATION, release)

  def canEditTaskConfigurationFacet(releaseId: String): Boolean = hasPermission(EDIT_RELEASE_TASK_CONFIGURATION_FACET, releaseId)

  def checkViewTask(task: Task): Unit = {
    if (!hasGlobalPermission(AUDIT_ALL) && !hasViewTaskPermissions(task, Permissions.getAuthenticatedUserName, null)) {
      throw new PermissionDeniedException(s"You are not allowed to view task [${task.getId}]")
    }
  }

  def checkViewFolder(containerId: String): Unit = {
    if (containerId != ROOT_FOLDER_ID && !hasGlobalPermission(AUDIT_ALL)) {
      check(VIEW_FOLDER, containerId)
    }
  }

  /**
   * A user can see a task only if he meets at least one of those 4 conditions:
   * - he has AUDIT_ALL global permission
   * - he has VIEW_PERMISSION on the release or template the task belongs to
   * - he is the owner of the task
   * - he belongs to a team assigned to the task
   */
  private[security] def hasViewTaskPermissions(task: Task, username: String, userRoles: JList[Role]): Boolean = {
    val releaseId: String = releaseIdFrom(task.getId)
    (hasGlobalPermission(AUDIT_ALL)
      || hasPermission(getViewPermission(releaseId), releaseId)
      || task.hasOwner(username)
      || isMemberOrRoleOf(releaseId, task.getTeam, userRoles))
  }

  def isMemberOrRoleOf(releaseId: String, teamName: String): Boolean = isMemberOrRoleOf(releaseId, teamName, null)

  private def isMemberOrRoleOf(releaseId: String, teamName: String, prefetchedUserRoles: JList[Role]): Boolean = {
    if (!StringUtils.hasText(teamName)) {
      return false
    }
    val teamsToCheck: Seq[Team] = teamService.findTeamsByNames(releaseId, util.Arrays.asList(teamName, Team.RELEASE_ADMIN_TEAMNAME)).toScala(Seq)
    val currentUser: String = Permissions.getAuthenticatedUserName
    if (teamsToCheck.exists(_.hasMember(currentUser))) {
      return true
    }
    val roles: JList[Role] = if (prefetchedUserRoles == null) roleService.getRolesFor(getAuthentication) else prefetchedUserRoles
    teamsToCheck.exists(_.hasAnyRole(roles))
  }

  def getViewPermission(releaseId: String): Permission = {
    if (releaseRepository.isTemplate(releaseId)) {
      VIEW_TEMPLATE
    } else if (Ids.isReleaseId(releaseId) && releaseRepository.isWorkflow(releaseId)) {
      VIEW_WORKFLOW_EXECUTION
    } else {
      VIEW_RELEASE
    }
  }

  private def getViewPermission(release: Release): Permission = {
    if (release.isTemplate) {
      VIEW_TEMPLATE
    } else if (release.isWorkflow) {
      VIEW_WORKFLOW_EXECUTION
    } else {
      VIEW_RELEASE
    }
  }

  private def owns(task: Task): Boolean = task.hasOwner(Permissions.getAuthenticatedUserName)

  private def checkAuthenticated(): Unit = {
    if (isNotAuthenticated) {
      throw new AuthenticationFailureException("Authentication is missing. Did you specify 'Run automated tasks as user' property of the release?")
    }
  }

  def isNotAuthenticated: Boolean = getAuthentication == null

  def isCurrentUserAdmin: Boolean = permissionEnforcer.isCurrentUserAdmin

  private def isPermissionApplicableTo(permission: Permission, containerId: String): Boolean = permission != null && permission.isApplicableTo(containerId)

  private def filterTasksSilently(taskIds: JList[String], checker: Consumer[String]): JList[String] = {
    checkTasksFromSingleRelease(taskIds) match {
      case Some(_) => filterSilently(taskIds, checker)
      case None => new util.ArrayList[String]
    }
  }

  def filterSilently(ids: JList[String], checker: Consumer[String]): JList[String] = {
    def filterId(id: String): Boolean = {
      try {
        checker.accept(id)
        true
      } catch {
        case _: Exception => false
      }
    }

    ids.asScala.filter(filterId).toList.asJava
  }

  def checkCopyTask(releaseId: String): Unit = {
    checkView(releaseId)
    checkEdit(releaseId)
    checkEditTask(releaseId)
  }

  def copyPhase(releaseId: String): Unit = {
    checkView(releaseId)
    checkEdit(releaseId)
  }

  def checkLockTaskPermission(releaseId: String): Unit = {
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(releaseId)) LOCK_TEMPLATE_TASK else LOCK_RELEASE_TASK
    check(permissionToCheck, releaseId)
  }

  def checkViewTeams(teamContainerId: String): Unit = {
    if (!hasGlobalPermission(AUDIT_ALL)) {
      if (isReleaseId(teamContainerId)) {
        checkEdit(teamContainerId)
      } else {
        if (isFolderId(teamContainerId)) {
          checkAny(teamContainerId, VIEW_FOLDER_SECURITY, EDIT_FOLDER_SECURITY, EDIT_FOLDER_TEAMS,
            EDIT_FOLDER_NOTIFICATIONS,
            EDIT_DELIVERY_PATTERN,
            EDIT_RELEASE_DELIVERY,
            VIEW_DELIVERY_PATTERN,
            VIEW_RELEASE_DELIVERY)
        }
        else {
          throw new Checks.IncorrectArgumentException("[%s] is not a release or a folder", teamContainerId)
        }
      }
    }
  }

  def checkEditNotification(folderId: String): Unit = if (folderId == null) check(ADMIN) else check(EDIT_FOLDER_NOTIFICATIONS, folderId)

  override def context(): PermissionContext = {
    this
  }

}
