package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.checks.Checks.{checkArgument, checkNotNull}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.ItemInUseException
import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters
import com.xebialabs.xlrelease.configuration.FeatureSettings
import com.xebialabs.xlrelease.domain.Team.{FOLDER_OWNER_TEAMNAME, RELEASE_ADMIN_TEAMNAME, TEMPLATE_OWNER_TEAMNAME}
import com.xebialabs.xlrelease.domain.distributed.events.EvictAllConfigurationCacheEvent
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.{BaseConfiguration, Release, ReleaseKind, Team}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.json.JsonUtils
import com.xebialabs.xlrelease.repository.IdMatchers.{ConfigurationId, VariableId}
import com.xebialabs.xlrelease.repository.Ids.{ROOT_FOLDER_ID, getParentId}
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.search.ReleaseSearchResult
import com.xebialabs.xlrelease.security.XLReleasePermissions.{EDIT_FOLDER_TEAMS, VIEW_APP_PIPELINES, VIEW_FOLDER, VIEW_TRIGGER}
import com.xebialabs.xlrelease.security.{PermissionChecker, XLReleasePermissions}
import com.xebialabs.xlrelease.utils.CiHelper
import com.xebialabs.xlrelease.utils.Limits.{ENABLED, MAX_FOLDER_DEPTH, TYPE_LIMITS}
import com.xebialabs.xlrelease.validation.{FolderOperationValidationError, FolderValidationErrors}
import com.xebialabs.xlrelease.views.TemplateFilters
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed

import java.util
import java.util.{List => JList}
import scala.jdk.CollectionConverters._

class FolderService(folders: FolderRepository,
                    teamService: TeamService,
                    releaseRepository: ReleaseRepository,
                    taskService: TaskService,
                    configurationRepository: ConfigurationRepository,
                    releaseVariableRepository: ReleaseVariableRepository,
                    ciIdService: CiIdService,
                    releaseSearchService: ReleaseSearchService,
                    permissions: PermissionChecker,
                    eventBus: XLReleaseEventBus,
                    archivingService: ArchivingService,
                    folderOperationValidators: Seq[_ <: FolderOperationValidator],
                    broadcastService: BroadcastService) extends Logging {

  val SLASH = "/"
  val SLASH_FOLDER = "/Folder"
  val MISSING_FOLDER = "missing folderId"

  @Timed
  def checkFolderExists(folderId: String): Unit = folders.checkFolderExists(folderId)

  @Timed
  def exists(folderId: String): Boolean = folders.exists(folderId)

  @Timed
  def getTitle(folderId: String): String = {
    if (folderId == ROOT_FOLDER_ID) {
      throw new NotFoundException(s"Folder $ROOT_FOLDER_ID cannot be found")
    } else {
      folders.getTitle(folderId).getOrElse {
        throw new NotFoundException(s"Could not find Folder $folderId")
      }
    }
  }

  @Timed
  def getPath(folderId: String): Seq[String] = {
    if (folderId == ROOT_FOLDER_ID) {
      throw new NotFoundException(s"Folder $ROOT_FOLDER_ID cannot be found")
    } else {
      folders.getPath(folderId)
    }
  }

  @Timed
  def findById(folderId: String, depth: Integer = Integer.MAX_VALUE): Folder = {
    folders.findById(folderId, depth.intValue()).getOrElse {
      throw new NotFoundException(s"Could not find Folder $folderId")
    }
  }

  @Timed
  def findViewableFoldersById(folderId: String, depth: Integer = Integer.MAX_VALUE, enforcePermission: Boolean = true): Folder = {
    folders.findViewableFoldersById(folderId, depth.intValue(), enforcePermission).getOrElse {
      throw new NotFoundException(s"Could not find Folder $folderId")
    }
  }

  @Timed
  def findByPath(path: String, depth: Int = Int.MaxValue): Folder = {
    folders.findByPath(path, depth)
  }

  @Timed
  def listViewableFolders(parentId: String,
                          page: Page,
                          decorateWithPermissions: Boolean = true,
                          enforcePermission: Boolean = true): JList[Folder] = {
    if (enforcePermission) {
      permissions.checkViewFolder(parentId)
    }
    folders.listViewableFolders(parentId, page, decorateWithPermissions, enforcePermission).asJava
  }

  @Timed
  def move(folderId: String, newParentId: String): Folder = {
    checkArgument(ROOT_FOLDER_ID != folderId, "Cannot move root folder")
    // check if new parent is the same as the folder to be moved
    if (folderId == newParentId) {
      throw new IllegalArgumentException("Cannot move folder to itself")
    }
    // check if new parent is the sub-folder of the folder to be moved
    if (newParentId.startsWith(folderId)) {
      throw new IllegalArgumentException("Cannot move folder to its sub-folder")
    }
    checkLimits(folderId, newParentId)
    checkFolderCanBeMoved(folderId, newParentId)
    eventBus.publishAndFailOnError(FolderMovingAction(folderId, getParentId(folderId), newParentId))
    val folder = folders.move(folderId, newParentId)

    // If the folder is moved to the root folder and folder doesn't have its own teams & permissions, create default teams for the folder
    // Hardcoded securedCi as FolderRow.Root has securityUid as 1 and it is not accessible in this class
    if (ROOT_FOLDER_ID == newParentId && folder.get$securedCi() == 1) {
      createDefaultTeamsForTheFolder(folder, newParentId)
    }

    eventBus.publish(FolderMovedEvent(folder, getParentId(folderId), newParentId))
    broadcastService.broadcast(EvictAllConfigurationCacheEvent(), publishEventOnSelf = true)
    folder
  }

  @Timed
  def rename(folderId: String, newName: String): Folder = {
    checkArgument(ROOT_FOLDER_ID != folderId, "Cannot rename root folder")
    checkArgument(!newName.trim.isEmpty, "Folder name cannot be blank")
    val folder = folders.rename(folderId, newName)
    eventBus.publish(FolderRenamedEvent(folder, newName))
    folder
  }

  @Timed
  def searchTemplates(folderId: String, releaseKind: ReleaseKind, page: Page, enforcePermission: Boolean = true): JList[Release] = {
    checkNotNull(folderId, MISSING_FOLDER)
    checkFolderExists(folderId)

    val filters = new TemplateFilters()
    filters.setParentId(folderId)
    filters.setKind(releaseKind)

    releaseSearchService.searchTemplates(filters, page.page, page.resultsPerPage, enforcePermission).getReleases
  }

  @Timed
  def searchReleases(folderId: String, filters: ReleasesFilters, page: Page = Page.default): ReleaseSearchResult = {
    checkNotNull(folderId, MISSING_FOLDER)
    checkFolderExists(folderId)

    filters.setParentId(folderId)

    releaseSearchService.search(filters, page.page, page.resultsPerPage)
  }

  @Timed
  def returnReleaseByFolderIdAndReleaseTitle(folderId: String, releaseTitle: String): Release = {
    checkNotNull(folderId, MISSING_FOLDER)
    checkFolderExists(folderId)

    releaseRepository.findByFolderIdAndReleaseTitle(folderId, releaseTitle)
  }

  @Timed
  def moveTemplate(folderId: String, templateId: String, shouldMergeTeams: Boolean = true): String = {
    val newTemplateId = s"$folderId/${Ids.getName(templateId)}"
    checkTemplateCanBeMoved(templateId, folderId)

    if (getParentId(templateId) == folderId) {
      logger.info(s"Attempted to move template $templateId to same folder $folderId. Will be ignored.")
      templateId
    } else {
      val templateTeams = teamService.getStoredTeams(templateId).asScala.toSeq
      eventBus.publishAndFailOnError(TemplateMovingAction(templateId, folderId))
      releaseRepository.move(templateId, newTemplateId)

      if (shouldMergeTeams) {
        mergeTeams(folderId, newTemplateId, templateTeams)
      } else {
        replaceTeams(folderId, newTemplateId)
      }

      eventBus.publish(ReleaseMovedEvent(templateId, newTemplateId))

      newTemplateId
    }
  }

  @Timed
  def create(parentId: String, folder: Folder): Folder = {
    val folderDepth = parentId.split(SLASH_FOLDER)

    val limitsSettings = getLimitsSettings
    val limitsEnabled: Boolean = limitsSettings.getProperty(ENABLED)

    if (limitsEnabled) {
      val allowedFolderDepth: Int = limitsSettings.getProperty(MAX_FOLDER_DEPTH)
      if (folderDepth.length - 1 > allowedFolderDepth) {
        throw new IllegalArgumentException(s"Creating Folder exceeds allowed sub-folder depth $allowedFolderDepth")
      }
    }
    create(parentId, folder, createDefaultTeams = true)
  }

  @Timed
  def create(parentId: String, folder: Folder, createDefaultTeams: Boolean): Folder = {
    val result = createWithoutPublishing(parentId, folder, createDefaultTeams)
    result.events.foreach(event => eventBus.publish(event))
    result.folder
  }

  private def checkLimits(folderId: String, newParentId: String): Unit = {
    val limitsSettings = getLimitsSettings
    val limitsEnabled: Boolean = getLimitsSettings.getProperty(ENABLED)

    if (limitsEnabled) {
      checkFolderDepth(folderId, newParentId, limitsSettings)
    }
  }

  private def checkFolderDepth(folderId: String, newParentId: String, limitsSettings: FeatureSettings): Unit = {
    val allowedFolderDepth: Int = limitsSettings.getProperty(MAX_FOLDER_DEPTH)
    val newParentIdFolderDepth = newParentId.split(SLASH_FOLDER).length - 1

    if (newParentIdFolderDepth > allowedFolderDepth) {
      throw new IllegalArgumentException(s"Moving folder/folders is not within the allowed sub-folder depth $allowedFolderDepth")
    }

    val newFolderIdSplit = folderId.split(SLASH)
    val newFolderIdSplitLength = newFolderIdSplit.length
    val newFolderIdInDb = newFolderIdSplit(newFolderIdSplitLength - 1)

    val findChildrenOfNewFolder = folders.findNumberOfChildrenForAFolder(newFolderIdInDb)
    val newToBeFolderLength = newParentIdFolderDepth + findChildrenOfNewFolder

    if (newToBeFolderLength > allowedFolderDepth) {
      throw new IllegalArgumentException(s"Moving folder/folders is not within the allowed sub-folder depth $allowedFolderDepth")
    }
  }

  private def getLimitsSettings: FeatureSettings = {
    configurationRepository.findFirstByType(Type.valueOf(TYPE_LIMITS)).orElse({
      val feature = Type.valueOf(TYPE_LIMITS).getDescriptor.newInstance[FeatureSettings]("")
      feature.generateId()
      feature
    })
  }

  case class FolderAndEvents(folder: Folder, events: Seq[XLReleaseEvent] = Seq.empty)

  def createWithoutPublishing(parentId: String, folder: Folder, createDefaultTeams: Boolean): FolderAndEvents = {
    checkArgument(!folder.getTitle.trim.isEmpty, "Folder name cannot be blank")
    checkArgument(folder.getTitle.length < 256, "Folder name must be 255 characters or less")
    checkFolderExists(parentId)
    folders.checkNameIsUnique(parentId, folder.getTitle)

    if (Ids.isNullId(folder.getId) || !Ids.isFolderId(folder.getId)) {
      folder.setId(ciIdService.getUniqueId(Type.valueOf(classOf[Folder]), parentId))
    } else {
      folder.setId(s"$parentId/${Ids.getName(folder.getId)}")
    }

    logger.info(s"Creating folder ${folder.getTitle} with id ${folder.getId}")

    val saved = folders.create(getParentId(folder.getId), folder)
    var events: Seq[XLReleaseEvent] = Seq(FolderCreatedEvent(folder))

    if (createDefaultTeams && parentId == ROOT_FOLDER_ID) {
      events = events ++ createDefaultTeamsForTheFolder(saved, parentId)
    } else {
      events = events :+ TeamsUpdatedEvent(folder.getId)
    }

    FolderAndEvents(saved, events)
  }

  @Timed
  def delete(folderId: String): Unit = {
    checkArgument(ROOT_FOLDER_ID != folderId, "Cannot delete root folder")
    checkFolderExists(folderId)
    checkFolderCanBeDeleted(folderId)
    eventBus.publishAndFailOnError(FolderDeletingAction(folderId))
    logger.info(s"Deleting folder $folderId")
    val folder = folders.findById(folderId).get
    folders.delete(folderId, archiveOrDelete)
    eventBus.publish(FolderDeletedEvent(folder))
  }

  private def archiveOrDelete(folderUid: Int, releaseId: String): Unit = {
    val status = releaseRepository.getStatus(releaseId)
    logger.trace(s"archiveOrDelete($folderUid, $releaseId): status=$status [inactive? ${status.isInactive}]")
    if (status.isInactive) {
      val release = releaseRepository.findById(releaseId)
      if (archivingService.shouldArchive(release)) {
        archivingService.preArchiveRelease(release)
      }
      logger.trace(s"archiveRelease($releaseId)")
      archivingService.archiveRelease(releaseId)
    } else {
      logger.trace(s"deleteRelease($releaseId)")
      releaseRepository.delete(releaseId)
    }
  }

  private def mergeTeams(folderId: String, templateId: String, templateTeams: Seq[Team]): Unit = {
    if (templateTeams.nonEmpty) {
      val folderTeams = teamService.getEffectiveTeams(folderId).asScala.toSeq
      val folderTeamsContainerId = folderTeams.headOption.map(team => getParentId(team.getId)).getOrElse(folderId)
      val mergedTeams = mergeTemplateAndFolderTeams(folderTeams, templateTeams)

      val folderTitle = getTitle(folderId)

      teamService.saveTeamsToPlatform(folderTeamsContainerId, mergedTeams.asJava)
      teamService.deleteTeamsFromPlatform(templateId)

      eventBus.publish(TeamsMergedEvent(templateId, folderTitle))
    }
  }

  private def replaceTeams(folderId: String, templateId: String): Unit = {
    resetTaskTeam(templateId)
    val folderTitle = getTitle(folderId)
    eventBus.publish(TeamsRemovedInTemplateEvent(templateId, folderTitle))

    teamService.deleteTeamsFromPlatform(templateId)
  }

  private def mergeTemplateAndFolderTeams(folderTeams: Seq[Team], templateTeams: Seq[Team]): Seq[Team] = {
    val folderTeamsMap = collection.mutable.Map(folderTeams.map { team => (team.getTeamName, team) }.toMap.toSeq: _*)
    //loop through the templates and either add the new teams to the folder or update the folder
    templateTeams.foreach(templateTeam => {
      folderTeamsMap.get(templateTeam.getTeamName) match {
        case None =>
          templateTeam.setId(null)
          folderTeamsMap += templateTeam.getTeamName -> templateTeam

        case Some(folderTeam) =>
          val memberSet = folderTeam.getMembers.asScala.toSet ++ templateTeam.getMembers.asScala.toSet
          val roleSet = folderTeam.getRoles.asScala.toSet ++ templateTeam.getRoles.asScala.toSet
          val permissionSet = folderTeam.getPermissions.asScala.toSet ++ templateTeam.getPermissions.asScala.toSet
          folderTeam.setMembers(memberSet.toSeq.asJava)
          folderTeam.setRoles(roleSet.toSeq.asJava)
          folderTeam.setPermissions(permissionSet.toSeq.asJava)
      }
    })

    folderTeamsMap.values.toSeq
  }

  private def resetTaskTeam(templateId: String): Unit = {
    val template: Release = releaseRepository.findById(templateId)
    if (!template.isWorkflow) {
      template.getAllTasks.forEach(task => {
        taskService.applyNewTeam(null, task, false)
      })
    }
  }

  private def createDefaultTeamsForTheFolder(folder: Folder, parentId: String): List[XLReleaseEvent] = {
    val currentUser: String = getAuthenticatedUserName

    def newTeam(): Team = Type.valueOf(classOf[Team]).getDescriptor.newInstance(null)

    val folderOwner = newTeam()
    folderOwner.setTeamName(FOLDER_OWNER_TEAMNAME)
    folderOwner.setPermissions(XLReleasePermissions.getFolderPermissions.asScala.filterNot(_ == EDIT_FOLDER_TEAMS.getPermissionName).asJava)
    folderOwner.getPermissions.addAll(XLReleasePermissions.getReleaseGroupPermissions)
    folderOwner.getPermissions.addAll(XLReleasePermissions.getDeliveryPermissions)
    folderOwner.getPermissions.addAll(XLReleasePermissions.getDashboardPermissions)
    folderOwner.getPermissions.addAll(XLReleasePermissions.getAppPipelinesPermissions)

    val templateOwner = newTeam()
    templateOwner.setTeamName(TEMPLATE_OWNER_TEAMNAME)
    templateOwner.setPermissions(XLReleasePermissions.getTemplateOnlyPermissions)
    templateOwner.getPermissions.add(VIEW_FOLDER.getPermissionName)
    templateOwner.getPermissions.add(VIEW_TRIGGER.getPermissionName)
    templateOwner.getPermissions.add(VIEW_APP_PIPELINES.getPermissionName)

    val releaseAdmin = newTeam()
    val releaseAdminPermissions = XLReleasePermissions.getReleasePermissions.asScala
      .concat(XLReleasePermissions.getTriggerPermissions.asScala)
      .concat(XLReleasePermissions.getWorkflowExecutionPermissions.asScala)
      .concat(Seq(VIEW_FOLDER.getPermissionName, VIEW_APP_PIPELINES.getPermissionName))

    releaseAdmin.setTeamName(RELEASE_ADMIN_TEAMNAME)
    releaseAdmin.addPermissions(releaseAdminPermissions.toArray)

    if (currentUser != null) {
      folderOwner.addMember(currentUser)
      templateOwner.addMember(currentUser)
      releaseAdmin.addMember(currentUser)
    }

    val result = teamService.saveTeamsToPlatformWithoutPublishing(folder.getId, Seq(folderOwner, templateOwner, releaseAdmin).asJava, true)
    result._2.asScala.toList
  }

  private def checkFolderCanBeMoved(folderId: String, newfolderIdOption: String): Unit = {
    val validationMessages = new util.ArrayList[FolderOperationValidationError]()
    folderOperationValidators.foreach(validator => validator.validateMoveOperation(folderId, newfolderIdOption, validationMessages))
    if (validationMessages.asScala.nonEmpty) {
      val errors = FolderValidationErrors("You cannot move the folder", validationMessages.asScala.toList)
      val json = JsonUtils.objectMapper.writeValueAsString(errors)
      throw new ItemInUseException(json)
    }
  }

  def getConfiguration(configurationId: String): BaseConfiguration = {
    configurationId match {
      case ConfigurationId(id) => configurationRepository.read(id)
      case VariableId(id) => releaseVariableRepository.findById(id)
    }
  }

  private def checkFolderCanBeDeleted(folderId: String): Unit = {
    val validationMessages = new util.ArrayList[FolderOperationValidationError]()
    folderOperationValidators.foreach(validator => validator.validateDeleteOperation(folderId, validationMessages))
    if (validationMessages.asScala.nonEmpty) {
      val categoryToMessagesMap = validationMessages.asScala.groupBy(_.category)
      val errorMessages = categoryToMessagesMap.map { case (category, messages) =>
        val message = messages.map(_.title).mkString("\"", "\", \"", "\"")
        if (messages.size == 1) {
          s"${category.toString.toLowerCase}: $message"
        } else {
          s"${category.toString.toLowerCase}s: $message"
        }
      }.mkString(", ")

      throw new ItemInUseException(s"You cannot delete this folder. The folder or its subfolders contain $errorMessages.")
    }
  }

  private def checkTemplateCanBeMoved(templateId: String, folderId: String): Unit = {
    val template = releaseRepository.findById(templateId)
    val refs = CiHelper.getExternalReferences(template).asScala

    if (!refs.filter(c => c.isInstanceOf[BaseConfiguration])
      .map(c => getConfiguration(c.getId))
      .forall(c => c.getFolderId == null || folderId.startsWith(c.getFolderId))) {
      throw new ItemInUseException("You cannot move this template. It contains references to configurations that are " +
        "not global or not inherited by the destination folder.")
    }
  }
}
