package com.xebialabs.xlrelease.spring.configuration;

import java.io.IOException;
import java.util.HashSet;
import java.util.Set;
import org.slf4j.Logger;
import org.springframework.context.ApplicationContextInitializer;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.convert.TypeDescriptor;
import org.springframework.core.convert.converter.ConditionalGenericConverter;
import org.springframework.core.convert.support.DefaultConversionService;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.PropertySource;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.support.DefaultPropertySourceFactory;
import org.springframework.core.io.support.EncodedResource;
import com.fasterxml.jackson.core.StreamReadConstraints;

import com.xebialabs.deployit.spring.HoconConfigPropertySource;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.config.XlrConfig;

import static com.xebialabs.xlrelease.spring.configuration.XlrProfiles.CLUSTER;
import static com.xebialabs.xlrelease.spring.configuration.XlrProfiles.DEFAULT;
import static com.xebialabs.xlrelease.spring.configuration.XlrProfiles.DEFAULT_AUTH;
import static com.xebialabs.xlrelease.spring.configuration.XlrProfiles.OIDC_AUTH;
import static org.slf4j.LoggerFactory.getLogger;

public class XlrWebApplicationInitializer implements ApplicationContextInitializer<ConfigurableApplicationContext> {

    private static final Logger LOGGER = getLogger(XlrWebApplicationInitializer.class);

    @Override
    public void initialize(final ConfigurableApplicationContext applicationContext) {
        final ConfigurableEnvironment environment = applicationContext.getEnvironment();
        addProfiles(environment);
        addConverters(environment);
        addPropertySources(environment);
        addDecryptingPropertySource(environment);
        setJacksonXlrDefaults();
    }

    private void addPropertySources(final ConfigurableEnvironment environment) {
        try {
            //hocon property sources
            var propertySource = new HoconConfigPropertySource(XlrConfig.defaultConfigName(), XlrConfig.getInstance().getRootConfig());
            environment.getPropertySources().addLast(propertySource);

            //for old customers using xl-release-server.conf instead of xl-release.conf
            DefaultPropertySourceFactory defaultPropertySourceFactory = new DefaultPropertySourceFactory();
            final PropertySource defaultPS = defaultPropertySourceFactory.createPropertySource("xl-release-server.conf", new EncodedResource(new ClassPathResource("xl-release-server.conf")));
            environment.getPropertySources().addLast(defaultPS);

        } catch (IOException e) {
            LOGGER.error("Some properties might not be loaded correctly, see following stacktrace", e);
        }
    }

    private void addProfiles(final ConfigurableEnvironment environment) {
        XlrConfig bootConfig = XlrConfig.getInstance();

        if (bootConfig.isClusterEnabled()) {
            LOGGER.info(String.format("Running in cluster mode: %s", bootConfig.clusterMode().configOption()));
            environment.addActiveProfile(CLUSTER);
            environment.addActiveProfile(bootConfig.clusterMode().configOption());
        } else {
            LOGGER.info("Running in standalone mode");
            environment.addActiveProfile(DEFAULT);
        }

        if (bootConfig.isOidcEnabled()) {
            LOGGER.info("Authentication provider: OIDC");
            environment.addActiveProfile(OIDC_AUTH);
        } else {
            LOGGER.info("Authentication provider: default");
            environment.addActiveProfile(DEFAULT_AUTH);
        }
    }

    private void addConverters(final ConfigurableEnvironment environment) {
        final DefaultConversionService conversionService = new DefaultConversionService();
        conversionService.addConverter(new PasswordConverter());
        environment.setConversionService(conversionService);
    }

    private void addDecryptingPropertySource(ConfigurableEnvironment environment) {
        environment.getPropertySources().stream()
                .filter(propertySource -> !propertySource.getClass().getSimpleName().contains("ConfigurationPropertySourcesPropertySource"))
                .forEach(propertySource -> {
                    environment.getPropertySources().replace(
                            propertySource.getName(),
                            new PropertySourceDecrypter(propertySource)
                    );
                });
    }

    public void setJacksonXlrDefaults() {
        int maxStringLength = XlrConfig.getInstance().maxStringLength();
        StreamReadConstraints constraints = StreamReadConstraints.builder().maxStringLength(maxStringLength).build();
        StreamReadConstraints.overrideDefaultStreamReadConstraints(constraints);
    }

    private class PasswordConverter implements ConditionalGenericConverter {

        private final Set<ConvertiblePair> convertiblePairs;

        PasswordConverter() {
            this.convertiblePairs = new HashSet<>();
        }

        @Override
        public boolean matches(final TypeDescriptor sourceType, final TypeDescriptor targetType) {
            final TypeDescriptor stringDescriptor = TypeDescriptor.valueOf(String.class);
            return sourceType == stringDescriptor && targetType == stringDescriptor;
        }

        @Override
        public Set<ConvertiblePair> getConvertibleTypes() {
            convertiblePairs.add(new ConvertiblePair(String.class, String.class));
            return convertiblePairs;
        }

        @Override
        public Object convert(final Object source, final TypeDescriptor sourceType, final TypeDescriptor targetType) {
            if (source == null) return null;
            if (PasswordEncrypter.getInstance().isEncodedAndDecryptable(source.toString())) {
                return PasswordEncrypter.getInstance().decrypt(source.toString());
            }
            return source;
        }
    }
}
