package com.xebialabs.xlrelease.utils

import com.xebialabs.xlplatform.utils.ClassLoaderUtils.classLoader
import com.xebialabs.xlrelease.Environment
import com.xebialabs.xlrelease.Environment._
import org.springframework.core.io.{ClassPathResource, FileSystemResource, Resource}
import org.springframework.security.util.InMemoryResource

import java.io.{File, IOException}
import scala.jdk.CollectionConverters.EnumerationHasAsScala
import scala.util.Try

object ResourceUtils {

  val WEB_ROOT = "web/"

  /**
   * Returns a handle to a File (in development mode) or a Classpath resource.
   */
  @throws(classOf[IOException])
  def resource(path: String): Resource = resource(path, "")

  /**
   * Returns a handle to a File (in development mode) or a Classpath web resource,
   * which means that it has to be under 'web/' folder on classpath.
   */
  @throws(classOf[IOException])
  def webResource(path: String): Resource = resource(path, WEB_ROOT)

  @throws(classOf[IOException])
  private def resource(path: String, pathPrefix: String): Resource =
    if (Environment.isDevelopment) {
      // We don't prefix paths with 'web/' in development mode by default to avoid moving huge set of
      // files to new path and get problems with cherry-picking to maintenance branches
      var file: Resource = fileSystemResource(path)
      if (file.exists()) file else {
        file = fileSystemResource(pathPrefix + path)
        if (file.exists()) file else new ClassPathResource(pathPrefix + path)
      }
    } else {
      new ClassPathResource(pathPrefix + path)
    }

  private def fileSystemResource(path: String): Resource =
    (Seq(WEB_PATH_BUILD, WEB_PATH_SRC, SERVER_RESOURCES_PATH, DEV_PLUGINS_PATH) ++ BUNDLED_PLUGIN_PATHS)
      .map(new File(_, path))
      .find(_.exists())
      .map(new FileSystemResource(_))
      .getOrElse(RESOURCE_NOT_FOUND)

  private val RESOURCE_NOT_FOUND: InMemoryResource = new InMemoryResource("") {
    override def exists: Boolean = false
  }

  def findFirstInClassPath(resource: String): Option[File] = {
    def fileExists(maybeFile: Option[File]): Boolean = {
      maybeFile.exists(file => file.exists())
    }

    classLoader.getResources(resource).asScala.map(url => Try(new File(url.toURI)).toOption).find(fileExists).flatten
  }

}
