package com.xebialabs.xlrelease.analytics.service

import com.xebialabs.xlrelease.analytics.service.AnalyticsService.{ANALYTICS_ID, ANALYTICS_METRIC_REPORT}
import com.xebialabs.xlrelease.api.utils.WordUtils.convertToTitleCase
import com.xebialabs.xlrelease.domain.metadata.MetadataEntry
import com.xebialabs.xlrelease.json.JsonUtils.objectMapper
import com.xebialabs.xlrelease.repository.MetadataRepository
import grizzled.slf4j.Logging
import org.apache.commons.codec.digest.DigestUtils
import org.apache.commons.lang3.StringUtils.stripAccents
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.time.Instant
import java.time.format.DateTimeFormatter.ISO_INSTANT
import java.util
import java.util.Optional
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

object AnalyticsService {
  val ANALYTICS_ID = "analytics_id"
  val ANALYTICS_METRIC_REPORT = "analytics_metric_report"
}

@Service
class AnalyticsService @Autowired()(metadataRepository: MetadataRepository) extends Logging {

  def generateAccountId(licensedName: String, serverUrl: String): String = {
    val accountName = convertToTitleCase(stripAccents(Option(licensedName).getOrElse(""))
      .replaceAll("[^a-zA-Z0-9]", " ").replaceAll("(\\d)([A-Za-z])", "$1 $2"))
    val accountEnvironment = DigestUtils.md5Hex(Option(serverUrl).getOrElse("")).take(6)
    s"${accountName}_${accountEnvironment}"
  }

  def getAccountId(licensedName: String, serverUrl: String): String = {
    Try {
      metadataRepository.getEntry(ANALYTICS_ID) match {
        case Some(entry) => entry.value
        case None =>
          val accountId = generateAccountId(licensedName, serverUrl)
          metadataRepository.create(ANALYTICS_ID, accountId)
          accountId
      }
    } match {
      case Success(accountId) => accountId
      case Failure(e) =>
        logger.warn("Exception while creating Analytics Identifier", e)
        ""
    }
  }

  def updateAccountId(licensedName: String, serverUrl: String): String = {
    Try {
      val accountId = generateAccountId(licensedName, serverUrl)
      if (metadataRepository.exists(ANALYTICS_ID)) {
        logger.debug(s"Updating Analytics Identifier")
        metadataRepository.update(ANALYTICS_ID, accountId)
      } else {
        logger.debug(s"Creating Analytics Identifier")
        metadataRepository.create(ANALYTICS_ID, accountId)
      }
      accountId
    } match {
      case Success(accountId) => accountId
      case Failure(e) =>
        logger.warn("Exception while updating Analytics Identifier", e)
        ""
    }
  }


  def getAnalyticsReport(): Optional[util.Map[String, Object]] = {
    Try {
      metadataRepository.getEntry(ANALYTICS_METRIC_REPORT).map { reportEntry =>
        logger.debug(s"Including analytics metric report ${reportEntry.value} into account properties")
        val report = readReportFromJsonEntry(reportEntry)
        logger.debug(s"Deleting  analytics metric report ${reportEntry.value} from metadata repository")
        metadataRepository.delete(ANALYTICS_METRIC_REPORT)
        report
      }
    } match {
      case Success(report) => Optional.ofNullable(report.orNull)
      case Failure(e) =>
        logger.warn("Exception while updating Analytics Identifier", e)
        Optional.empty()
    }
  }

  private def readReportFromJsonEntry(entry: MetadataEntry) = {
    val report = objectMapper.readValue(Option(entry.content).getOrElse("{}"), classOf[Map[String, Object]])
    report.map {
      case (key, value: List[_]) => key -> new util.ArrayList[Any](value.asJava)
      case other => other
    }.asJava
  }

  def generateAnalyticsReport(systemInfo: Map[String, Any]): Unit = {
    Try {
      val systemInfoMap = objectMapper.convertValue(systemInfo, classOf[Map[String, Any]])
      val analyticsReport = transformReport(systemInfoMap)

      if (analyticsReport.nonEmpty) {
        val analyticsReportEntry = MetadataEntry(
          s"${ANALYTICS_METRIC_REPORT}_${ISO_INSTANT.format(Instant.now())}",
          objectMapper.writeValueAsString(analyticsReport)
        )
        logger.debug(s"Generated analytics metrics report ${analyticsReportEntry.value}: ${analyticsReportEntry.content}")
        if (metadataRepository.exists(ANALYTICS_METRIC_REPORT)) {
          logger.debug(s"Updating analytics metrics report")
          metadataRepository.update(ANALYTICS_METRIC_REPORT, analyticsReportEntry)
        } else {
          logger.debug(s"Creating analytics metrics report")
          metadataRepository.create(ANALYTICS_METRIC_REPORT, analyticsReportEntry)
        }
      }
    } match {
      case Success(_) => logger.debug("Analytics metrics report generated successfully")
      case Failure(e) => logger.warn(s"Unable to generate analytics metrics report, error: ${e.getMessage}", e)
    }
  }

  private def transformReport(map: Map[String, Any]): Map[String, Any] = map.flatMap {
    case (key, nestedMap: Map[String, Any]) => transformReport(nestedMap)
    case (key, value) => Map(transformKey(key) -> value)
  }

  private def transformKey(key: String): String = s"Release_${key.capitalize}"
}