package com.xebialabs.xlrelease.script;

import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.xlrelease.script.ScriptLifeCycle.ScriptState.DONE;
import static com.xebialabs.xlrelease.script.ScriptLifeCycle.ScriptState.REGISTERED;
import static com.xebialabs.xlrelease.script.ScriptLifeCycle.ScriptState.STARTED;
import static java.lang.Thread.currentThread;

/**
 * <p>
 * A Life cycle finite state machine for script execution.
 * </p>
 * <p>
 * Its role is to maintain execution states of scripts and interrupt them.
 * </p>
 * A script can be in one the following states :
 * <ul>
 * <li>UNREGISTERED : when the script is not registered, it can't be interrupted. It is the initial state</li>
 * <li>REGISTERED : when its execution ID has been created, but the script is not started yet. If interruption is asked, an InterruptedException will be thrown when trying to enter STARTED state</li>
 * <li>STARTED : when the script has started. If interruption is asked, the thread will be interrupted if possible</li>
 * <li>DONE : when the script has terminated, it can't be interrupted any more.</li>
 * </ul>
 * <pre>
 * {@literal
 *     Possible transitions :
 *
 *                            start()
 *           REGISTERED ---------------- STARTED
 *             ^ |                          |
 *             | | unregister()             | end()
 *  register() | |                          |
 *             | v                          v
 *          UNREGISTERED ---------------- DONE
 *                          unregister()
 * }
 * </pre>
 */
@Component
public class ScriptLifeCycle {
    private static final Logger logger = LoggerFactory.getLogger(ScriptLifeCycle.class);
    private Map<String, State> executionStates = newHashMap();

    public synchronized void tryAborting(String executionId) {
        if (executionStates.containsKey(executionId)) {
            executionStates.get(executionId).tryInterrupting();
        }
    }

    public synchronized void register(String executionId) {
        logger.trace("REGISTER {}", executionId);
        checkState(executionId != null, "executionId cannot be null");
        checkState(!executionStates.containsKey(executionId), "Expected executionId: '%s' to be unregistered", executionId);
        executionStates.put(executionId, new State());
    }

    public synchronized void start(String executionId) throws InterruptedException {
        logger.trace("START {}", executionId);
        checkState(executionId != null, "executionId cannot be null");
        checkState(executionStates.containsKey(executionId), "Expected executionId: '%s' to be registered", executionId);
        executionStates.get(executionId).begin(executionId);
    }

    public synchronized void reset(String executionId) throws InterruptedException {
        logger.trace("RESET {}", executionId);
        checkState(executionId != null, "executionId cannot be null");
        checkState(executionStates.containsKey(executionId), "Expected executionId: '%s' to be registered", executionId);
        ScriptState scriptState = executionStates.get(executionId).scriptState;
        checkState(scriptState == REGISTERED || scriptState == DONE, "Expected executionId: '%s' to be registered or done", executionId);
        executionStates.get(executionId).reset(executionId);
    }

    public synchronized void end(String executionId) {
        logger.trace("END {}", executionId);
        checkState(executionId != null, "executionId cannot be null");
        checkState(executionStates.containsKey(executionId), "Expected executionId: '%s' to be registered", executionId);
        executionStates.get(executionId).end(executionId);
    }

    public synchronized void unregister(String executionId) {
        logger.trace("UNREGISTER {}", executionId);
        checkState(executionId != null, "executionId cannot be null");
        checkState(executionStates.containsKey(executionId), "Expected executionId: '%s' to be registered", executionId);
        ScriptState scriptState = executionStates.get(executionId).scriptState;
        checkState(scriptState == REGISTERED || scriptState == DONE, "Expected executionId: '%s' to be registered or done", executionId);

        executionStates.remove(executionId);
    }

    enum ScriptState {
        REGISTERED, STARTED, DONE
    }

    static class State {
        private ScriptState scriptState = REGISTERED;
        private boolean interruptionRequested = false;
        private Thread thread = null;

        public void tryInterrupting() {
            this.interruptionRequested = true;

            if (scriptState == STARTED) {
                thread.interrupt();
            }
        }

        public void begin(String executionId) throws InterruptedException {
            checkState(scriptState == REGISTERED, "Expected executionId: '%s' to be registered", executionId);

            if (interruptionRequested) {
                interruptionRequested = false;
                throw new InterruptedException();
            }

            scriptState = STARTED;
            thread = currentThread();
        }

        /**
         * This happens when there is a need to run multiple scripts at once
         * For example during facet checks
         */
        public void reset(String executionId) throws InterruptedException {
            checkState(scriptState == DONE, "Expected executionId: '%s' to be done", executionId);

            if (interruptionRequested) {
                interruptionRequested = false;
                throw new InterruptedException();
            }

            scriptState = REGISTERED;
        }

        public void end(String executionId) {
            checkState(scriptState == STARTED, "Expected executionId: '%s' to be started", executionId);

            scriptState = DONE;

            // Reset interrupted state
            Thread.interrupted();
        }
    }
}
