package com.xebialabs.xlrelease.utils

import com.fasterxml.jackson.databind.{JsonNode, ObjectMapper}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlplatform.utils.ResourceManagement.using
import com.xebialabs.xlrelease.db.ArchivedReleases
import com.xebialabs.xlrelease.db.ArchivedReleases.{REPORT_RELEASES_END_DATE_COLUMN, REPORT_RELEASES_ID_COLUMN, REPORT_RELEASES_LOGSJSON_COLUMN, REPORT_RELEASES_RELEASEJSON_COLUMN, REPORT_RELEASES_TABLE_NAME}
import com.xebialabs.xlrelease.repository.Ids.getName
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils.deleteDirectory
import org.apache.commons.io.IOUtils
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.io.{File, FileInputStream, FileOutputStream}
import java.nio.file.Paths
import java.sql.ResultSet
import java.text.SimpleDateFormat
import java.util.Date
import java.util.zip.{ZipEntry, ZipOutputStream}
import scala.jdk.CollectionConverters._

class ArchivedReleaseExporter(val reportingJdbcTemplate: JdbcTemplate,
                              val rootDir: File,
                              val archivedReleases: ArchivedReleases) extends Logging {
  private val objMapper = new ObjectMapper()
  val formatter = new SimpleDateFormat("yyyy/MM/dd")

  def exportReleaseData(releaseId: String): String = {
    val releaseData = getReleaseData(releaseId)
    val dateFragment = formatter.format(releaseData.endDate)
    val releaseName = getName(releaseId)

    // Directory the zip file will be written to
    val exportDir = Paths.get(rootDir.getPath, s"$dateFragment").toFile

    // Directory to write all files that will eventually be zipped
    val tempDir = Paths.get(exportDir.getPath, s"$releaseName").toFile
    tempDir.mkdirs()

    val releaseFile = writeJson(releaseData.releaseJson, tempDir, "release")
    val activityLogFile = writeJson(releaseData.activityLogJson, tempDir, "activity-log")

    val attachments = writeAttachments(releaseId, tempDir)

    val exportZip = createZip(Seq(releaseFile, activityLogFile) ++ attachments, exportDir, releaseName)

    deleteDirectory(tempDir)
    exportZip.getPath
  }

  private val STMT_GET_JSON: String =
    s"""|SELECT
        | $REPORT_RELEASES_RELEASEJSON_COLUMN,
        | $REPORT_RELEASES_LOGSJSON_COLUMN,
        | $REPORT_RELEASES_END_DATE_COLUMN
        |FROM $REPORT_RELEASES_TABLE_NAME
        |WHERE  $REPORT_RELEASES_ID_COLUMN = ?""".stripMargin

  private case class ReleaseData(releaseJson: JsonNode, activityLogJson: JsonNode, endDate: Date)

  private def getReleaseData(releaseId: String): ReleaseData = {
    reportingJdbcTemplate.query(STMT_GET_JSON, jsonMapper, releaseId).asScala.headOption.getOrElse(
      throw new NotFoundException(s"Could not find archived release [$releaseId]")
    )
  }

  private val jsonMapper: RowMapper[ReleaseData] = (rs: ResultSet, _: Int) => {
    val releaseJsonBlob = rs.getBlob(REPORT_RELEASES_RELEASEJSON_COLUMN)
    val activityLogsBlob = rs.getBlob(REPORT_RELEASES_LOGSJSON_COLUMN)
    ReleaseData(
      if (releaseJsonBlob != null) objMapper.readTree(releaseJsonBlob.getBinaryStream) else null,
      if (activityLogsBlob != null) objMapper.readTree(activityLogsBlob.getBinaryStream) else null,
      rs.getDate(REPORT_RELEASES_END_DATE_COLUMN)
    )
  }

  private def writeJson(json: JsonNode, exportDir: File, filename: String): File = {
    val jsonFile = exportDir.toPath.resolve(s"$filename.json").toFile

    using(new FileOutputStream(jsonFile)) { file =>
      objMapper.writerWithDefaultPrettyPrinter().writeValue(file, json)
    }
    jsonFile
  }

  private def writeAttachments(releaseId: String, exportDir: File): Seq[File] = {
    archivedReleases.getAttachmentsFileNames(releaseId).map(kv => writeAttachment(exportDir, kv._1, kv._2)).toSeq.flatten
  }

  private def writeAttachment(exportDir: File, attachmentId: String, filename: String): Option[File] = {
    archivedReleases.getAttachment(attachmentId).map(a => {
      val attachmentDir = Paths.get(exportDir.getPath, s"attachments/${getName(attachmentId)}").toFile
      attachmentDir.mkdirs()

      val attachmentFile = attachmentDir.toPath.resolve(filename).toFile
      using(new FileOutputStream(attachmentFile)) {
        file => file.write(a.getInputStream().readAllBytes())
      }
      attachmentFile
    })
  }

  private def createZip(files: Seq[File], exportDir: File, zipFilename: String): File = {
    def filename(file: File) = {
      file.getPath.stripPrefix(exportDir.getPath).stripPrefix(File.separator)
    }

    val zipFile = Paths.get(exportDir.getPath, s"$zipFilename.zip").toFile
    using(new ZipOutputStream(new FileOutputStream(zipFile))) { zipStream =>
        files.foreach(file => {
          zipStream.putNextEntry(new ZipEntry(filename(file)))
          using(new FileInputStream(file)) { content =>
            IOUtils.copy(content, zipStream)
          }
          zipStream.closeEntry()
        })
    }
    zipFile
  }
}
