package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.config.{ArchivingSettingsManager, XlrConfig}
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.events.{ReleaseAbortedEvent, ReleaseCompletedEvent}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.metrics.MetricsConfiguration.TAG_EXECUTOR
import com.xebialabs.xlrelease.repository.ReleaseRepository
import com.xebialabs.xlrelease.utils.PrefixedThreadFactory
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import io.micrometer.core.instrument.MeterRegistry
import io.micrometer.core.instrument.binder.jvm.ExecutorServiceMetrics
import org.springframework.stereotype.Service

import java.util.concurrent.{Executors, ScheduledExecutorService, TimeUnit}
import scala.annotation.tailrec
import scala.concurrent.duration._

@Service
@EventListener
class PreArchiveService(xlrConfig: XlrConfig,
                        archivingService: ArchivingService,
                        releaseRepository: ReleaseRepository,
                        archivingConfig: ArchivingSettingsManager,
                        meterRegistry: MeterRegistry) extends Logging {

  private lazy val maxThreadCount: Int = xlrConfig.executors.preArchivingExecutor.maxThreadsCount

  private var pool: ScheduledExecutorService = createPool

  private var enabled: Boolean = false

  private var catchingUp: Boolean = false

  def isEnabled: Boolean = enabled

  def start(): Unit = {
    logger.info("Starting PreArchiveService")

    logger.trace("Setting preArchivingEnabled callback...")
    archivingConfig.subscribeToPreArchivingEnabledChanges { becomesEnabled =>
      this.synchronized {
        if (becomesEnabled) {
          logger.trace("disabled -> enabled")
          enabled = true
          preArchiveInactiveReleasesInBackground()
        } else {
          logger.trace("enabled -> disabled")
          enabled = false
        }
      }
    }

    enabled = archivingConfig.getPreArchivingEnabled
    if (enabled) {
      preArchiveInactiveReleases()
    }
    logger.debug("PreArchiveService started.")
  }


  @Subscribe
  def preArchiveCompletedRelease(event: ReleaseCompletedEvent): Unit = preArchive(event.release)

  @Subscribe
  def preArchiveAbortedRelease(event: ReleaseAbortedEvent): Unit = preArchive(event.release)

  @Timed
  protected def preArchive(release: Release, checkEnabled: Boolean = true, recoverIfExists: Boolean = true): Unit = {
    if (!checkEnabled || (enabled && !catchingUp)) {
      submit { () =>
        try {
          logger.info(s"Pre-archiving $release")
          archivingService.preArchiveRelease(release)
          logger.debug(s"Pre-archived $release")
        } catch {
          case e: Exception =>
            logger.warn("Something went wrong while pre-archiving", e)
            if (!recoverIfExists) {
              throw e
            }
            // Remove Data from archive db if exist and try again
            if (archivingService.exists(release.getId)) {
              logger.debug(s"Removing $release from Archive Database")
              if (archivingService.deletePreArchiveRelease(release.getId)) {
                logger.debug(s"Pre-archiving again $release")
                preArchive(release, checkEnabled, recoverIfExists = false)
              } else {
                logger.debug(s"Removing failed $release from Archive Database")
              }
            } else {
              logger.debug(s"Release $release does not exist in Archive Database", e)
              throw e
            }
        }
      }
    } else {
      logger.debug(s"Pre-archiving is disabled")
    }
  }

  protected def preArchiveInactiveReleasesInBackground(): Unit = synchronized {
    if (!catchingUp) {
      catchingUp = true
      submit(() => {
        preArchiveInactiveReleases()
        catchingUp = false
      })
    }
  }

  @tailrec
  private def preArchiveInactiveReleases(page: Int = 0): Unit = {
    val pageSize = archivingConfig.getSearchPageSize
    logger.debug(s"Fetching page $page ($pageSize releases) for pre-archiving...")
    val found = releaseRepository.findPreArchivableReleases(page, pageSize)
    found.filter(archivingService.shouldArchive).foreach(preArchive(_, checkEnabled = false))

    if (enabled && found.size == pageSize) {
      preArchiveInactiveReleases(page + 1)
    }
  }

  private def submitWithDelay(delay: FiniteDuration)(job: () => Unit): Unit = pool.schedule(runnable(job), delay.toSeconds, TimeUnit.SECONDS)

  private def submit[U](job: () => U): Unit = pool.submit(runnable(job))

  private def runnable[U](body: () => U): Runnable = () => body()


  def disable(): AnyVal = {
    try {
      pool.shutdown()
      pool.awaitTermination(10, TimeUnit.SECONDS)
    } catch {
      case e: Exception => logger.error("Unable to disable pre-archive service")
    }
  }

  def enable(): Unit = {
    if (pool.isShutdown || pool.isTerminated) {
      pool = createPool
    }
  }

  private def createPool: ScheduledExecutorService = {
    val threadPool: ScheduledExecutorService = Executors.newScheduledThreadPool(maxThreadCount, new PrefixedThreadFactory("pre-archiving", true))
    if (xlrConfig.metrics.enabled) {
      ExecutorServiceMetrics.monitor(meterRegistry, threadPool, "pre-archiving", TAG_EXECUTOR)
    } else {
      threadPool
    }
  }
}
