package com.xebialabs.deployit.plumbing;

import java.io.IOException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import jakarta.servlet.*;
import jakarta.servlet.http.HttpServletResponse;

import org.eclipse.jetty.http.BadMessageException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.xlrelease.XLReleaseBootstrapper;

import static java.lang.String.format;
import static jakarta.servlet.http.HttpServletResponse.SC_GONE;
import static jakarta.ws.rs.core.MediaType.TEXT_PLAIN;

public class VersionEnforcerFilter implements Filter {
    private static final String VERSION_PARAMETER = "xlrv";

    private static final Pattern VERSION_PATTERN = Pattern.compile("^[0-9]+\\.[0-9]+(\\.[0-9]+)?(-(alpha|beta|rc|SNAPSHOT|[0-9]+)(\\.[0-9]+)?)?$");
    private static final Logger logger = LoggerFactory.getLogger(VersionEnforcerFilter.class);

    private String serverVersion;

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        serverVersion = CurrentVersion.get();
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        boolean doFilter = true;
        try {
            String[] values = request.getParameterValues(VERSION_PARAMETER);
            if (values != null) {
                String clientVersion = values[0];

                if (!isClientVersionValid(clientVersion)) {
                    response.setContentType(TEXT_PLAIN);
                    response.getWriter().write(format("%s : provided version format is incorrect", XLReleaseBootstrapper.PRODUCT_DIGITALAI()));
                    ((HttpServletResponse) response).setStatus(SC_GONE);
                    doFilter = false;
                }

                if (!clientVersion.equals(serverVersion) && doFilter) {
                    // Version has changed
                    response.setContentType(TEXT_PLAIN);
                    response.getWriter().write(format("%s version changed : you are using %s, but server version is now %s", XLReleaseBootstrapper.PRODUCT_DIGITALAI(), clientVersion, serverVersion));
                    ((HttpServletResponse) response).setStatus(SC_GONE);
                    doFilter = false;
                }
            }
        } catch (BadMessageException jettyException) {
            // ignore jetty error here and let it fail elsewhere in the chain
            logger.debug("Jetty was unable to handle request. Will pass request to another filter in the chain.", jettyException);
        }

        if (doFilter) {
            chain.doFilter(request, response);
        }
    }

    @Override
    public void destroy() {
    }

    private boolean isClientVersionValid(final String clientVersion) {
        Matcher versionMatcher = VERSION_PATTERN.matcher(clientVersion);
        return versionMatcher.matches();
    }
}
