package com.xebialabs.xlrelease.service;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import jakarta.annotation.Nullable;

import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plumbing.serialization.ResolutionContext;
import com.xebialabs.xlrelease.api.internal.InternalMetadataDecoratorService;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.events.*;
import com.xebialabs.xlrelease.domain.folder.Folder;
import com.xebialabs.xlrelease.domain.status.TaskStatus;
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective;
import com.xebialabs.xlrelease.domain.tasks.TaskUpdater;
import com.xebialabs.xlrelease.domain.variables.ListOfStringValueProviderConfiguration;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.repository.TaskRepository;
import com.xebialabs.xlrelease.repository.query.TaskBasicData;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.user.User;
import com.xebialabs.xlrelease.utils.CiHelper;
import com.xebialabs.xlrelease.utils.MismatchedVariableHolder;
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils;
import com.xebialabs.xlrelease.views.MovementIndexes;

import io.micrometer.core.annotation.Timed;

import static com.google.common.base.Objects.equal;
import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.base.Strings.isNullOrEmpty;
import static com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.xlrelease.api.internal.ReleaseGlobalAndFolderVariablesDecorator.GLOBAL_AND_FOLDER_VARIABLES;
import static com.xebialabs.xlrelease.domain.CustomScriptTask.UNKNOWN_TYPE;
import static com.xebialabs.xlrelease.domain.PythonScript.PYTHON_SCRIPT_ID;
import static com.xebialabs.xlrelease.domain.Task.UNKNOWN_TASK_TYPE;
import static com.xebialabs.xlrelease.domain.TaskWithPropertiesDefinition.ofKind;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.COMPLETED;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.COMPLETED_IN_ADVANCE;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.PENDING;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.PLANNED;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.SKIPPED;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.SKIPPED_IN_ADVANCE;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.WAITING_FOR_INPUT;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_RELEASE_TASK;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TEMPLATE_TASK;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_VERIFY_CONCURRENT_MODIFICATION;
import static com.xebialabs.xlrelease.events.XLReleaseOperations.publishEvents;
import static com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi;
import static com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID;
import static com.xebialabs.xlrelease.repository.Ids.findFolderId;
import static com.xebialabs.xlrelease.repository.Ids.isPhaseId;
import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;
import static com.xebialabs.xlrelease.user.User.AUTHENTICATED_USER;
import static com.xebialabs.xlrelease.utils.CiHelper.rewriteWithNewId;
import static com.xebialabs.xlrelease.variable.VariablePersistenceHelper.fixUpVariableIds;
import static com.xebialabs.xlrelease.variable.VariablePersistenceHelper.scanAndBuildNewVariables;
import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang.StringUtils.equalsIgnoreCase;

@Service
public class TaskService {
    public static final String TASK_COPY_MESSAGE_ITEM_FORMAT_1 = "* %s\n";
    public static final String TASK_COPY_MESSAGE_ITEM_FORMAT_2 = "* %s (%s) \n";
    public static final String TASK_COPY_MESSAGE_DELIMITER = "**---**\n";

    private TaskRepository taskRepository;

    private ExecutionService executionService;

    private ReleaseService releaseService;

    private CommentService commentService;

    private PermissionChecker permissions;

    private TaskAccessService taskAccessService;

    private TaskTypeConversion taskTypeConversionService;

    private TeamService teamService;

    private InternalMetadataDecoratorService decoratorService;

    private CiIdService ciIdService;

    private ArchivingService archivingService;

    private XLReleaseEventBus eventBus;

    private PhaseService phaseService;

    private ConfigurationRepository configurationRepository;

    private Map<Class<? extends Task>, TaskUpdater> taskUpdatersPerType = new HashMap<>();

    private final XlrConfig xlrConfig;

    private FolderService folderService;

    private VariableService variableService;

    private ServerConfiguration serverConfiguration;

    private SharedConfigurationService sharedConfigurationService;

    private List<TaskCopyHandler> taskCopyHandlers;

    @Autowired
    public TaskService(TaskRepository taskRepository,
                       ExecutionService executionService,
                       ReleaseService releaseService,
                       CommentService commentService,
                       PermissionChecker permissions,
                       TaskAccessService taskAccessService,
                       TaskTypeConversion taskTypeConversionService,
                       TeamService teamService,
                       InternalMetadataDecoratorService decoratorService,
                       CiIdService ciIdService,
                       ArchivingService archivingService,
                       XLReleaseEventBus eventBus,
                       PhaseService phaseService,
                       ConfigurationRepository configurationRepository,
                       XlrConfig xlrConfig,
                       FolderService folderService,
                       VariableService variableService,
                       ServerConfiguration serverConfiguration,
                       SharedConfigurationService sharedConfigurationService,
                       List<TaskCopyHandler> taskCopyHandlers) {
        this.taskRepository = taskRepository;
        this.executionService = executionService;
        this.releaseService = releaseService;
        this.commentService = commentService;
        this.permissions = permissions;
        this.taskAccessService = taskAccessService;
        this.taskTypeConversionService = taskTypeConversionService;
        this.teamService = teamService;
        this.decoratorService = decoratorService;
        this.ciIdService = ciIdService;
        this.archivingService = archivingService;
        this.eventBus = eventBus;
        this.phaseService = phaseService;
        this.configurationRepository = configurationRepository;
        this.xlrConfig = xlrConfig;
        this.folderService = folderService;
        this.variableService = variableService;
        this.serverConfiguration = serverConfiguration;
        this.sharedConfigurationService = sharedConfigurationService;
        this.taskCopyHandlers = taskCopyHandlers;
    }

    @Autowired
    public void setTaskUpdaters(final List<? extends TaskUpdater> taskUpdaters) {
        taskUpdaters.forEach(updater -> {
            Class<? extends Task> taskType = updater.getTaskClass();
            if (taskUpdatersPerType.containsKey(taskType)) {
                throw new IllegalStateException(format("Two updaters are registered for the same task type %s: %s and" +
                                " %s",
                        taskType, taskUpdatersPerType.get(taskType), updater));
            }
            taskUpdatersPerType.put(taskType, updater);
        });
    }

    @Timed
    public Task getTaskWithoutDecoration(String taskId) {
        return taskRepository.findById(taskId);
    }

    @Timed
    public <T extends Task> T findById(String taskId) {
        return this.findById(taskId, ResolveOptions.WITH_DECORATORS());
    }

    @Timed
    public <T extends Task> T findById(String taskId, ResolveOptions resolveOptions) {
        T task = taskRepository.findById(taskId, resolveOptions);
        teamService.decorateWithEffectiveTeams(task.getRelease());
        decoratorService.decorate(task.getRelease(), singletonList(GLOBAL_AND_FOLDER_VARIABLES()));
        return task;
    }

    @Timed
    public List<Task> findById(List<String> taskIds) {
        return taskIds.stream().map(this::<Task>findById).collect(toList());
    }

    @Timed
    public List<Task> findByIdIncludingArchived(List<String> taskIds) {
        return taskIds.stream().map(this::findByIdIncludingArchived).collect(toList());
    }

    @Timed
    public List<TaskBasicData> findTasksForPolling(List<String> taskIds) {
        if (taskIds.isEmpty()) {
            return emptyList();
        }
        return taskRepository.findTasksBasicData(taskIds);
    }

    @Timed
    public Task findByIdIncludingArchived(String taskId) {
        return this.findByIdIncludingArchived(taskId, ResolveOptions.WITH_DECORATORS());
    }

    @Timed
    public Task findByIdIncludingArchived(String taskId, ResolveOptions resolveOptions) {
        if (exists(taskId)) {
            return findById(taskId, resolveOptions);
        } else if (archivingService.exists(taskId)) {
            Task task = archivingService.getTask(taskId);
            decoratorService.decorate(task.getRelease(), singletonList(GLOBAL_AND_FOLDER_VARIABLES()));
            return task;
        } else {
            throw new LogFriendlyNotFoundException(format("Task [%s] does not exist in the repository or archive",
                    taskId));
        }
    }

    @Timed
    public String getUniqueId(String parent) {
        return ciIdService.getUniqueId(Type.valueOf(Task.class), parent);
    }

    @Timed
    public boolean exists(String taskId) {
        return taskRepository.exists(taskId);
    }

    @Timed
    public Task create(String containerId, Task task) {
        return create(containerId, task, null);
    }

    @Timed
    public Task create(String containerId, Task task, Integer position) {
        TaskContainer container = findContainerById(containerId);
        return createTask(container, null, task, position, true);
    }

    @Timed
    public Task create(TaskContainer taskContainer, @Nullable String generatedTaskId, Task task, Integer position) {
        return createTask(taskContainer, generatedTaskId, task, position, false);
    }

    private void checkAreTaskPropertiesUpdatable(Task oldTask, Task newTask) {
        var pc = permissions.context();
        if (newTask.ownerHasBeenReassigned(oldTask)) {
            pc.checkReassignTaskToUser(oldTask.getId(), newTask.getOwner());
        }

        if (newTask.teamHasBeenReassigned(oldTask)) {
            pc.checkReassignTaskPermission(oldTask.getRelease().getId());
        }

        if (newTask.delayDuringBlackoutHasChanged(oldTask)) {
            pc.checkEditBlackoutPermission(oldTask.getRelease().getId());
        }

        if (newTask.failureHandlerHasChanged(oldTask)) {
            pc.checkEditFailureHandlerPermission(oldTask.getRelease().getId());
        }

        if (newTask.preconditionHasChanged(oldTask)) {
            pc.checkEditPreconditionPermission(oldTask.getRelease().getId());
        }

        checkTaskIsUpdatable(oldTask);

        checkArgument(isScheduledStartDateUpdatable(oldTask, newTask),
                "Can't update scheduled start date of task '%s' that is neither planned or pending",
                oldTask.getTitle());
    }

    public Task updateTaskWith(String taskId, Task updated) {
        return updateTaskWith(taskId, updated,
                Collections.unmodifiableSet(new HashSet<>(Arrays.asList(UPDATE_RELEASE_TASK, UPDATE_TEMPLATE_TASK))),
                false);
    }

    @Timed
    public Task updateTaskWith(String taskId, Task updated, Set<TaskUpdateDirective> updateDirectives,
                               boolean overrideLock) {
        checkArgument(updated.getFlagStatus() != null, "Flag status is required.");
        checkArgument(isNotBlank(updated.getTitle()), "Task title is required.");
        taskAccessService.checkIfAuthenticatedUserCanUseTask(updated);

        Task task = findById(taskId);
        if (task instanceof CustomScriptTask && ((CustomScriptTask) task).getPythonScript() != null) {
            checkArgument(!((CustomScriptTask) task).isUnknown(), format("You can not update '%s' task", UNKNOWN_TYPE));
        } else if (task instanceof ContainerTask) {
            checkArgument(!((ContainerTask) task).isUnknown(), format("You can not update '%s' task",
                    UNKNOWN_TASK_TYPE));
        }
        checkAreTaskPropertiesUpdatable(task, updated);

        TaskUpdater taskUpdater = getTaskUpdater(task.getClass());
        checkArgument(null != taskUpdater, "Cannot update task because there is no updater defined for task type " +
                "'%s'", task.getType().toString());

        checkIfConfigurationIsInherited(task.getRelease(), CiHelper.getExternalReferences(updated));

        Task original = cloneCi(task);
        Changes changes = taskUpdater.update(task, updated, updateDirectives);

        scanAndBuildNewVariables(task.getRelease(), task, ciIdService);

        Release release = task.getRelease();
        release.updateRealFlagStatus();

        if (!overrideLock) {
            LockedTaskOperationChecks.checkTaskUpdate(original, task);
        }

        taskRepository.updateTaskAndReleaseFlagStatus(task, release);
        eventBus.publish(new TaskUpdatedEvent(original, task));
        publishEvents(changes.getOperations(), eventBus);
        return findById(original.getId());
    }

    public Task updateTaskVariables(String taskId, List<Variable> variables, DateTime modifiedAt,
                                    Set<TaskUpdateDirective> updateDirectives) {
        Task task = getTaskWithoutDecoration(taskId);
        checkArgument(task.getTaskType().instanceOf(Type.valueOf(UserInputTask.class)), "Can only add update " +
                "variables to UserInputTask, not a " + task.getType().toString());
        taskAccessService.checkIfAuthenticatedUserCanUseTask(task);
        checkTaskIsUpdatable(task);
        Release release = task.getRelease();

        for (Variable variable : variables) {
            checkArgument(((UserInputTask) task).getVariables().contains(variable), String.format("The variable [%s] " +
                    "does not belong to this task", variable.getKey()));
            permissions.checkEditVariable(release, task, variable);
        }

        UserInputTask original = cloneCi((UserInputTask) task);
        UserInputTask updated = cloneCi((UserInputTask) task);

        if (updateDirectives.contains(UPDATE_VERIFY_CONCURRENT_MODIFICATION)) {
            updated.set$ciAttributes(new CiAttributes(task.get$ciAttributes().getCreatedBy(),
                    task.get$ciAttributes().getCreatedAt(), null, modifiedAt,
                    task.get$ciAttributes().getScmTraceabilityDataId()));
        }

        for (Variable variable : variables) {
            updated.removeVariable(variable.getId());
            updated.getVariables().add(variable);
        }

        TaskUpdater taskUpdater = getTaskUpdater(task.getClass());
        checkArgument(null != taskUpdater, "Cannot update task because there is no updater defined for task type " +
                "'%s'", task.getType().toString());

        Changes changes = taskUpdater.update(task, updated, updateDirectives);

        scanAndBuildNewVariables(task.getRelease(), task, ciIdService);
        LockedTaskOperationChecks.checkTaskUpdate(original, task);

        taskRepository.updateTaskAndReleaseFlagStatus(task, release);
        eventBus.publish(new TaskUpdatedEvent(original, task));
        publishEvents(changes.getOperations(), eventBus);
        return task;
    }

    @Timed
    public void setStatusLine(String taskId, String statusLine) {
        eventBus.publish(new TaskStatusLineUpdated(taskId, statusLine));
        taskRepository.updateTaskStatusLine(taskId, statusLine);
    }

    @Timed
    public Task changeTaskType(String taskId, Type newTaskType) {
        Task task = taskRepository.findById(taskId);
        LockedTaskOperationChecks.checkTaskTypeChange(task);
        taskAccessService.checkIfAuthenticatedUserCanUseTask(task);
        return taskTypeConversionService.changeActiveTaskType(task, newTaskType);
    }

    @Timed
    public Task completeTask(String taskId, String commentText) {
        Task task = findById(taskId);
        Release release = task.getRelease();

        if (task.isPlanned()) {
            checkIsStatusCompletableInAdvance(task);
            executionService.markTaskAsDone(release, COMPLETED_IN_ADVANCE, taskId, commentText, AUTHENTICATED_USER);
        } else {
            checkIsStatusUpdatable(task, "complete");
            executionService.markTaskAsDone(release, COMPLETED, taskId, commentText, AUTHENTICATED_USER);
        }

        return findById(taskId);
    }

    @Timed
    public List<String> completeTasks(List<String> taskIds, String commentText) {
        return executeBulkActionSilently(taskIds, (taskId) -> completeTask(taskId, commentText));
    }

    @Timed
    public Task skipTask(String taskId, String commentText, User user) {
        Task task = findById(taskId);
        LockedTaskOperationChecks.checkSkipTask(task);

        Release release = task.getRelease();
        if (task.isPlanned()) {
            checkIsTaskSkippableInAdvance(task);
            executionService.markTaskAsDone(release, SKIPPED_IN_ADVANCE, taskId, commentText, user);
        } else {
            checkIsTaskSkippable(findById(taskId));
            checkArgument(!isNullOrEmpty(commentText), "Comment is mandatory when skipping a task.");

            executionService.markTaskAsDone(release, SKIPPED, taskId, commentText, user);
        }

        return findById(taskId);
    }

    @Timed
    public List<String> skipTasks(List<String> taskIds, String commentText, User user) {
        // Lists.reverse() due to REL-5789. XLR will auto start next task when the current is completed/skipped.
        return executeBulkActionSilently(Lists.reverse(taskIds), (taskId) -> skipTask(taskId, commentText, user));
    }

    @Timed
    public Task failTask(String taskId, String commentText) {
        checkArgument(!isNullOrEmpty(commentText), "Comment is mandatory when failing a task.");
        Task task = findById(taskId);
        checkArgument(!isScriptOrCustomScriptTask(task), "Script tasks can't be failed, they can be aborted");
        checkIsStatusUpdatable(task, "fail");

        executionService.fail(task, commentText, AUTHENTICATED_USER);

        return task;
    }

    @Timed
    public List<String> failTasks(List<String> taskIds, String commentText) {
        return executeBulkActionSilently(taskIds, (taskId) -> failTask(taskId, commentText));
    }

    @Timed
    public Task abortTask(String taskId, String commentText) {
        checkArgument(!isNullOrEmpty(commentText), "Comment is mandatory when aborting a task.");
        Task task = findById(taskId);
        checkCanTaskBeAborted(task);

        executionService.abortTask(task, commentText);

        return findById(taskId);
    }

    @Timed
    public List<String> abortTasks(List<String> taskIds, String commentText) {
        return executeBulkActionSilently(taskIds, (taskId) -> abortTask(taskId, commentText));
    }

    @Timed
    public List<String> reopenTasks(List<String> taskIds, String commentText) {
        return executeBulkActionSilently(taskIds, (taskId) -> reopenTask(taskId, commentText));
    }

    @Timed
    public Task reopenTask(String taskId, String commentText) {
        checkArgument(!isNullOrEmpty(commentText), "Comment is mandatory when starting a pending task.");
        Task task = findById(taskId);
        checkCanTaskBeReopen(task);

        executionService.reopenTask(task, commentText);

        return findById(taskId);
    }

    @Timed
    public Task retryTask(String taskId, String commentText) {
        checkArgument(!isNullOrEmpty(commentText), "Comment is mandatory when retrying a task.");
        Task task = findById(taskId);
        checkCanTaskBeRetried(task);

        executionService.retry(task, commentText);

        return task;
    }

    @Timed
    public List<String> retryTasks(List<String> taskIds, String commentText) {
        return executeBulkActionSilently(taskIds, (taskId) -> retryTask(taskId, commentText));
    }

    @Timed
    public Task startPendingTask(String taskId, String commentText) {
        checkArgument(!isNullOrEmpty(commentText), "Comment is mandatory when starting a pending task.");
        Task task = findById(taskId);
        executionService.startPendingTask(taskId, task.getRelease(), commentText, AUTHENTICATED_USER);

        return findById(taskId);
    }

    @Timed
    public Task startWithInput(String taskId, List<Variable> variables) {
        return executionService.startWithInput(taskId, variables);
    }

    @Timed
    public List<Comment> getCommentsOfTask(String taskId) {
        if (exists(taskId)) {
            return commentService.findByTask(taskId);
        } else if (archivingService.exists(taskId)) {
            return archivingService.getTask(taskId).getComments();
        } else {
            throw new LogFriendlyNotFoundException(format("Task [%s] does not exist in the repository or archive",
                    taskId));
        }
    }

    @Timed
    public Comment addComment(String taskId, String commentText) {
        checkArgument(!isNullOrEmpty(commentText), "A commentText is required.");
        return commentService.create(taskId, commentText, AUTHENTICATED_USER, true);
    }

    @Timed
    public List<String> addComments(List<String> taskIds, String commentText) {
        return new ArrayList<>(executeBulkActionSilently(taskIds, taskId -> addComment(taskId, commentText)));
    }

    @Timed
    public void applyNewTeam(final String newTeam, final Task task) {
        applyNewTeam(newTeam, task, true);
    }

    @Timed
    public void applyNewTeam(final String newTeam, final Task task, final boolean doTaskLockOperationCheck) {
        checkArgument(!task.getRelease().isWorkflow(), "Workflow does not support Teams functionality.");
        if (doTaskLockOperationCheck && xlrConfig.isLockTaskAssignmentDisabled()) {
            LockedTaskOperationChecks.checkUpdateTeams(task);
        }
        String previousTeam = task.getTeam();
        if (!Objects.equal(newTeam, previousTeam)) {
            checkTaskIsUpdatable(task);
            Task original = cloneCi(task.getRelease()).getTask(task.getId());
            task.setTeam(newTeam);

            scanAndBuildNewVariables(task.getRelease(), task, ciIdService);

            taskRepository.update(task);
            eventBus.publish(new TaskUpdatedEvent(original, task));
        }
    }

    @Timed
    public Task reassignToOwner(String taskId, String newOwner) {
        Task task = taskRepository.findById(taskId);
        checkArgument(!task.getRelease().isWorkflow(), "Workflow task owner changes are only allowed through Workflow" +
                " owner modifications.");
        if (xlrConfig.isLockTaskAssignmentDisabled()) {
            LockedTaskOperationChecks.checkUpdateOwner(task, newOwner, permissions);
        }
        applyNewOwner(newOwner, task);
        return task;
    }

    @Timed
    public List<String> reassignTasks(List<String> taskIds, String newTeam, String newOwner) {
        return executeBulkActionSilently(taskIds, (taskId) -> {
            Task task = taskRepository.findById(taskId);
            checkArgument(!task.getRelease().isWorkflow(), "Workflow does not support a task owner or team " +
                    "reassignment.");
            applyNewTeam(newTeam, task);
            reassignToOwner(taskId, newOwner);
            return null;
        });
    }

    @Timed
    public List<String> deleteTasks(List<String> taskIds) {
        return executeBulkActionSilently(taskIds, (taskId) -> {
            delete(taskId);
            return null;
        });
    }

    @Timed
    public void delete(String taskId) {
        Task task = findById(taskId);
        checkArgument(task.isPlanned(), "Only planned tasks can be deleted. Task '%s' is %s.", task.getTitle(),
                task.getStatus());
        LockedTaskOperationChecks.checkDeleteTask(task);

        Release release = task.getRelease();
        release.deleteTask(task);
        release.updateRealFlagStatus();

        clearLinks(task.getContainer(), task);

        taskRepository.delete(task);

        eventBus.publish(new TaskDeletedEvent(task));
    }

    @Timed
    public Task copyTask(String taskToCopyId, String targetContainerId, int targetPosition) {
        checkArgument(StringUtils.isNotEmpty(taskToCopyId), "Id of task to copy must be provided");
        checkArgument(StringUtils.isNotEmpty(targetContainerId), "Target container id must be provided");

        Task taskToCopy = findById(taskToCopyId);
        final TaskContainer taskContainer = findContainerById(targetContainerId);
        checkArgument(targetPosition <= (taskContainer.getTasks().size()), "Target position must be between 0 and %s" +
                ".", taskContainer.getTasks().size());
        return copyTask(taskToCopy, taskContainer, targetPosition);
    }

    @Timed
    public void notifyOverdueTasks(Release release, List<String> taskIds) {
        forReleaseTasks(release, taskIds, this::sendOverdue);
    }

    private void sendOverdue(Task task) {
        eventBus.publish(new TaskOverdueEvent(task));
        task.setOverdueNotified(true);
    }

    @Timed
    public void notifyTasksDueSoon(Release release, List<String> taskIds) {
        forReleaseTasks(release, taskIds, this::sendDueSoon);
    }

    private void forReleaseTasks(Release release, List<String> taskIds, Consumer<Task> taskOperation) {
        Task[] tasks = taskIds.stream().map(taskId -> {
            Task task = release.getTask(taskId);
            taskOperation.accept(task);
            return task;
        }).toArray(Task[]::new);
        taskRepository.updateTasks(tasks);
    }

    private void sendDueSoon(Task task) {
        eventBus.publish(new TaskDueSoonEvent(task));
        task.setDueSoonNotified(true);
    }

    @Timed
    public Task duplicateTask(String originTaskId) {
        Task taskToCopy = findById(originTaskId);
        TaskContainer container = taskToCopy.getContainer();
        int targetPosition = container.getTasks().indexOf(taskToCopy) + 1;
        return copyTask(taskToCopy, container, targetPosition);
    }

    @Timed
    public Task moveTask(MovementIndexes movementIndexes) {
        if (isTaskMovedWithinSameContainer(movementIndexes)) {
            return moveTaskWithinContainer(movementIndexes.getOriginContainerId(), movementIndexes.getOriginIndex(),
                    movementIndexes.getTargetIndex());
        } else {
            return moveTaskBetweenContainers(movementIndexes.getOriginContainerId(), movementIndexes.getOriginIndex()
                    , movementIndexes.getTargetContainerId(), movementIndexes.getTargetIndex());
        }
    }

    @Timed
    public String getTitle(String id) {
        return taskRepository.getTitle(id);
    }

    @Timed
    public String getType(String id) {
        return taskRepository.getType(id);
    }

    @Timed
    public Task lockTask(String id) {
        return setLock(id, true);
    }

    @Timed
    public Task unlockTask(String id) {
        return setLock(id, false);
    }

    @Timed
    public Set<String> getAllTags(int limitNumber) {
        return taskRepository.getAllTags(limitNumber);
    }

    @Timed
    public TaskStatus getStatus(String taskId) {
        return taskRepository.getStatus(taskId);
    }

    @Timed
    public Map<String, TaskStatus> getTaskStatuses(String releaseId) {
        return taskRepository.getTaskStatuses(releaseId);
    }

    @Timed
    public Boolean isUserInTeam(String taskId, String userName) {
        Task task = findById(taskId);
        Release release = task.getRelease();
        if (task.hasTeam()) {
            try {
                Team team = teamService.findTeamByName(release.getId(), task.getTeam()).orElseThrow();
                return team.hasMember(userName);
            } catch (NoSuchElementException e) {
                return false;
            }
        }
        return false;
    }

    private Task setLock(String id, boolean lock) {
        Task task = taskRepository.findById(id);
        checkTaskIsUpdatable(task);

        List<Task> updatedTasks = new ArrayList<>();
        task.getAllTasks().stream()
                .filter(candidateTask -> candidateTask.isLocked() != lock)
                .forEach(taskToUpdate -> {
                    taskToUpdate.setLocked(lock);
                    taskRepository.updateTaskProperties(taskToUpdate);
                    updatedTasks.add(taskToUpdate);
                });
        // this should be enough to update the release JSON, afaik we only need
        // the original release to update the tags
        releaseService.updateReleaseProperties(null, task.getRelease());
        String releaseId = task.getRelease().getId();
        TaskEvent event = lock ? new TasksLockedEvent(releaseId, updatedTasks) : new TasksUnlockedEvent(releaseId,
                updatedTasks);
        eventBus.publish(event);
        return task;
    }

    private void applyNewOwner(final String newOwner, final Task task) {
        String previousOwner = task.getOwner();
        if (!equalsIgnoreCase(newOwner, previousOwner)) {
            checkTaskIsUpdatable(task);

            Task original = cloneCi(task.getRelease()).getTask(task.getId());
            task.setOwner(newOwner);
            task.setOverdueNotified(false);
            task.setDueSoonNotified(false);

            scanAndBuildNewVariables(task.getRelease(), task, ciIdService);

            taskRepository.update(task);

            eventBus.publish(new TaskUpdatedEvent(original, task));
        }
    }

    private Task moveTaskWithinContainer(String containerId, Integer originIndex, Integer targetIndex) {
        TaskContainer taskContainer = findContainerById(containerId);

        List<Task> tasks = taskContainer.getTasks();
        Task taskToMove = tasks.get(originIndex);
        Task targetTask = getTaskFromContainer(taskContainer, targetIndex);
        checkArgument(((PlanItem) taskContainer).isUpdatable() &&
                areAllTasksMovable(taskToMove, targetTask), "Only planned and completed in advance tasks can be moved");
        LockedTaskOperationChecks.checkMoveTask(taskToMove, taskContainer, taskContainer);
        tasks.remove(taskToMove);
        tasks.add(targetIndex, taskToMove);


        Task savedTask = taskRepository.moveTask(taskToMove, taskToMove, taskContainer, taskContainer);

        eventBus.publish(new TaskMovedEvent(savedTask, originIndex, targetIndex, taskToMove.getId(), containerId,
                containerId));

        return savedTask;
    }

    private boolean isTaskMovedWithinSameContainer(MovementIndexes movementIndexes) {
        return movementIndexes.getOriginContainerId().equals(movementIndexes.getTargetContainerId());
    }

    private Task moveTaskBetweenContainers(String originContainerId, Integer originIndex, String targetContainerId,
                                           Integer targetIndex) {
        Release release = releaseService.findById(releaseIdFrom(originContainerId));
        TaskContainer originContainer = getTaskContainer(release, originContainerId);
        TaskContainer targetContainer = getTaskContainer(release, targetContainerId);

        List<Task> originContainerTasks = originContainer.getTasks();
        Task taskToMove = originContainerTasks.get(originIndex);
        Task targetTask = getTaskFromContainer(targetContainer, targetIndex);
        checkArgument(areAllTasksMovable(taskToMove, targetTask) &&
                ((PlanItem) originContainer).isUpdatable() &&
                ((PlanItem) targetContainer).isUpdatable(), "Only planned and completed in advance tasks can be moved");
        LockedTaskOperationChecks.checkMoveTask(taskToMove, originContainer, targetContainer);

        String newId = getUniqueId(targetContainer.getId());

        clearLinks(originContainer, taskToMove);
        Task movedTask = cloneCi(taskToMove);
        rewriteWithNewId(movedTask, newId);

        originContainer.getTasks().remove(taskToMove);
        targetContainer.addTask(movedTask, targetIndex);
        movedTask.setContainer(targetContainer);

        Task savedTask = taskRepository.moveTask(taskToMove, movedTask, originContainer, targetContainer);

        eventBus.publish(new TaskMovedEvent(savedTask, originIndex, targetIndex, taskToMove.getId(),
                originContainerId, targetContainerId));

        return savedTask;
    }

    private Task getTaskFromContainer(TaskContainer taskContainer, Integer index) {
        List<Task> containerTasks = taskContainer.getTasks();
        if (containerTasks.size() > index) {
            return containerTasks.get(index);
        } else {
            return null;
        }
    }

    private boolean areAllTasksMovable(final Task... tasks) {
        return Arrays.stream(tasks).filter(java.util.Objects::nonNull).allMatch(Task::isMovable);
    }

    private void clearLinks(TaskContainer originContainer, Task task) {
        if (originContainer instanceof ParallelGroup) {
            Set<Link> allLinks = ((ParallelGroup) task.getContainer()).getLinks();
            // do not use remove methods from set here
            Set<Link> links =
                    allLinks.stream().filter(l -> !((ParallelGroup) task.getContainer()).getLinksOf(task).contains(l)).collect(Collectors.toSet());
            ((ParallelGroup) task.getContainer()).setLinks(links);
        }
    }


    private TaskContainer getTaskContainer(Release release, String containerId) {
        if (isPhaseId(containerId)) {
            return release.getPhase(containerId);
        } else {
            return (TaskContainer) release.getTask(containerId);
        }
    }

    private Task copyTask(final Task taskToCopy, final TaskContainer taskContainer, final int targetPosition) {
        validateTaskCopy(taskToCopy, taskContainer, targetPosition);

        String targetFolderId = findFolderId(taskContainer.getId());
        String targetReleaseId = releaseIdFrom(taskContainer.getId());
        Release targetRelease = releaseService.findById(targetReleaseId);

        Task newTask;
        if (isDifferentContainer(taskToCopy, targetReleaseId, targetFolderId)) {
            newTask = copyTaskBetweenContainers(taskToCopy, targetRelease, taskContainer, targetPosition);
        } else {
            newTask = copyTaskInsideContainer(taskToCopy, targetRelease, taskContainer, targetPosition);
        }

        return newTask;
    }

    private void validateTaskCopy(Task taskToCopy, TaskContainer taskContainer, int targetPosition) {
        LockedTaskOperationChecks.checkCopyTask(taskToCopy);
        boolean isInProgressParallelGroup = taskContainer instanceof ParallelGroup && ((ParallelGroup) taskContainer).isInProgress();
        boolean isUpdatableOrDoneInAdvance = taskToCopy.isUpdatable() || taskToCopy.isDoneInAdvance();
        checkArgument(!isInProgressParallelGroup, "Can't copy task within 'in progress' parallel group.");
        checkArgument(isUpdatableOrDoneInAdvance, "Can't copy task '%s' because it is %s.", taskToCopy.getTitle(), taskToCopy.getStatus());
        checkArgument(targetPosition >= 0, "Target position must be greater or equal to 0.");
    }

    private boolean isDifferentContainer(Task taskToCopy, String targetReleaseId, String targetFolderId) {
        return !taskToCopy.getRelease().getId().equalsIgnoreCase(targetReleaseId) || !targetFolderId.equals(findFolderId(taskToCopy.getId()));
    }

    private boolean isScriptOrCustomScriptTask(Task task) {
        return task instanceof ScriptTask || task instanceof CustomScriptTask;
    }

    private void checkIsStatusUpdatable(Task task, String action) {
        if (!canUpdateInProgressStatus(task)) {
            throw new IllegalArgumentException("You can not " + action + " a task that is : not in progress, " +
                    "automated, or unassigned");
        }
        if (task.isTaskGroup()) {
            throw new IllegalArgumentException("You can not " + action + " a task group");
        }
    }

    private void checkIsReleaseActive(Release release) {
        checkState(release.isActive(), "The release '%s' must be in progress, failing or failed.", release.getTitle());
    }

    private void checkIsStatusCompletableInAdvance(Task task) {
        checkIsReleaseActive(task.getRelease());
        LockedTaskOperationChecks.checkCompleteTaskInAdvance(task);
        boolean isAutomated = task.getProperty("automated");
        if (!canUpdatePlannedStatus(task) || isAutomated) {
            throw new IllegalArgumentException("You can not complete in advance a task that is : not planned or " +
                    "unassigned or automated");
        }
    }

    private void checkIsTaskSkippable(Task task) {
        LockedTaskOperationChecks.checkSkipTask(task);
        if (!canUpdateInProgressStatus(task) && !canUpdateFailedStatus(task)) {
            throw new IllegalArgumentException("You can not skip a task that is : (in progress and automated or " +
                    "unassigned) or (failed and not automated or unassigned)");
        }
        if (task.isTaskGroup() && !((TaskGroup) task).isSkippableOrRetriable()) {
            throw new IllegalArgumentException("You can not skip this task group");
        }
    }

    private void checkIsTaskSkippableInAdvance(Task task) {
        LockedTaskOperationChecks.checkSkipTask(task);
        if (!canUpdatePlannedStatus(task)) {
            throw new IllegalArgumentException("You can not skip in advance a task that is : not planned or " +
                    "unassigned");
        }
        checkIsReleaseActive(task.getRelease());
    }

    private void checkCanTaskBeRetried(Task task) {
        if (!canUpdateFailedStatus(task)) {
            throw new IllegalArgumentException("You can not retry a task that is : failed and not automated or " +
                    "unassigned");
        }
        if (task instanceof TaskGroup && !((TaskGroup) task).isSkippableOrRetriable()) {
            throw new IllegalArgumentException("You can not retry this task group");
        }
        if (task instanceof CustomScriptTask && ((CustomScriptTask) task).isUnknown()) {
            throw new IllegalArgumentException(format("You can not retry '%s' task", UNKNOWN_TYPE));
        }
        if (task instanceof ContainerTask && ((ContainerTask) task).isUnknown()) {
            throw new IllegalArgumentException(format("You can not retry '%s' task", UNKNOWN_TASK_TYPE));
        }
    }

    private void checkCanTaskBeReopen(Task task) {
        checkIsReleaseActive(task.getRelease());
        if (!task.isDoneInAdvance()) {
            throw new IllegalArgumentException("You can not reopen a task that is not done in advance");
        }
    }

    private void checkCanTaskBeAborted(final Task task) {
        if (task.isInProgress() || task.isAbortScriptInProgress()) {
            checkArgument(task instanceof BaseScriptTask || task instanceof ContainerTask || task instanceof CreateReleaseTask,
                    "Unable to abort a task '%s' of type '%s'. " +
                            "When a task is running or queued, only ScriptTask, CustomScriptTask, CreateReleaseTask or container based " +
                            "task may be cancelled",
                    task.getId(), task.getTaskType()
            );
        } else {
            checkArgument(task.isPreconditionInProgress() || task.isFailureHandlerInProgress() || task.isFacetInProgress(),
                    "Only task currently evaluating a precondition, on failure handler state or verifying may be " +
                            "cancelled, but task status is %s", task.getStatus().name());
        }
    }

    private boolean canUpdateInProgressStatus(Task task) {
        return task.isInProgress() && !task.isAutomated() && (task.hasOwner() || task.hasTeam());
    }

    private boolean canUpdatePlannedStatus(Task task) {
        return task.isPlanned() && (task.hasOwner() || task.hasTeam());
    }

    private boolean canUpdateFailedStatus(Task task) {
        boolean isAutomated = task.getProperty("automated");
        return (task.isFailed() || task.isFailureHandlerInProgress()) && (isAutomated || task.hasOwner() || task.hasTeam());
    }

    private <T> List<T> executeBulkActionSilently(List<T> items, Function<T, ?> action) {
        return items.stream().filter(item -> {
            try {
                action.apply(item);
                return true;
            } catch (Exception e) {
                return false;
            }
        }).collect(toList());
    }

    private void checkTaskIsUpdatable(Task task) {
        checkArgument(task.isUpdatable(), "Can't update task '%s' because it is in state %s", task.getTitle(),
                task.getStatus());
    }

    private boolean isScheduledStartDateUpdatable(Task original, Task updated) {
        return original.isPlanned() || original.isPending() || equal(original.getScheduledStartDate(),
                updated.getScheduledStartDate());
    }

    private TaskUpdater getTaskUpdater(Class<? extends Task> taskType) {
        TaskUpdater updater = taskUpdatersPerType.get(taskType);
        return updater != null ? updater : taskUpdatersPerType.get(Task.class); // the DefaultTaskUpdater
    }

    private TaskContainer findContainerById(final String containerId) {
        if (Ids.isPhaseId(containerId)) {
            return phaseService.findById(containerId);
        } else {
            Task container = findById(containerId);
            if (!(container instanceof TaskGroup)) {
                throw new IllegalArgumentException("Task can only be added to phases or task groups");
            }
            return (TaskGroup) container;
        }
    }

    private boolean isGlobalOrAncestor(String folderId, Release release) {
        return folderId == null || release.getId().startsWith(folderId);
    }

    private void checkIfConfigurationIsInherited(Release release, Set<ConfigurationItem> refs) {
        if (!refs.stream().filter(r -> r instanceof BaseConfiguration)
                .allMatch(r -> isGlobalOrAncestor(((BaseConfiguration) r).getFolderId(), release))) {
            throw new IllegalArgumentException("The configuration is not inherited by the release of the updated task");
        }
    }

    private Task createTask(TaskContainer taskContainer, @Nullable String generatedTaskId, Task task, Integer position, boolean checkPasswords) {
        addTaskToContainer(taskContainer, generatedTaskId, task, position);
        configureTask(task, checkPasswords, true);

        Task createdTask = taskRepository.create(task);
        eventBus.publish(new TaskCreatedEvent(createdTask));

        return createdTask;
    }

    private Task copyTaskBetweenContainers(Task taskToCopy, Release targetRelease, TaskContainer taskContainer, int targetPosition) {
        StringBuilder taskCopyMessage = generateTaskCopyMessage(taskToCopy);
        String targetFolderId = findFolderId(taskContainer.getId());
        Set<String> referencedVariableKeys = taskToCopy.getReferencedVariableKeys();
        taskToCopy.getAttachments().clear();

        //Process folder references that need attention in target folder
        if (!targetFolderId.equals(findFolderId(taskToCopy.getId()))) {
            generateMissingTeamsMessageForTaskCopy(taskToCopy, targetFolderId, taskCopyMessage);

            if (!targetFolderId.equals(ROOT_FOLDER_ID) && !findFolderId(taskToCopy.getId()).equals(ROOT_FOLDER_ID)) {
                handleFolderConfigurationReferences(taskToCopy, taskCopyMessage, targetFolderId);
                List<Variable> sourceFolderVariables = taskToCopy.getRelease().getFolderVariables().getVariables().stream()
                        .filter(variable -> referencedVariableKeys.contains(variable.getKey())).toList();
                generateMissingFolderVariablesMessageForTaskCopy(sourceFolderVariables,
                        targetRelease.getFolderVariables().getVariables(), taskCopyMessage);
                generateMismatchedFolderVariablesMessageForTaskCopy(sourceFolderVariables,
                        targetRelease.getFolderVariables().getVariables(), taskCopyMessage);
            }
        }

        taskCopyHandlers.forEach(taskCopyHandler -> taskCopyHandler.handle(taskToCopy, taskContainer, taskCopyMessage));

        //Process release variables that need attention in target release
        List<Variable> sourceReleaseVariables = extractSourceReleaseVariables(taskToCopy, referencedVariableKeys);
        List<Variable> targetReleaseVariables = targetRelease.getVariables();
        generateMismatchedReleaseVariablesMessageForTaskCopy(sourceReleaseVariables, targetReleaseVariables, taskCopyMessage);
        List<Variable> missingReleaseVariablesInTarget = findMissingVariables(sourceReleaseVariables, targetReleaseVariables);
        List<Variable> existingReleaseVariablesInTarget = findExistingVariablesInTarget(sourceReleaseVariables, targetReleaseVariables);
        generateValueProviderMessageForTaskCopy(missingReleaseVariablesInTarget, taskCopyMessage);
        generateMissingReleaseVariablesMessageForTaskCopy(missingReleaseVariablesInTarget, taskCopyMessage);

        String newTaskId = getUniqueId(taskContainer.getId());
        Task copiedTaskTemplate = prepareCopiedTaskTemplate(taskToCopy, targetRelease, newTaskId);
        addTaskToContainer(taskContainer, newTaskId, copiedTaskTemplate, targetPosition);

        copiedTaskTemplate.getReferencedVariables().forEach(variable -> existingReleaseVariablesInTarget.stream()
                .filter(replacement -> replacement.getKey().equals(variable.getKey()))
                .findFirst()
                .ifPresent(replacement -> variable.setId(replacement.getId())));

        missingReleaseVariablesInTarget.forEach(variable -> variableService.addVariable(copiedTaskTemplate.getRelease(), variable));

        configureTask(copiedTaskTemplate, false, false);

        Task createdTask = taskRepository.create(copiedTaskTemplate);

        copyCommentsIfTemplate(taskToCopy, createdTask);
        commentService.create(createdTask, taskCopyMessage.toString(), AUTHENTICATED_USER, true);
        eventBus.publish(new TaskCopiedFromEvent(createdTask, getReleaseType(taskToCopy), taskToCopy.getRelease().getTitle()));

        return createdTask;
    }

    private Task copyTaskInsideContainer(Task taskToCopy, Release targetRelease, TaskContainer taskContainer, int targetPosition) {
        String newTaskId = getUniqueId(taskContainer.getId());
        Task copiedTaskTemplate = prepareCopiedTaskTemplate(taskToCopy, targetRelease, newTaskId);
        addTaskToContainer(taskContainer, newTaskId, copiedTaskTemplate, targetPosition);
        configureTask(copiedTaskTemplate, false, false);

        Task createdTask = taskRepository.create(copiedTaskTemplate);

        copyCommentsIfTemplate(taskToCopy, createdTask);
        eventBus.publish(new TaskCopiedEvent(createdTask));

        return createdTask;
    }

    private Task prepareCopiedTaskTemplate(Task taskToCopy, Release targetRelease, String newTaskId) {
        Task copiedTaskTemplate = cloneCi(taskToCopy);
        updateTaskTitleIfExists(copiedTaskTemplate, targetRelease);
        copiedTaskTemplate.resetToPlanned();
        rewriteWithNewId(copiedTaskTemplate, newTaskId);
        return copiedTaskTemplate;
    }

    private void updateTaskTitleIfExists(Task copiedTaskTemplate, Release targetRelease) {
        if (targetRelease.getAllTasks().stream().anyMatch(task -> task.getTitle() != null &&
                task.getTitle().equals(copiedTaskTemplate.getTitle()))) {
            copiedTaskTemplate.setTitle(copiedTaskTemplate.getTitle() + " (copy)");
        }
    }

    private void addTaskToContainer(TaskContainer taskContainer, String generatedTaskId, Task task, Integer position) {
        taskAccessService.checkIfAuthenticatedUserCanUseTask(task);
        checkContainerConditionsForAdding(taskContainer, position);
        addToContainerAtPosition(taskContainer, task, position);
        if (generatedTaskId == null) {
            final String id = getUniqueId(task.getContainer().getId());
            task.setId(id);
        }
    }

    private void copyCommentsIfTemplate(Task taskToCopy, Task createdTask) {
        if (taskToCopy.getRelease().isTemplate()) {
            commentService.findByTask(taskToCopy.getId()).forEach(comment -> commentService.create(createdTask, comment.getText(), comment.getAuthor(), true));
        }
    }

    private void configureTask(Task task, boolean checkPasswords, boolean prepopulateCIReferences) {
        task.checkDatesValidity();

        checkIfConfigurationIsInherited(task.getRelease(), CiHelper.getExternalReferences(task));
        task.setStatus(PLANNED);
        task.setStatusLine(null);

        if (task instanceof CustomScriptTask) {
            PythonScript pythonScript = ((CustomScriptTask) task).getPythonScript();
            ResolutionContext resolutionContext = ResolutionContext.apply(Ids.findFolderId(task.getId()));
            checkArgument(pythonScript != null, "Missing pythonScript definition");
            checkArgument(!UNKNOWN_TYPE.equals(pythonScript.getType()), format("You can not create '%s' task",
                    UNKNOWN_TYPE));
            pythonScript.setId(task.getId() + "/" + PYTHON_SCRIPT_ID);
            pythonScript.setCustomScriptTask((CustomScriptTask) task);
            if (prepopulateCIReferences) {
                populateCIReference(pythonScript, resolutionContext);
            }
        } else if (task instanceof CreateReleaseTask) {
            List<Variable> variables = ((CreateReleaseTask) task).getTemplateVariables();
            fixUpVariableIds(task.getId(), variables, ciIdService);
        } else if (task instanceof ContainerTask) {
            checkArgument(!UNKNOWN_TASK_TYPE.equals(task.getType()), format("You can not create '%s' task",
                    UNKNOWN_TASK_TYPE));
        }

        scanAndBuildNewVariables(task.getRelease(), task, ciIdService);

        List<String> aliens = task.getReferencedVariables().stream()
                .filter(variable -> variable.getId() != null && task.getId() != null &&
                        !releaseIdFrom(variable.getId()).equals(releaseIdFrom(task.getId())))
                .map(Variable::getKey)
                .collect(Collectors.toList());
        if (!aliens.isEmpty()) {
            throw new NotFoundException(
                    format("Unable to create task with variables [%s] referencing another release.",
                            String.join(", ", aliens))
            );
        }

        if (checkPasswords) {
            PasswordVerificationUtils.replacePasswordPropertiesInCiIfNeededJava(Optional.empty(), task);
        }
    }

    // When a custom task has a CI reference, we pre-populate this property with the first CI of the good type
    // e.g. jenkins tasks will have his jenkinsServer property populated with the first Jenkins Server we find
    private void populateCIReference(PythonScript pythonScript, ResolutionContext resolutionContext) {
        for (PropertyDescriptor propertyDescriptor : pythonScript.getInputProperties()) {
            if (propertyDescriptor.getKind() == CI && propertyDescriptor.isRequired() && propertyDescriptor.get(pythonScript) == null) {
                Type referencedType = propertyDescriptor.getReferencedType();

                if (referencedType.isSubTypeOf(Type.valueOf(Configuration.class))) {
                    Configuration configuration =
                            configurationRepository.<Configuration>findFirstByType(referencedType, resolutionContext).orElse(null);
                    propertyDescriptor.set(pythonScript, configuration);
                }
            }
        }
    }

    private void checkContainerConditionsForAdding(TaskContainer container, Integer position) {
        LockedTaskOperationChecks.checkCreateTaskInContainer(container);
        final String message = "Can't add a task to the %s '%s' because it is in state %s.";
        if (container instanceof Phase) {
            Phase phase = (Phase) container;
            checkArgument(phase.isUpdatable(), message, "phase", phase.getTitle(), phase.getStatus());
            if (position != null) {
                checkArgument(position >= 0 && position <= phase.getTasks().size(), "Task index out of bounds");
                if (position < phase.getTasks().size()) {
                    Task taskAfterPosition = phase.getTasks().get(position);
                    checkArgument(taskAfterPosition.isPlanned() || taskAfterPosition.isDoneInAdvance(),
                            "Can't add a task before a task that is active or done");
                }
            }
        } else if (container instanceof SequentialGroup) {
            SequentialGroup group = (SequentialGroup) container;
            checkArgument(group.isUpdatable(), message, "sequential group", group.getTitle(), group.getStatus());
        } else if (container instanceof ParallelGroup) {
            ParallelGroup group = (ParallelGroup) container;
            checkArgument(group.getStatus().isOneOf(PLANNED, PENDING, WAITING_FOR_INPUT),
                    message, "parallel group", group.getTitle(), group.getStatus());
        }
    }

    private void addToContainerAtPosition(TaskContainer container, Task task, Integer position) {
        container.addTask(task, position != null ? position : container.getTasks().size());
        task.setContainer(container);
    }

    private List<Variable> extractSourceReleaseVariables(Task taskToCopy, Set<String> referencedVariableKeys) {
        return taskToCopy.getRelease().getVariables().stream()
                .filter(variable -> referencedVariableKeys.contains(variable.getKey()))
                .toList();
    }

    private void initMissingDataSectionTaskCopyMessage(final StringBuilder missingDataComment) {
        String missingDataString = "**Missing data in destination.** Create or update the following:\n";
        if (!missingDataComment.toString().contains(missingDataString)) {
            missingDataComment.append(TASK_COPY_MESSAGE_DELIMITER).append(missingDataString);
        }
    }

    private void initMismatchedVariablesSectionTaskCopyMessage(StringBuilder missingDataComment) {
        String missingDataString = "**Type mismatch.** Review and update:\n";
        if (!missingDataComment.toString().contains(missingDataString)) {
            missingDataComment.append(TASK_COPY_MESSAGE_DELIMITER).append("**Type mismatch.** Review and update:\n");
        }
    }

    private StringBuilder generateTaskCopyMessage(Task taskToCopy) {
        StringBuilder taskCopyMessage = new StringBuilder();
        taskCopyMessage.append("Task copied from ").append(getReleaseType(taskToCopy)).append(" ")
                .append(formatReleaseUrl(taskToCopy)).append(" task ").append(formatTaskUrl(taskToCopy)).append(".\n");
        return taskCopyMessage;
    }

    private void generateMissingTeamsMessageForTaskCopy(Task task, String targetFolderId, StringBuilder taskCopyMessage) {
        List<String> missingTeamNames = findMissingTeamNames(task, targetFolderId);

        if (!missingTeamNames.isEmpty()) {
            initMissingDataSectionTaskCopyMessage(taskCopyMessage);
            taskCopyMessage.append("\nTeams:\n");
            missingTeamNames.forEach(teamName -> taskCopyMessage.append(String.format(TASK_COPY_MESSAGE_ITEM_FORMAT_1, teamName)));
            taskCopyMessage.append("\n");
        }
    }

    private List<String> findMissingTeamNames(Task taskToCopy, String targetFolderId) {
        Set<String> taskTeamNames = taskToCopy.getAllTasks().stream()
                .map(Task::getTeam)
                .filter(java.util.Objects::nonNull)
                .collect(Collectors.toSet());

        Folder targetFolder = folderService.findById(targetFolderId, 0);
        Set<String> destinationFolderTeamNames = teamService.getEffectiveTeams(targetFolder).stream()
                .map(Team::getTeamName)
                .collect(Collectors.toSet());

        return taskTeamNames.stream()
                .filter(teamName -> !destinationFolderTeamNames.contains(teamName))
                .collect(Collectors.toList());
    }

    private void generateMismatchedFolderVariablesMessageForTaskCopy(List<Variable> sourceFolderVariables,
                                                                     List<Variable> targetFolderVariables,
                                                                     StringBuilder taskCopyMessage) {
        List<MismatchedVariableHolder> variablesOfDifferentType =
                findVariablesWithDifferingType(sourceFolderVariables, targetFolderVariables);
        if (!variablesOfDifferentType.isEmpty()) {
            initMismatchedVariablesSectionTaskCopyMessage(taskCopyMessage);
            taskCopyMessage.append("\nFolder variables: \n");
            variablesOfDifferentType.forEach(entry -> taskCopyMessage.append(String.format("* %s (source: %s, destination: %s)\n",
                    entry.getSourceVariable().getKey(),
                    typeToDisplayValue(entry.getSourceVariable()),
                    typeToDisplayValue(entry.getTargetVariable()))));
            taskCopyMessage.append("\n");
        }
    }

    private void generateMissingFolderVariablesMessageForTaskCopy(List<Variable> sourceFolderVariables,
                                                                  List<Variable> targetFolderVariables,
                                                                  StringBuilder taskCopyMessage) {
        List<Variable> missingVariables = findMissingVariables(sourceFolderVariables, targetFolderVariables);
        if (!missingVariables.isEmpty()) {
            initMissingDataSectionTaskCopyMessage(taskCopyMessage);
            taskCopyMessage.append("\nFolder variables: \n");
            missingVariables.forEach(variable -> taskCopyMessage.append(String.format(TASK_COPY_MESSAGE_ITEM_FORMAT_2, variable.getKey(),
                    typeToDisplayValue(variable))));
            taskCopyMessage.append("\n");
        }
    }

    private void generateMismatchedReleaseVariablesMessageForTaskCopy(List<Variable> sourceVariables,
                                                                      List<Variable> targetVariables,
                                                                      StringBuilder taskCopyMessage) {
        List<MismatchedVariableHolder> variablesOfDifferentType = findVariablesWithDifferingType(sourceVariables,
                targetVariables);
        if (!variablesOfDifferentType.isEmpty()) {
            initMismatchedVariablesSectionTaskCopyMessage(taskCopyMessage);
            taskCopyMessage.append("\nRelease variables: \n");
            variablesOfDifferentType.forEach(entry -> taskCopyMessage.append(String.format("* %s (source: %s, destination: %s)\n",
                    entry.getSourceVariable().getKey(),
                    typeToDisplayValue(entry.getSourceVariable()),
                    typeToDisplayValue(entry.getTargetVariable()))));
            taskCopyMessage.append("\n");
        }
    }

    private void generateValueProviderMessageForTaskCopy(List<Variable> missingVariables, StringBuilder taskCopyMessage) {
        List<Variable> valueProviderVariables =
                missingVariables.stream().filter(variable -> variable.getValueProvider() != null &&
                        !variable.getValueProvider().getType()
                                .instanceOf(Type.valueOf(ListOfStringValueProviderConfiguration.class))).toList();
        if (!valueProviderVariables.isEmpty()) {
            taskCopyMessage.append(TASK_COPY_MESSAGE_DELIMITER).append("**Value provider needed.** Configure for variables:\n");
            valueProviderVariables.forEach(variable -> taskCopyMessage.append(String.format("* %s (%s, Value provider) \n",
                    variable.getKey(), typeToDisplayValue(variable))));
            taskCopyMessage.append("\n");
        }
    }

    private void generateMissingReleaseVariablesMessageForTaskCopy(List<Variable> missingVariables, StringBuilder taskCopyMessage) {
        if (!missingVariables.isEmpty()) {
            taskCopyMessage.append(TASK_COPY_MESSAGE_DELIMITER).append("**Info.** Variables created:\n");
            missingVariables.forEach(variable -> taskCopyMessage.append(String.format(TASK_COPY_MESSAGE_ITEM_FORMAT_2, variable.getKey(), typeToDisplayValue(variable))));
            taskCopyMessage.append("\n");
        }
    }

    private void generateMissingFolderConnectionsMessageForTaskCopy(List<ConfigurationItem> missingFolderConfigurationRefsInTarget,
                                                                    StringBuilder taskCopyMessage) {
        if (missingFolderConfigurationRefsInTarget.isEmpty()) {
            return;
        }

        Set<String> missingFolderConfigurations = missingFolderConfigurationRefsInTarget.stream()
                .map(ref -> {
                    Configuration configItem = sharedConfigurationService.findById(ref.getId());
                    return String.format(TASK_COPY_MESSAGE_ITEM_FORMAT_2, configItem.getTitle(), configItem.getType().toString());
                })
                .collect(Collectors.toSet());

        if (!missingFolderConfigurations.isEmpty()) {
            initMissingDataSectionTaskCopyMessage(taskCopyMessage);
            taskCopyMessage.append("\nConnections: \n");
            missingFolderConfigurations.forEach(taskCopyMessage::append);
        }
    }

    private String getReleaseType(final Task task) {
        return task.getRelease().isTemplate()
                ? (task.getRelease().isWorkflow() ? "workflow-template" : "template")
                : (task.getRelease().isWorkflow() ? "workflow-execution" : "release");
    }

    private String typeToDisplayValue(Variable variable) {
        if (variable.getValueProvider() != null) {
            return "Multi-select List box";
        } else {
            return switch (variable.getType().getName()) {
                case "StringVariable" -> "Text";
                case "PasswordStringVariable" -> "Password";
                case "BooleanVariable" -> "Checkbox (boolean)";
                case "IntegerVariable" -> "Number (integer)";
                case "ListStringVariable" -> "List";
                case "ListOfStringValueProviderConfiguration" -> "Multi-select List box";
                case "DateVariable" -> "Date";
                case "MapStringStringVariable" -> "Key-value map";
                case "SetStringVariable" -> "Set";
                case "ReferenceVariable" -> "Reference";
                default -> "Unknown";
            };
        }
    }

    private List<Variable> findMissingVariables(List<Variable> sourceVariables, List<Variable> targetVariables) {
        Map<String, Variable> targetVariableMap = targetVariables.stream()
                .collect(Collectors.toMap(Variable::getKey, Function.identity()));
        return sourceVariables.stream().filter(entry -> !targetVariableMap.containsKey(entry.getKey())).toList();
    }

    private List<Variable> findExistingVariablesInTarget(List<Variable> sourceVariables,
                                                         List<Variable> targetVariables) {
        Map<String, Variable> sourceVariableMap = sourceVariables.stream()
                .collect(Collectors.toMap(Variable::getKey, Function.identity()));
        return targetVariables.stream()
                .filter(entry -> sourceVariableMap.containsKey(entry.getKey()))
                .collect(Collectors.toList());
    }

    private List<MismatchedVariableHolder> findVariablesWithDifferingType(List<Variable> sourceVariables,
                                                                          List<Variable> targetVariables) {
        Map<String, Variable> targetVariableMap = targetVariables.stream()
                .collect(Collectors.toMap(Variable::getKey, Function.identity()));
        return sourceVariables.stream()
                .filter(entry -> targetVariableMap.containsKey(entry.getKey()) &&
                        !targetVariableMap.get(entry.getKey()).getType().equals(entry.getType()))
                .map(mismatchedVariable -> new MismatchedVariableHolder(mismatchedVariable,
                        targetVariableMap.get(mismatchedVariable.getKey())))
                .collect(Collectors.toList());
    }

    private String formatReleaseUrl(Task task) {
        String url = getReleaseBaseUrl(task);
        return getFormattedUrl(task.getRelease().getTitle(), url);
    }

    private String formatTaskUrl(Task task) {
        String taskUrl = getReleaseBaseUrl(task) + "?openTaskDetailsModal=" +
                Ids.getName(task.getPhase().getId()) +
                "-" +
                Ids.getName(task.getId());
        return getFormattedUrl(task.getTitle(), taskUrl);
    }

    private String getFormattedUrl(String title, String url) {
        return String.format("[%s](%s)", title, url);
    }

    private String getReleaseBaseUrl(Task task) {
        String releaseId = releaseIdFrom(task.getRelease().getId()).substring("Applications/".length())
                .replace('/', '-');
        return String.format("%s#/%s/%s", serverConfiguration.getServerUrl(), getReleaseUrlIdentifier(task), releaseId);
    }

    private String getReleaseUrlIdentifier(Task taskToCopy) {
        if (taskToCopy.getRelease().isTemplate()) {
            if (taskToCopy.getRelease().isWorkflow()) {
                return "workflows";
            } else {
                return "templates";
            }
        } else {
            return "releases";
        }
    }

    private void handleFolderConfigurationReferences(Task task, StringBuilder taskCopyMessage, String targetFolderId) {
        List<ConfigurationItem> removedFolderConfigurationReferences = new ArrayList<>();
        task.getAllTasks().forEach(childTask -> findMissingConfigurationReferences(childTask, removedFolderConfigurationReferences, targetFolderId));
        generateMissingFolderConnectionsMessageForTaskCopy(removedFolderConfigurationReferences, taskCopyMessage);
    }

    private void findMissingConfigurationReferences(Task task, List<ConfigurationItem> removedFolderConfigurationReferences, String targetFolderId) {
        Set<ConfigurationItem> taskFolderConfigurationReferences = CiHelper.getExternalReferences(task).stream()
                .filter(ref -> ref.getId().startsWith("Configuration/"))
                .filter(ref -> {
                    Optional<String> folderId = Optional.ofNullable(configurationRepository.read(ref.getId()).getFolderId());
                    return folderId.isPresent() && !targetFolderId.startsWith(folderId.get());
                })
                .collect(Collectors.toSet());

        if (!taskFolderConfigurationReferences.isEmpty()) {
            if (task instanceof CustomScriptTask) {
                PythonScript pythonScript = ((CustomScriptTask) task).getPythonScript();
                List<PropertyDescriptor> inputProperties = pythonScript.getInputProperties().stream()
                        .filter(ofKind(CI)).collect(Collectors.toList());

                taskFolderConfigurationReferences.stream()
                        .filter(ref -> inputProperties.stream().anyMatch(property -> property.getReferencedType().equals(ref.getType())))
                        .forEach(ref -> {
                            inputProperties.stream()
                                    .filter(property -> property.getReferencedType().equals(ref.getType()))
                                    .forEach(property -> property.set(pythonScript, null));
                            removedFolderConfigurationReferences.add(ref);
                        });
            } else {
                task.getTaskType().getDescriptor().getPropertyDescriptors().stream()
                        .filter(pd -> "input".equals(pd.getCategory()))
                        .forEach(pd -> taskFolderConfigurationReferences.stream()
                                .filter(ref -> ref.getType().equals(pd.getReferencedType()))
                                .forEach(ref -> {
                                    pd.set(task, null);
                                    removedFolderConfigurationReferences.add(ref);
                                }));
            }
        }
    }
}
