package com.xebialabs.xlrelease.actors.utils

import com.xebialabs.xlrelease.actors.ReleaseSupervisorActor.ReleasePoisonPill
import grizzled.slf4j.Logging
import org.apache.pekko.actor._
import org.apache.pekko.event.LoggingReceive

import java.util.UUID
import java.util.concurrent.TimeoutException
import scala.concurrent.duration._
import scala.concurrent.{ExecutionContextExecutor, Future, Promise}
import scala.language.postfixOps


trait ActorLifecycleUtils extends Logging {

  import messages._

  private object messages {

    case object GiveUpSearch

    case object GiveUpTermination

    case object GiveUpStart

    case object Retry

  }

  def findAndTerminate(path: String,
                       terminationTimeout: FiniteDuration = 5 seconds,
                       searchTimeout: FiniteDuration = 1 second,
                       terminationMsg: Option[AnyRef] = Some(ReleasePoisonPill)
                      )(implicit system: ActorSystem): Future[String] = {
    logger.debug(s"Terminating all actors by path [$path]")
    actorTermination(path, terminationTimeout = terminationTimeout, searchTimeout = searchTimeout, terminateWithMessageOpt = terminationMsg)
  }

  def actorStart(path: String, timeout: FiniteDuration = 2 seconds,
                 periodicity: FiniteDuration = 100 millis)(implicit system: ActorSystem): Future[String] = {
    val started = Promise[String]()
    system.actorOf(Props(new StartupWaitingActor(path, started, timeout = timeout, periodicity = periodicity)))
    started.future
  }

  private[utils] def actorTermination(path: String, terminationTimeout: FiniteDuration = 5 seconds,
                                      searchTimeout: FiniteDuration = 1 second,
                                      terminateWithMessageOpt: Option[AnyRef] = None)(implicit system: ActorSystem): Future[String] = {
    val terminated = Promise[String]()
    system.actorOf(Props(new TerminatingActor(path, terminated, terminationTimeout, searchTimeout, terminateWithMessageOpt)))
    terminated.future
  }


  private class TerminatingActor(path: String,
                         terminatedPromise: Promise[String],
                         terminationTimeout: FiniteDuration = 5 seconds,
                         searchTimeout: FiniteDuration = 1 second,
                         terminateWithMessageOpt: Option[AnyRef] = None) extends Actor with ActorLogging {

    private val msgId = UUID.randomUUID().toString

    override def preStart(): Unit = {
      context.system.actorSelection(path) ! Identify(msgId)
      implicit val dispatcher: ExecutionContextExecutor = context.system.dispatcher
      scheduledMessages = List(
        context.system.scheduler.scheduleOnce(searchTimeout, self, GiveUpSearch),
        context.system.scheduler.scheduleOnce(terminationTimeout, self, GiveUpTermination))
      super.preStart()
    }

    override def receive: Receive = LoggingReceive {
      collectIds() orElse handleGiveUpSearch()
    }

    def inProgress(refs: Seq[ActorRef]): Receive = LoggingReceive {
      collectIds(refs) orElse catchTerminated(refs) orElse handleGiveUpTermination(refs)
    }

    def collectIds(refs: Seq[ActorRef] = Nil): Receive = {
      case ActorIdentity(`msgId`, Some(a)) =>
        if (terminateWithMessageOpt.isDefined) {
          logger.debug(s"Identified actor [${a.path.toStringWithoutAddress}], giving it a ${terminateWithMessageOpt.get}")
          a ! terminateWithMessageOpt.get
        } else {
          logger.debug(s"Identified actor [${a.path.toStringWithoutAddress}]")
        }
        context.watch(a)
        context.become(inProgress(refs :+ a))
      case ActorIdentity(`msgId`, None) =>
        logger.debug(s"No release actors to terminate by path `$path`")
        terminatedPromise.success(path)
        cancelScheduledMessages()
        context.stop(self)

    }

    def catchTerminated(expectedActorRefs: Seq[ActorRef]): Receive = {
      case Terminated(terminatedActor) =>
        logger.debug(s"Terminated actor [${terminatedActor.path.toStringWithoutAddress}]")
        val rest = expectedActorRefs.filterNot(_ == terminatedActor)
        if (rest.isEmpty) {
          logger.debug(s"Finished terminating all release actors by path [$path]")
          terminatedPromise.success(path)
          cancelScheduledMessages()
          context.stop(self)
        } else {
          logger.debug(s"Waiting for ${rest.size} more actors to terminate")
          context.become(inProgress(rest))
        }
      case GiveUpSearch =>
          logger.debug(s"Ignoring GiveUpSearch while terminating an actor")
    }

    def handleGiveUpSearch(): Receive = {
      case GiveUpSearch =>
        logger.debug(s"No actors responded by path [$path] in $searchTimeout, probably none exist")
        terminatedPromise.success(path)
        cancelScheduledMessages()
        context.stop(self)
    }

    def handleGiveUpTermination(refs: Seq[ActorRef] = Nil): Receive = {
      case GiveUpTermination if refs.nonEmpty =>
        val actors = refs.map(_.path.toStringWithoutAddress).mkString(", ")
        logger.warn(s"Could not terminate some actors within $terminationTimeout: [$actors]")
        terminatedPromise.failure(new TimeoutException(s"Actors $actors were not terminated within $terminationTimeout"))
        cancelScheduledMessages()
        context.stop(self)
    }

    private var scheduledMessages: List[Cancellable] = _
    private def cancelScheduledMessages(): Unit = scheduledMessages.foreach(_.cancel())
  }

  private class StartupWaitingActor(path: String,
                                    startedPromise: Promise[String],
                                    timeout: FiniteDuration = 2 seconds,
                                    periodicity: FiniteDuration = 100 millis) extends Actor with ActorLogging {

    private val msgId = UUID.randomUUID().toString

    override def preStart(): Unit = {
      super.preStart()
      implicit val dispatcher: ExecutionContextExecutor = context.system.dispatcher
      scheduledMessages = List(
        context.system.scheduler.scheduleOnce(timeout, self, GiveUpStart),
        context.system.scheduler.schedule(0 seconds, periodicity, self, Retry))
    }

    override def receive: Receive = LoggingReceive {
      case Retry =>
        context.system.actorSelection(path) ! Identify(msgId)
      case msg@ActorIdentity(`msgId`, Some(actorRef)) if !startedPromise.isCompleted =>
        startedPromise.success(actorRef.path.toStringWithoutAddress)
        cancelScheduledMessages()
        context.stop(self)

      case GiveUpStart =>
        startedPromise.failure(new TimeoutException(s"Could not find any actors with path [$path] within $timeout"))
        cancelScheduledMessages()
        context.stop(self)
    }

    private var scheduledMessages: List[Cancellable] = _
    private def cancelScheduledMessages(): Unit = scheduledMessages.foreach(_.cancel())
  }

}
