package com.xebialabs.xlrelease.domain.tasks.customscript;

import java.util.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.domain.CustomScriptTask;
import com.xebialabs.xlrelease.domain.PythonScript;
import com.xebialabs.xlrelease.domain.TaskDefinition;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import com.xebialabs.xlrelease.variable.VariableViewHelper;
import com.xebialabs.xlrelease.views.TaskFullView;
import com.xebialabs.xlrelease.views.converters.BaseTaskViewConverter;

import scala.Tuple2;

import static com.xebialabs.deployit.booter.local.utils.Strings.isNotEmpty;
import static com.xebialabs.xlrelease.builder.TaskBuilder.newCustomScript;
import static com.xebialabs.xlrelease.domain.CustomScriptTask.PYTHON_SCRIPT_PREFIX;

@Component
public class CustomScriptTaskViewConverter extends BaseTaskViewConverter<CustomScriptTask> {

    private final ConfigurationRepository configurationRepository;

    @Autowired
    public CustomScriptTaskViewConverter(final ConfigurationRepository configurationRepository) {
        this.configurationRepository = configurationRepository;
    }

    @Override
    protected Class<CustomScriptTask> getTaskClass() {
        return CustomScriptTask.class;
    }

    @Override
    public TaskFullView toFullView(final CustomScriptTask task, final List<Type> allowedTaskTypesForAuthenticatedUser) {
        TaskFullView customTaskView;
        if (task.getPythonScript() != null) {
            customTaskView = super.toFullView(task, allowedTaskTypesForAuthenticatedUser);

            PythonScript pythonScript = task.getPythonScript();
            customTaskView.setScriptDefinitionType(pythonScript.getType().toString());
            Map<String, String> variableMapping = task.getVariableMapping();

            pythonScript.getInputProperties().forEach(pd -> customTaskView.getInputProperties().put(pd.getName(), getViewValue(pd, pythonScript, variableMapping, false)));
            pythonScript.getOutputProperties().forEach(pd -> customTaskView.getOutputProperties().put(pd.getName(), getViewValue(pd, pythonScript, variableMapping, true)));

            customTaskView.setTypeDisplayGroup(TaskDefinition.getDisplayGroup(pythonScript.getType()));
            customTaskView.setTypeDisplayName(TaskDefinition.getDisplayName(pythonScript.getType()));
            customTaskView.setOutputVarHelpText(pythonScript.getOutputVarHelpText());
            customTaskView.setColor(pythonScript.getTaskColor());
            customTaskView.setCustomIconLocation(pythonScript.getIconLocation());
            customTaskView.setCustomIconClass(pythonScript.getIconClass());
            customTaskView.setConfigurationUri(pythonScript.getConfigurationUri());
            customTaskView.setKeepPreviousOutputPropertiesOnRetry(task.isKeepPreviousOutputPropertiesOnRetry());

        } else {
            customTaskView = new TaskFullView();
            customTaskView.setId(task.getId());
            customTaskView.setType(task.getType().toString());
        }

        return customTaskView;
    }

    @Override
    protected CustomScriptTask fromView(final TaskFullView view) {
        CustomScriptTask customScriptTask = newCustomScript(view.getScriptDefinitionType())
                .withId(view.getId())
                .withStatusLine(view.getStatusLine())
                .withKeepPreviousOutputPropertiesOnRetry(view.isKeepPreviousOutputPropertiesOnRetry())
                .build();
        PythonScript pythonScript = customScriptTask.getPythonScript();
        fillProperties(view.getInputProperties(), pythonScript.getInputProperties(), pythonScript);
        fillProperties(view.getOutputProperties(), pythonScript.getOutputProperties(), pythonScript);
        pythonScript.setConfigurationUri(view.getConfigurationUri());
        return customScriptTask;
    }

    @Override
    protected void fillVariableMappings(final TaskFullView view, final CustomScriptTask task) {
        // variable mappings updated by the fillProperties method
    }

    private void fillProperties(final Map<String, Object> viewProperties, final Collection<PropertyDescriptor> propertyDescriptors, final PythonScript pythonScript) {
        propertyDescriptors.forEach( pd ->
            {
                final String propertyName = pd.getName();
                final Object viewValue = viewProperties.get(propertyName);
                if (viewValue == null) {
                    pythonScript.setProperty(propertyName, null);
                } else {
                    Tuple2<String, Object> variableAndValue = VariableViewHelper.fromView(pd, viewValue, configurationRepository);
                    String variable = variableAndValue._1();
                    Object value = variableAndValue._2();
                    if (value != null) {
                        switch (pd.getKind()) {
                            case SET_OF_STRING:
                            case SET_OF_CI:
                                value = new HashSet((Collection<?>) value);
                                break;
                            case LIST_OF_CI:
                            case LIST_OF_STRING:
                                value = new ArrayList((Collection<?>) value);
                                break;
                            default:
                                // nothing
                        }
                    }
                    pythonScript.setProperty(propertyName, value);
                    updateVariableMappingOfProperty(pythonScript, propertyName, variable);
                }
            }

        );
    }

    private void updateVariableMappingOfProperty(final PythonScript pythonScript, final String propertyName, final String variableName) {
        final String fqPropertyName = PYTHON_SCRIPT_PREFIX + propertyName;
        final Map<String, String> variableMapping = pythonScript.getCustomScriptTask().getVariableMapping();
        if (isNotEmpty(variableName)) {
            variableMapping.put(fqPropertyName, variableName);
        } else {
            variableMapping.remove(fqPropertyName);
        }
    }

    private Object getViewValue(final PropertyDescriptor pd, final PythonScript pythonScript, final Map<String, String> variableMapping, final boolean useViewForStrings) {
        return VariableViewHelper.toView(pd, variableMapping.get(CustomScriptTask.PYTHON_SCRIPT_PREFIX + pd.getName()), pythonScript.getProperty(pd.getName()), useViewForStrings);
    }
}
