package com.xebialabs.xlrelease.db

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.security.{PermissionEnforcer, Permissions, RoleService}
import com.xebialabs.xlrelease.api.v1.forms.{ReleaseOrderMode, ReleasesFilters}
import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.sql.SqlBuilder.{Dialect, getOrderAscOrDesc}
import com.xebialabs.xlrelease.db.sql.archiving.SelectArchivedReleasesBuilder
import com.xebialabs.xlrelease.domain.{Release, ReleaseKind}
import com.xebialabs.xlrelease.domain.status.ReleaseStatus.{ABORTED, COMPLETED}
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.domain.utils.{AdaptiveReleaseId, ArchiveDbReleaseId}
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.repository.query.ReleaseBasicDataExt
import com.xebialabs.xlrelease.views.ReleaseOverview
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

@Repository
class ArchivedReleasesSearch @Autowired()(@Qualifier("reportingJdbcTemplate") jdbcTemplate: JdbcTemplate,
                                          implicit val permissionEnforcer: PermissionEnforcer,
                                          @Qualifier("reportingSqlDialect") implicit val reportingSqlDialect: Dialect,
                                          implicit val roleService: RoleService) extends Logging {


  def overviewReleases(filters: ReleasesFilters, limit: Long, offset: Long): List[ReleaseOverview] = {
    if (limit == 0 || isFiltersEmpty(filters)) {
      logger.trace("Filters would find nothing, returning empty list.")
      List.empty[ReleaseOverview]
    } else {
      val releasesTableColumns = Seq(
        s"$REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_ID_COLUMN",
        s"$REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_TITLE_COLUMN",
        s"$REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_STATUS_COLUMN",
        s"$REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_START_DATE_COLUMN",
        s"$REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_END_DATE_COLUMN",
        s"$REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_KIND",
        s"$REPORT_RELEASES_TABLE_ALIAS.$COMMON_RELEASE_OWNER_COLUMN",
      )

      val totalTasksSubQuery =
        s"""
           |(SELECT COUNT(1) FROM $REPORT_TASKS_TABLE_NAME tasks
           |  WHERE tasks.$REPORT_TASKS_RELEASEID_COLUMN = $REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_ID_COLUMN
           |) total
           |""".stripMargin

      val completedTasksSubQuery =
        s"""
           |(SELECT COUNT(1) FROM $REPORT_TASKS_TABLE_NAME tasks
           |  WHERE tasks.$REPORT_TASKS_RELEASEID_COLUMN = $REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_ID_COLUMN
           |    AND tasks.$REPORT_TASKS_STATUS_COLUMN in ('${TaskStatus.COMPLETED.value()}', '${TaskStatus.SKIPPED.value()}', '${TaskStatus.COMPLETED_IN_ADVANCE.value()}', '${TaskStatus.SKIPPED_IN_ADVANCE.value()}')
           |) completed
           |""".stripMargin

      val columns = releasesTableColumns ++ Seq(
        totalTasksSubQuery,
        completedTasksSubQuery
      )
      val relType = Type.valueOf(classOf[Release])
      val mapper: RowMapper[ReleaseOverview] = (rs: ResultSet, rowNum: Int) => {
        val overview = new ReleaseOverview()
        overview.setType(relType.toString)
        overview.setArchived(true)
        overview.setId(rs.getString(1))
        overview.setTitle(rs.getString(2))
        val status = ReleaseStatus.valueOf(rs.getString(3).toUpperCase)
        overview.setStatus(status)
        overview.setStartDate(rs.getTimestamp(4))
        overview.setEndDate(rs.getTimestamp(5))
        overview.setKind(ReleaseKind.valueOf(rs.getString(6).toUpperCase))
        overview.setOwner(rs.getString(7))
        overview.setRiskScore(0)
        val total = rs.getInt(8)
        val completed = rs.getInt(9)
        val calculatedProgress = if (total > 0) Math.round((completed * 100) / total) else 0
        overview.setCalculatedProgressPercent(calculatedProgress)
        overview.setTotalTasks(total)
        overview.setTotalRemainingTasks(total - completed)
        overview
      }
      val queryBuilder = searchQuery(filters, columns, Some(limit), Some(offset))
      val (sql, params) = queryBuilder.build()
      logger.debug(s"Fetching archived releases with query: $sql")
      jdbcTemplate.query(sql, params.toArray, mapper).asScala.toList
    }
  }

  def searchReleases(filters: ReleasesFilters, limit: Option[Long] = None, offset: Option[Long] = None): List[String] = {
    searchColumnsByReleasesFilters(filters, Seq(REPORT_RELEASES_RELEASEJSON_COLUMN), releaseContentMapper, limit, offset)
  }

  def searchReleasesByIds(releaseIds: Seq[String]): Seq[String] = {
    if (releaseIds.isEmpty) {
      Seq.empty
    } else {
      val sqlQuery = new SelectArchivedReleasesBuilder(REPORT_RELEASES_RELEASEJSON_COLUMN)
      val (sql, params) = sqlQuery.withReleaseIds(releaseIds).build()
      logger.debug(s"Fetching archived releases with query: $sql")
      jdbcTemplate.query(sql, params.toArray, releaseContentMapper).asScala.toList
    }
  }

  def searchReleasesBasicInfoByIds(releaseIds: Seq[String]): Seq[ReleaseBasicDataExt] = releaseIds match {
    case Nil => Seq.empty
    case ids => {
      val sqlQuery = new SelectArchivedReleasesBuilder(REPORT_RELEASES_ID_COLUMN, REPORT_RELEASES_TITLE_COLUMN,
        REPORT_RELEASES_STATUS_COLUMN, REPORT_RELEASES_START_DATE_COLUMN, REPORT_RELEASES_END_DATE_COLUMN)
      val (sql, params) = sqlQuery.withReleaseIds(ids).build()
      logger.debug(s"Fetching archived releases with query: $sql")
      jdbcTemplate.query(sql, params.toArray, (rs: ResultSet, _: Int) => ReleaseBasicDataExt(
        id = rs.getString(REPORT_RELEASES_ID_COLUMN),
        title = rs.getString(REPORT_RELEASES_TITLE_COLUMN),
        status = rs.getString(REPORT_RELEASES_STATUS_COLUMN),
        startDate = rs.getTimestamp(REPORT_RELEASES_START_DATE_COLUMN),
        endDate = rs.getTimestamp(REPORT_RELEASES_END_DATE_COLUMN)
      )).asScala.toSeq
    }
  }

  def searchReleaseIdsAndOrderCriterion(filters: Seq[ReportFilter],
                                        order: Option[ReleaseOrderMode],
                                        limit: Option[Long] = None,
                                        offset: Option[Long] = None
                                       ): List[(AdaptiveReleaseId, Any)] = {


    val sqlBuilder = new SelectArchivedReleasesBuilder(
      Seq(REPORT_RELEASES_ID_COLUMN, getOrderCriterionColumn(order)).filter(_ != null).map(col => s"$REPORT_RELEASES_TABLE_ALIAS.$col"): _*
    )
    filters.foreach(_.apply(sqlBuilder))
    order.foreach(o => applyOrder(sqlBuilder, o, Some(true)))
    applyLimits(sqlBuilder, limit, offset)

    val (sql, params) = sqlBuilder.build()
    logger.debug(s"Fetching archived releases with query: $sql")
    val mapper: RowMapper[(AdaptiveReleaseId, Any)] = (rs: ResultSet, _: Int) => {
      (
        ArchiveDbReleaseId(rs.getString(1)),
        order match {
          case None => null
          case Some(ReleaseOrderMode.start_date) |
               Some(ReleaseOrderMode.end_date) => rs.getTimestamp(2)
          case Some(ReleaseOrderMode.title) => rs.getString(2)
          case _ => throw new IllegalArgumentException(s"unable to sort using [${order.get}]")
        }
      )
    }
    jdbcTemplate.query(sql, params.toArray, mapper).asScala.toList
  }

  private def getOrderCriterionColumn(order: Option[ReleaseOrderMode]): String = order match {
    case None => null
    case Some(ReleaseOrderMode.start_date) => REPORT_RELEASES_START_DATE_COLUMN
    case Some(ReleaseOrderMode.end_date) => REPORT_RELEASES_END_DATE_COLUMN
    case Some(ReleaseOrderMode.title) => REPORT_RELEASES_TITLE_COLUMN
    case Some(ReleaseOrderMode.risk) => throw new IllegalStateException("Ordering by risk is not possible in archived db")
  }

  private def searchColumnsByReleasesFilters[T](filters: ReleasesFilters,
                                                columns: Seq[String],
                                                mapper: (ResultSet, Int) => T,
                                                limit: Option[Long] = None,
                                                offset: Option[Long] = None
                                               ): List[T] = {
    if (limit.contains(0) || isFiltersEmpty(filters)) {
      logger.trace("Filters would find nothing, returning empty list.")
      List.empty[T]
    } else {
      val (sql, params) = searchQuery(filters, columns, limit, offset).build()
      logger.debug(s"Fetching archived releases with query: $sql")
      jdbcTemplate.query(sql, params.toArray, mapper).asScala.toList
    }
  }

  def countReleasesByStatus(filters: ReleasesFilters): Map[ReleaseStatus, Int] = {
    if (isFiltersEmpty(filters)) {
      logger.trace("Filters would find nothing, returning empty map.")
      Map.empty[ReleaseStatus, Int]
    } else {
      val (sql, params) = countQuery(filters).build()
      logger.debug(s"Fetching archived releases with query: $sql")
      jdbcTemplate.query(sql, params.toArray, countByStatusMapper).asScala.toMap
    }
  }

  def countReleasesByStatus(filters: Seq[ReportFilter]): Map[ReleaseStatus, Int] = {
    val sqlBuilder = new SelectArchivedReleasesBuilder(
      s"$REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_STATUS_COLUMN",
      s"COUNT($REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_ID_COLUMN)"
    )
    filters.foreach(_.apply(sqlBuilder))
    sqlBuilder.withPreArchived(false)
    sqlBuilder.groupBy(s"$REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_STATUS_COLUMN")

    val (sql, params) = sqlBuilder.build()
    jdbcTemplate.query(sql, params.toArray, countByStatusMapper).asScala.toMap
  }

  private def isFiltersEmpty(filters: ReleasesFilters): Boolean =
    filters.hasASelectedFilter && (!filters.withOnlyArchived() && !filters.getStatuses.asScala.exists(s => s == COMPLETED || s == ABORTED))

  private def searchQuery(filters: ReleasesFilters,
                          columns: Seq[String],
                          limit: Option[Long] = None,
                          offset: Option[Long] = None): SelectArchivedReleasesBuilder = {
    val sqlQuery = buildQuery(filters)(columns: _*)
    applyOrder(sqlQuery, filters.getOrderBy, None)
    applyLimits(sqlQuery, limit, offset)
    sqlQuery
  }

  private def applyOrder(sqlBuilder: SelectArchivedReleasesBuilder, order: ReleaseOrderMode, forceAscOrDesc: Option[Boolean]): SelectArchivedReleasesBuilder = {
    order match {
      case ReleaseOrderMode.end_date | ReleaseOrderMode.risk | null =>
        sqlBuilder.orderBy(s"$REPORT_RELEASES_END_DATE_COLUMN ${getOrderAscOrDesc(forceAscOrDesc, default = false)}, $REPORT_RELEASES_ID_COLUMN ASC")
      case ReleaseOrderMode.start_date =>
        sqlBuilder.orderBy(s"$REPORT_PHASES_START_DATE_COLUMN ${getOrderAscOrDesc(forceAscOrDesc, default = false)}, $REPORT_RELEASES_ID_COLUMN ASC")
      case ReleaseOrderMode.title =>
        sqlBuilder.orderBy(s"$REPORT_RELEASES_TITLE_COLUMN ${getOrderAscOrDesc(forceAscOrDesc, default = true)}")
    }
    sqlBuilder
  }

  private def applyLimits(sqlBuilder: SelectArchivedReleasesBuilder,
                          limit: Option[Long] = None,
                          offset: Option[Long] = None): SelectArchivedReleasesBuilder = {
    limit.foreach(l =>
      offset.fold(sqlBuilder.limit(l))(o => sqlBuilder.limitAndOffset(l, o))
    )
    sqlBuilder
  }

  private def countQuery(filters: ReleasesFilters): SelectArchivedReleasesBuilder = {
    buildQuery(filters)(
      s"$REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_STATUS_COLUMN",
      s"COUNT($REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_ID_COLUMN)"
    ).groupBy(s"$REPORT_RELEASES_TABLE_ALIAS.$REPORT_RELEASES_STATUS_COLUMN")
  }

  private def buildQuery(filters: ReleasesFilters)(columns: String*): SelectArchivedReleasesBuilder = {
    val startDate = filters.getQueryStartDate
    val endDate = filters.getQueryEndDate

    val sqlQuery = new SelectArchivedReleasesBuilder(columns: _*)
    sqlQuery
      .withTitle(filters.getTitle)
      .withKind(filters.getKind)
      .withStartDate(startDate)
      .withEndDate(endDate)
      .withTags(filters.getTags)

    if (filters.getKind == ReleaseKind.RELEASE) {
      sqlQuery.withSecurity(userSpecific = true)
    } else {
      sqlQuery.withSecurityOrOwner(userSpecific = true)
    }

    sqlQuery.withPreArchived(false)

    if (filters.hasASelectedFilter) {
      sqlQuery.withOneOfStatuses(filters.getStatuses.asScala.filter(s => s == COMPLETED || s == ABORTED).map(_.value()).toSeq: _*)
    }

    if (filters.getParentId != null) {
      sqlQuery.withFolder(filters.getParentId)
    }

    if (filters.getTemplateId != null) {
      sqlQuery.withOriginTemplateId(filters.getTemplateId)
    }

    if (filters.withOnlyMine()) {
      sqlQuery.withOwner(Permissions.getAuthenticatedUserName)
    }
    if (filters.withOnlyFlagged()) {
      sqlQuery.withNonOkFlag()
    }
    sqlQuery
  }

  private val countByStatusMapper: RowMapper[(ReleaseStatus, Int)] =
    (rs, _) =>
      ReleaseStatus.valueOf(rs.getString(1).toUpperCase) -> rs.getInt(2)

}
