package com.xebialabs.xlrelease.domain.utils

import com.xebialabs.xlrelease.repository.Ids

sealed trait AdaptiveReleaseId {
  def withOnlyOneParentOrApplicationsForArchiveDb(): String

  def folderlessReleaseId(): String

  def fullReleaseIdWithApplications(): String

  def fullReleaseIdWithoutApplications(): String

  def combineWith(adaptiveReleaseId: AdaptiveReleaseId): AdaptiveReleaseId

  protected def impossible(): String = {
    throw new IllegalArgumentException(s"[${this.getClass.getSimpleName}] " +
      s"does not have enough data to do what you want")
  }

  protected def ensureSame[T](a: T, b: T): Unit = {
    if (!a.equals(b)) {
      throw new IllegalArgumentException(s"Unable to combine two release ids: " +
        s"[$a] and [$b] does not match ")
    }
  }

  protected def combineFolderlessWithArchive(a: FolderlessReleaseId, b: ArchiveDbReleaseId): ArchiveDbReleaseId = {
    ensureSame(a.folderlessReleaseId(), b.folderlessReleaseId())
    b
  }

  protected def combineFolderlessWithFull(a: FolderlessReleaseId, b: FullReleaseId): FullReleaseId = {
    ensureSame(a.folderlessReleaseId(), b.folderlessReleaseId())
    b
  }

  protected def combineArchiveWithFull(a: ArchiveDbReleaseId, b: FullReleaseId): FullReleaseId = {
    ensureSame(a.folderlessReleaseId(), b.folderlessReleaseId())
    ensureSame(a.parentFolderFolderlessId, b.foldersFolderlessIds.lastOption)
    b
  }
}


case class FolderlessReleaseId private(
                                id: String
                              ) extends AdaptiveReleaseId {
  override def withOnlyOneParentOrApplicationsForArchiveDb(): String = impossible()

  override def folderlessReleaseId(): String = id

  override def fullReleaseIdWithApplications(): String = impossible()

  override def fullReleaseIdWithoutApplications(): String = impossible()

  override def combineWith(other: AdaptiveReleaseId): AdaptiveReleaseId = other match {
    case folderlessReleaseId: FolderlessReleaseId =>
      ensureSame(folderlessReleaseId.folderlessReleaseId(), id)
      this
    case archiveDbReleaseId: ArchiveDbReleaseId =>
      combineFolderlessWithArchive(this, archiveDbReleaseId)
    case fullReleaseId: FullReleaseId =>
      combineFolderlessWithFull(this, fullReleaseId)
  }
}

object FolderlessReleaseId {
  def apply(id: String) : FolderlessReleaseId = {
    // validate
    val pc = id.split(Ids.SEPARATOR).toSeq.filter(_.trim().nonEmpty)
    if (pc.length != 1) {
      throw new IllegalArgumentException(s"Id [$id] is not suitable for FolderlessReleaseId")
    }
    new FolderlessReleaseId(pc.head)
  }
}

case class ArchiveDbReleaseId private(
                               releaseFolderlessId: String,
                               parentFolderFolderlessId: Option[String]
                             ) extends AdaptiveReleaseId {
  override def withOnlyOneParentOrApplicationsForArchiveDb(): String = {
    parentFolderFolderlessId
      .orElse(Some(Ids.ROOT_FOLDER_ID))
      .map(folderId => s"$folderId${Ids.SEPARATOR}$releaseFolderlessId")
      .get
  }

  override def folderlessReleaseId(): String = releaseFolderlessId

  override def fullReleaseIdWithApplications(): String = impossible()

  override def fullReleaseIdWithoutApplications(): String = impossible()

  override def combineWith(other: AdaptiveReleaseId): AdaptiveReleaseId = other match {
    case folderlessReleaseId: FolderlessReleaseId =>
      ensureSame(folderlessReleaseId.folderlessReleaseId(), releaseFolderlessId)
      this
    case archiveDbReleaseId: ArchiveDbReleaseId =>
      ensureSame(this, archiveDbReleaseId)
      this
    case fullReleaseId: FullReleaseId =>
      combineArchiveWithFull(this, fullReleaseId)
  }
}

object ArchiveDbReleaseId {
  def apply(id: String): ArchiveDbReleaseId = {
    val chunks = id.split(Ids.SEPARATOR).toSeq.filter(_.trim.nonEmpty)
    if (chunks.length == 1) {
      ArchiveDbReleaseId(chunks.head, None)
    } else if (chunks.length == 2) {
      if (chunks.head.equals(Ids.ROOT_FOLDER_ID)) {
        ArchiveDbReleaseId(chunks.last, None)
      } else {
        ArchiveDbReleaseId(chunks.last, Some(chunks.head))
      }
    } else {
      throw new IllegalArgumentException(s"Release ID [$id] does not look like archive db release id");
    }
  }
}

case class FullReleaseId private(
                          releaseFolderlessId: String,
                          foldersFolderlessIds: Seq[String]
                        ) extends AdaptiveReleaseId {
  override def withOnlyOneParentOrApplicationsForArchiveDb(): String = {
    ArchiveDbReleaseId(releaseFolderlessId, foldersFolderlessIds.lastOption)
      .withOnlyOneParentOrApplicationsForArchiveDb()
  }

  override def folderlessReleaseId(): String = releaseFolderlessId

  override def fullReleaseIdWithApplications(): String =
    (Seq(Ids.ROOT_FOLDER_ID) ++
      foldersFolderlessIds ++
      Seq(releaseFolderlessId))
      .mkString(Ids.SEPARATOR)

  override def fullReleaseIdWithoutApplications(): String =
    (foldersFolderlessIds ++
      Seq(releaseFolderlessId))
      .mkString(Ids.SEPARATOR)

  override def combineWith(other: AdaptiveReleaseId): AdaptiveReleaseId = other match {
    case folderlessReleaseId: FolderlessReleaseId =>
      combineFolderlessWithFull(folderlessReleaseId, this)
    case archiveDbReleaseId: ArchiveDbReleaseId =>
      combineArchiveWithFull(archiveDbReleaseId, this)
    case fullReleaseId: FullReleaseId =>
      ensureSame(fullReleaseId, this)
      this
  }
}

object FullReleaseId {
  def apply(fullReleaseId: String): FullReleaseId = {
    val chunks = fullReleaseId
      .split(Ids.SEPARATOR)
      .toSeq
      .filter(_.trim.nonEmpty)
      .zipWithIndex
      .filter(t => t._2 > 0 || !t._1.equals(Ids.ROOT_FOLDER_ID))
      .map(_._1)

    new FullReleaseId(chunks.last, chunks.slice(0, chunks.length - 1))
  }

  def apply(folderlessReleaseId: String, folderId: String): FullReleaseId = {
    val chunks = folderId
      .split(Ids.SEPARATOR)
      .toSeq
      .filter(_.trim.nonEmpty)
      .zipWithIndex
      .filter(t => t._2 > 0 || !t._1.equals(Ids.ROOT_FOLDER_ID))
      .map(_._1)

    new FullReleaseId(folderlessReleaseId, chunks)
  }
}