package com.xebialabs.xlrelease.domain.validators

import com.xebialabs.deployit.booter.local.utils.Strings.isBlank
import com.xebialabs.deployit.checks.Checks.{IncorrectArgumentException, checkArgument, checkNotNull}
import com.xebialabs.xlplatform.security.dto.PasswordValidationResult
import com.xebialabs.xlplatform.security.validator.{Invalid, StringValidator, Valid}
import com.xebialabs.xlrelease.api.v1.forms.UserAccount
import jakarta.mail.internet.{AddressException, InternetAddress}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util
import java.util.regex.Pattern
import scala.jdk.CollectionConverters._

object UserAccountValidator {
  def isValidEmailAddress(email: String): Boolean = {
    try {
      val emailAddr = new InternetAddress(email)
      emailAddr.validate()
      true
    } catch {
      case ex: AddressException =>
        false
    }
  }
}

@Component
class UserAccountValidator @Autowired()(val passwordValidator: StringValidator) {
  def check(userAccount: UserAccount): Unit = {
    checkNotNull(userAccount, "User Account can not be null.", "")
    checkUsername(userAccount)
    checkPassword(userAccount)
  }

  def checkUsername(userAccount: UserAccount): Unit = {
    checkArgument(!isBlank(userAccount.getUsername), "User name cannot be empty.")
    if (!userAccount.isExternal) {
      val pattern = Pattern.compile("^(?![\\.\\-_])(?!.*[\\.\\-_]{2})[\\w\\.\\-_]*[^\\W_]$", Pattern.UNICODE_CHARACTER_CLASS)
      val matcher = pattern.matcher(userAccount.getUsername)
      checkArgument(matcher.matches, "All the alphanumeric characters and the special characters dash (-), underscore (_), and dot (.) are allowed. The special characters cannot be repeated more than once in a row. The username must not start or end with a special character.")
    }
  }

  def checkPassword(userAccount: UserAccount): Unit = {
    checkArgument(!isBlank(userAccount.getPassword), "Password is required when creating a user.")
    if (!userAccount.isExternal) checkPassword(userAccount.getPassword)
  }

  def checkPassword(password: String): Unit = {
    val invalids = passwordValidator.validate(password).filter(_.isInstanceOf[Invalid])
    if (invalids.nonEmpty) {
      throw new IncorrectArgumentException((List("Password needs to meet following requirements:") ++ invalids.map(_.message)).mkString(System.lineSeparator + "\t"))
    }
  }

  def validatePassword(password: String): util.List[PasswordValidationResult] = passwordValidator
    .validate(password)
    .map(vr => new PasswordValidationResult(vr.isInstanceOf[Valid], vr.message))
    .sortBy(_.isValid)(Ordering.Boolean.reverse)
    .asJava
}
