package com.xebialabs.xlrelease.export

import com.google.common.base.Strings
import com.xebialabs.deployit.checks.Checks._
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plumbing.export.ImportTemplateConfigurationItemReaderWriter
import com.xebialabs.deployit.plumbing.export.UnresolvedReferencesConfigurationItemConverter.DECRYPTION_FAILED_VALUE
import com.xebialabs.deployit.plumbing.serialization.ResolutionContext
import com.xebialabs.deployit.repository.{WorkDirContext, WorkDirFactory}
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.xlrelease.`export`.ImportType.ImportType
import com.xebialabs.xlrelease.domain.{Attachment, Release, ReleaseTrigger, TemplateLogo}
import com.xebialabs.xlrelease.export.TemplateExporter.{DATA_MODEL_VERSION_FIELD_NAME, MANIFEST_JSON_FILE, RELEASE_TEMPLATE_JSON_FILE, XLR_VERSION_FIELD_NAME}
import com.xebialabs.xlrelease.export.TemplateImporter._
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.RetryTitleGenerator.getNextTitle
import com.xebialabs.xlrelease.service.{ReleaseService, TaskAccessService}
import com.xebialabs.xlrelease.utils.CiHelper.getNestedCis
import com.xebialabs.xlrelease.views.ImportResult
import jakarta.ws.rs.core.MediaType.APPLICATION_JSON_TYPE
import org.apache.commons.io.{FilenameUtils, IOUtils}
import org.codehaus.jettison.json.{JSONArray, JSONObject}
import org.springframework.stereotype.{Component, Service}

import java.io._
import java.util
import java.util.Optional
import java.util.regex.Pattern
import java.util.regex.Pattern.compile
import java.util.zip.ZipInputStream
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.Using

object ImportType extends Enumeration {
  type ImportType = Value
  val Json: ImportType = Value(".json")
  val Xlr: ImportType = Value(".xlr")
  val Yaml: ImportType = Value(".yaml")
  val Zip: ImportType = Value(".zip")
}

case class TemplateImportContext(destinationFolderId: String,
                                 importType: ImportType,
                                 isDefaultTemplate: Boolean = false,
                                 version: String = XLR_3_0_0_VERSION)

object TemplateImportContext {
  final val DEFAULT_VERSION = XLR_3_0_0_VERSION

  def toImportType(extension: String): ImportType = {
    ImportType.values.find(_.toString == extension).getOrElse {
      throw new IllegalArgumentException(s"Unsupported file format: $extension. " +
        s"Supported formats include Template (Release/xlr), Releasefile (Groovy/zip), and Template as-code (YAML/zip or YAML/yaml).")
    }
  }
}

@Service
class TemplateImporter(maybeTemplateImporters: Optional[util.List[BaseTemplateImporter[_]]]) {

  lazy val importers: List[BaseTemplateImporter[_]] = maybeTemplateImporters.orElse(util.Collections.emptyList[BaseTemplateImporter[_]]()).asScala.toList

  def importTemplate(inputStream: InputStream, importContext: TemplateImportContext): util.List[ImportResult] = {
    val previousWorkDir = Option(WorkDirContext.get())
    WorkDirContext.initWorkdir(WorkDirFactory.IMPORT_WORKDIR_PREFIX)
    try {
      importers.find(_.supports(importContext)).map(_.importTemplate(inputStream, importContext: TemplateImportContext)).getOrElse(
        throw new IllegalStateException(s"Unable to import a template for the following context: $importContext. Maybe some template importers are missing?")
      )
    } finally {
      Option(WorkDirContext.get).foreach(_.delete())
      WorkDirContext.clear()
      previousWorkDir.foreach(WorkDirContext.setWorkDir)
    }
  }

  // backwards compatibility methods:
  def importReleaseStream(inputStream: InputStream, importType: ImportType, destinationFolderId: String, version: String): util.List[ImportResult] = {
    val importVersion = if (null == version) TemplateImportContext.DEFAULT_VERSION else version
    val importContext = TemplateImportContext(destinationFolderId, importType, version = importVersion)
    importTemplate(inputStream, importContext)
  }

  def importDefaultTemplate(inputStream: InputStream): util.List[ImportResult] = {
    importTemplate(inputStream, TemplateImportContext(null, ImportType.Json, isDefaultTemplate = true))
  }

  // methods used in tests:
  def importZippedTemplate(inputStream: InputStream): util.List[ImportResult] = {
    importZippedTemplate(inputStream, null)
  }

  def importZippedTemplate(inputStream: InputStream, destinationFolderId: String): util.List[ImportResult] = {
    importTemplate(inputStream, TemplateImportContext(destinationFolderId, ImportType.Xlr))
  }

  def importReleaseStream(inputStream: InputStream, importType: ImportType, destinationFolderId: String): util.List[ImportResult] = {
    importReleaseStream(inputStream, importType, destinationFolderId, null)
  }

  def importReleaseStream(inputStream: InputStream, importType: ImportType): util.List[ImportResult] = {
    importReleaseStream(inputStream, importType, null)
  }

  def importJsonTemplate(inputStream: InputStream): util.List[ImportResult] = {
    importJsonTemplate(inputStream, null)
  }

  def importJsonTemplate(inputStream: InputStream, destinationFolderId: String): util.List[ImportResult] = {
    importTemplate(inputStream, TemplateImportContext(destinationFolderId, ImportType.Json))
  }

}

object TemplateImporter {
  final val FILE_NAME_PATTERN: Pattern = compile("(.*)/[A-Za-z0-9]*-(.*)")
  final val XLR_3_0_0_VERSION = "3.0.0"
  final val BUFFER_SIZE = 8 * 1024
}

abstract class BaseTemplateImporter[T <: TemplateContent](taskAccessService: TaskAccessService,
                                                          releaseService: ReleaseService,
                                                          maybeTemplateProcessors: Optional[util.List[TemplateImportProcessor]]) {

  private val templateProcessors: util.List[TemplateImportProcessor] = maybeTemplateProcessors.orElse(util.Collections.emptyList[TemplateImportProcessor]())


  def supports[U <: TemplateImportContext](importContext: U): Boolean

  protected def doReadTemplateContent(inputStream: InputStream, importContext: TemplateImportContext): T

  protected def doUpgradeTemplateContent(templateContent: T): util.List[String]

  private[export] def doConvertToTemplate(templateContent: T, destinationFolderId: String): Release

  def importTemplate(inputStream: InputStream, importContext: TemplateImportContext): util.List[ImportResult] = {
    val templateContent: T = doReadTemplateContent(inputStream, importContext: TemplateImportContext)
    val upgradeWarnings: util.List[String] = doUpgradeTemplateContent(templateContent)
    importTemplateContent(templateContent, importContext.destinationFolderId, upgradeWarnings)
  }

  private def importTemplateContent(templateContent: T, destinationFolderId: String, warnings: util.List[String]): util.List[ImportResult] = {
    val destinationId = if (destinationFolderId == null) Ids.ROOT_FOLDER_ID else destinationFolderId

    val template: Release = doConvertToTemplate(templateContent, destinationId)
    template.getAttachments.forEach(attachment => {
      attachment.setFile(templateContent.localFiles.get(attachment.getExportFilename))
    })
    Option(template.getLogo).foreach { logo =>
      logo.setFile(templateContent.localFiles.get(logo.getPortableFilename))
    }

    checkArgument(template.isTemplate, "Only templates can be imported")
    taskAccessService.checkIfAuthenticatedUserCanUseTasks(template.getAllTasks)

    templateProcessors.forEach(processor => warnings.addAll(processor.process(template, destinationId)))

    removeMissingGateDependencies(template, warnings)
    removeTriggersForRootFolder(destinationId, template, warnings)
    resetUndecryptedPasswords(template, warnings)

    while (templateExistsWithTitle(destinationId, template.getTitle)) template.setTitle(getNextTitle(template.getTitle))
    if (!template.hasScheduledStartDate) template.setScheduledStartDate(template.findFirstSetDate)

    releaseService.importTemplate(template, destinationId)

    util.Arrays.asList(new ImportResult(template.getId, template.getTitle, warnings))
  }

  private def resetUndecryptedPasswords(template: Release, warnings: util.Collection[String]): Unit = {
    var issueWarning = false
    val configurationItems = getNestedCis(template)
    configurationItems.forEach { configurationItem =>
      val propertyDescriptors = configurationItem.getType.getDescriptor.getPropertyDescriptors
      for (propertyDescriptor <- propertyDescriptors.asScala) {
        val decryptionFailed = resetUndecryptedPassword(configurationItem, propertyDescriptor)
        if (decryptionFailed) issueWarning = true
      }
    }
    if (issueWarning) warnings.add("Passwords could not be decrypted and have been reset.")
  }

  private def resetUndecryptedPassword(ci: ConfigurationItem, propertyDescriptor: PropertyDescriptor): Boolean = {
    if ((propertyDescriptor.getKind == PropertyKind.STRING) && propertyDescriptor.isPassword) {
      val passwordValue = propertyDescriptor.get(ci).asInstanceOf[String]
      if (passwordValue == DECRYPTION_FAILED_VALUE) {
        propertyDescriptor.set(ci, null)
        return true
      }
    }
    false
  }

  private def removeMissingGateDependencies(template: Release, warnings: util.Collection[String]): Unit = {
    template.getAllGates.forEach { gateTask =>
      gateTask.getDependencies.removeIf { dependency =>
        val shouldBeRemoved = !dependency.hasVariableTarget && !dependency.hasResolvedTarget && !dependency.isArchived
        if (shouldBeRemoved) warnings.add(String.format("Gate Task '%s' has a dependency on an unknown Release/Phase/Task", gateTask.getTitle))
        shouldBeRemoved
      }
    }
  }

  private def removeTriggersForRootFolder(destinationId: String, template: Release, warnings: util.Collection[String]): Unit = {
    if (Ids.isRoot(destinationId) && !template.getReleaseTriggers.isEmpty) {
      template.setReleaseTriggers(new util.ArrayList[ReleaseTrigger]())
      warnings.add("Triggers could not be imported on the root folder.")
    }
  }

  private def templateExistsWithTitle(folderId: String, title: String): Boolean = releaseService.templateExistsWithTitle(folderId, title)

}

@Component
class JsonTemplateImporter(templateImportUpgrader: TemplateImportUpgrader,
                           taskAccessService: TaskAccessService,
                           releaseService: ReleaseService,
                           importTemplateConfigurationItemReaderWriter: ImportTemplateConfigurationItemReaderWriter,
                           templateProcessors: Optional[util.List[TemplateImportProcessor]])
  extends BaseTemplateImporter[JsonTemplateContent](taskAccessService, releaseService, templateProcessors) {

  override def supports[U <: TemplateImportContext](importContext: U): Boolean = {
    !importContext.isDefaultTemplate && importContext.importType == ImportType.Json
  }

  override def doConvertToTemplate(templateContent: JsonTemplateContent, destinationFolderId: String): Release = {
    val resolutionContext = ResolutionContext(destinationFolderId)
    val singleReleaseStream = new ByteArrayInputStream(templateContent.template.toString.getBytes)
    val configurationItem = importTemplateConfigurationItemReaderWriter.readFrom(resolutionContext, APPLICATION_JSON_TYPE, singleReleaseStream)
    checkArgument(configurationItem.isInstanceOf[Release], "Template file doesn't contain a Release")
    configurationItem.asInstanceOf[Release]
  }

  override def doReadTemplateContent(inputStream: InputStream, importContext: TemplateImportContext): JsonTemplateContent = {
    val jsonString = readJson(inputStream)
    val jsonArray = new JSONArray(jsonString)
    checkArgument(jsonArray.length >= 1, "Template file is empty")
    checkArgument(jsonArray.get(0).isInstanceOf[JSONObject], "Invalid json file")

    val localFiles = new util.HashMap[String, LocalFile]
    val version = Option(importContext.version).getOrElse(XLR_3_0_0_VERSION)
    JsonTemplateContent(jsonArray.get(0).asInstanceOf[JSONObject], version, localFiles)
  }

  override def doUpgradeTemplateContent(templateContent: JsonTemplateContent): util.List[String] = {
    templateImportUpgrader.upgrade(templateContent.template, templateContent.version).getWarnings
  }

  @throws[IOException]
  private[export] def readJson(inputStream: InputStream): String = {
    val outputStream = new ByteArrayOutputStream
    IOUtils.copy(inputStream, outputStream, BUFFER_SIZE)
    new String(outputStream.toByteArray)
  }
}

@Component
class XlrJsonTemplateImporter(templateImportUpgrader: TemplateImportUpgrader,
                              taskAccessService: TaskAccessService,
                              releaseService: ReleaseService,
                              templateProcessors: Optional[util.List[TemplateImportProcessor]],
                              importTemplateConfigurationItemReaderWriter: ImportTemplateConfigurationItemReaderWriter)
  extends JsonTemplateImporter(templateImportUpgrader, taskAccessService, releaseService, importTemplateConfigurationItemReaderWriter, templateProcessors) {

  override def supports[U <: TemplateImportContext](importContext: U): Boolean = {
    importContext.importType == ImportType.Xlr
  }

  override def doReadTemplateContent(inputStream: InputStream, importContext: TemplateImportContext): JsonTemplateContent = {
    val workDir = WorkDirContext.get
    var releaseJson: JSONObject = null
    var version: String = null
    val localFiles = new util.HashMap[String, LocalFile]

    def readArtifact(stream: ZipInputStream, entryName: String, fileName: String): Unit = {
      val localFile = workDir.newFile(fileName)
      val outputStream = localFile.getOutputStream
      IOUtils.copy(stream, outputStream, BUFFER_SIZE)
      outputStream.close()
      localFiles.put(entryName, localFile)
    }

    Using.resource(new ZipInputStream(inputStream)) { stream =>
      for (entry <- LazyList.continually(stream.getNextEntry).takeWhile(_ != null)) {
        val entryName = entry.getName

        if (entryName == RELEASE_TEMPLATE_JSON_FILE) {
          releaseJson = new JSONObject(readJson(stream))
        }
        else if (entryName.startsWith(Attachment.EXPORT_DIRECTORY)) {
          val fileName = extractFileName(entryName)
          readArtifact(stream, entryName, fileName)
        } else if (entryName.startsWith(TemplateLogo.EXPORT_DIRECTORY)) {
          val fileName = entryName.split("/").last
          readArtifact(stream, entryName, fileName)
        } else if (entryName == MANIFEST_JSON_FILE) {
          val manifestJson = new JSONObject(readJson(stream))
          version = if (manifestJson.has(DATA_MODEL_VERSION_FIELD_NAME)) {
            manifestJson.getString(DATA_MODEL_VERSION_FIELD_NAME)
          } else {
            manifestJson.getString(XLR_VERSION_FIELD_NAME)
          }
        }

        stream.closeEntry()
      }
    }

    checkArgument(releaseJson != null, "Missing file in template export : " + RELEASE_TEMPLATE_JSON_FILE)
    checkArgument(version != null, "Missing file in template export : " + MANIFEST_JSON_FILE)
    JsonTemplateContent(releaseJson, version, localFiles)
  }

  private def extractFileName(entryName: String): String = {
    val matcher = FILE_NAME_PATTERN.matcher(entryName)
    if (matcher.find) {
      matcher.group(2)
    }
    else {
      throw new RuntimeException("File name not found in : '" + entryName + "'")
    }
  }

}

@Component
class DefaultTemplateImporter(templateImportUpgrader: TemplateImportUpgrader,
                              taskAccessService: TaskAccessService,
                              releaseService: ReleaseService,
                              templateProcessors: Optional[util.List[TemplateImportProcessor]],
                              importTemplateConfigurationItemReaderWriter: ImportTemplateConfigurationItemReaderWriter)
  extends JsonTemplateImporter(templateImportUpgrader, taskAccessService, releaseService, importTemplateConfigurationItemReaderWriter, templateProcessors) {

  override def supports[U <: TemplateImportContext](importContext: U): Boolean = {
    importContext.isDefaultTemplate && importContext.importType == ImportType.Json
  }

  override def doUpgradeTemplateContent(templateContent: JsonTemplateContent): util.List[String] = {
    // skip upgrade of default templates
    new util.ArrayList[String]()
  }
}

trait TemplateContent {
  def localFiles: util.Map[String, LocalFile]
}

case class JsonTemplateContent(template: JSONObject, version: String, localFiles: util.Map[String, LocalFile]) extends TemplateContent


trait UnzippedTemplateImporter[T <: TemplateContent] {

  def name(): String

  def requiredFileName(): String

  def supports(filesInsideZip: Seq[String]): Boolean = {
    filesInsideZip.contains(requiredFileName())
  }

  def doReadTemplateContent(zipEntries: Map[String, LocalFile], importContext: TemplateImportContext): T

  def doUpgradeTemplateContent(templateContent: T): util.List[String]

  def doConvertToTemplate(templateContent: T, destinationFolderId: String): Release
}

@Component
class ZipTemplateImporter(taskAccessService: TaskAccessService,
                          releaseService: ReleaseService,
                          templateProcessors: Optional[util.List[TemplateImportProcessor]],
                          unzippedTemplateImportersDelegates: util.List[UnzippedTemplateImporter[_]])
  extends BaseTemplateImporter[TemplateContent](taskAccessService, releaseService, templateProcessors) {

  override def supports[U <: TemplateImportContext](importContext: U): Boolean = {
    importContext.importType == ImportType.Zip
  }

  private def findUnzipTemplateImporter(filesInsideZip: Seq[String]): UnzippedTemplateImporter[TemplateContent] = {
    unzippedTemplateImportersDelegates.asScala.find(_.supports(filesInsideZip)) match {
      case Some(unzippedTemplateImporter) => unzippedTemplateImporter.asInstanceOf[UnzippedTemplateImporter[TemplateContent]]
      case _ => throw new IllegalStateException(buildErrorMessage())
    }
  }

  override protected def doReadTemplateContent(inputStream: InputStream, importContext: TemplateImportContext): TemplateContent = {
    val files = toZipEntryMap(inputStream)
    findUnzipTemplateImporter(files.keySet.toSeq).doReadTemplateContent(files, importContext)
  }

  override protected def doUpgradeTemplateContent(templateContent: TemplateContent): util.List[String] = {
    findUnzipTemplateImporter(templateContent.localFiles.keySet.asScala.toSeq).doUpgradeTemplateContent(templateContent)
  }

  override private[export] def doConvertToTemplate(templateContent: TemplateContent, destinationFolderId: String): Release = {
    findUnzipTemplateImporter(templateContent.localFiles.keySet.asScala.toSeq).doConvertToTemplate(templateContent, destinationFolderId)
  }

  private def buildErrorMessage(): String = {
    val importersErrors = unzippedTemplateImportersDelegates.asScala.map { importer =>
      importer.name() + ": " + importer.requiredFileName()
    }.mkString(", ")
    s"""
       |The ZIP file does not not contain any of the supported files for the following template importers:
       |$importersErrors.
       |""".stripMargin

  }

  private def toZipEntryMap(inputStream: InputStream): Map[String, LocalFile] = {
    val acc: mutable.Map[String, LocalFile] = new mutable.HashMap[String, LocalFile]()
    val workDir = WorkDirContext.get
    Using.resource(new ZipInputStream(inputStream)) { stream =>
      for (entry <- LazyList.continually(stream.getNextEntry).takeWhile(_ != null)) {
        // check that file names are correct
        val fileName = extractFileName(entry.getName)
        if (!Strings.isNullOrEmpty(fileName)) {
          val localFile = workDir.newFile(fileName)
          Using.resource(localFile.getOutputStream) { out =>
            IOUtils.copy(stream, out, TemplateImporter.BUFFER_SIZE)
          }
          acc += (entry.getName -> localFile)
        }
        stream.closeEntry()
      }
    }
    acc.toMap
  }

  private def extractFileName(entryName: String): String = {
    FilenameUtils.getName(entryName)
  }
}
