package com.xebialabs.xlrelease.repository

import com.xebialabs.xlrelease.domain.{Release, ReleaseExtension}
import com.xebialabs.xlrelease.repository.Ids.{SEPARATOR, getName}
import com.xebialabs.xlrelease.repository.ReleaseExtensionsRepository._
import grizzled.slf4j.Logging

import scala.collection.mutable
import scala.jdk.CollectionConverters._

class ReleaseExtensionsRepository(generic: GenericReleaseExtensionRepository) extends Logging {

  protected val genericExtensions: mutable.Set[String] = mutable.Set.empty
  protected val specialized: mutable.Map[String, ReleaseExtensionSpecializedRepository[_ <: ReleaseExtension]] = mutable.Map.empty

  protected def supportedExtensions: Set[String] = genericExtensions.toSet ++ specialized.keySet

  def registerGeneric[T <: ReleaseExtension](name: String): Unit = {
    logger.info(s"Registering generic ReleaseExtension: /$name")
    genericExtensions += name
  }

  def register[T <: ReleaseExtension, R <: ReleaseExtensionSpecializedRepository[T]](specializedExtensionRepository: R): Unit = {
    import specializedExtensionRepository.name
    logger.info(s"Registering ReleaseExtension: /$name")
    specialized += (name -> specializedExtensionRepository)
  }

  def exists(extensionId: String): Boolean = {
    logger.debug(s"exists($extensionId)")
    dispatch(extensionId)(_.exists)
  }

  def read[T <: ReleaseExtension](extensionId: String): Option[T] = {
    logger.debug(s"read($extensionId)")
    dispatch(extensionId)(_.read).map(_.asInstanceOf[T])
  }

  def create[T <: ReleaseExtension](extension: T): T = {
    logger.debug(s"create(${extension.getId}, [${extension.getClass.getSimpleName}])")
    dispatch(extension)(repo => repo.create(extension.asInstanceOf[repo.ITEM])).asInstanceOf[T]
  }

  def createAll(release: Release, extensionsIds: Set[String]): Int = {
    logger.debug(s"createAll(${release.getId}) [${release.getExtensions.size} extensions]")

    val extensionsNames = extensionsIds.map(getName)
    val genericNames = genericExtensions.toSet intersect extensionsNames
    val specializedNames = (extensionsNames diff genericNames) intersect specialized.keySet
    val releaseExtensions = release.getExtensions.asScala.map(ext => getName(ext.getId) -> ext).toMap

    val releaseId = release.getId
    val fullId: String => String = releaseId + Ids.SEPARATOR + _

    val genericCreated = generic.createAll(release, genericNames.map(fullId))
    val specializedCreated = specializedNames.map { name =>
      releaseExtensions.get(name) match {
        case Some(ext) =>
          create(ext)
          1
        case None => 0
      }
    }.sum
    genericCreated + specializedCreated
  }

  def update(extension: ReleaseExtension): Boolean = {
    logger.debug(s"update(${extension.getId}, [${extension.getClass.getSimpleName}])")
    dispatch(extension)(repo => repo.update(extension.asInstanceOf[repo.ITEM]))
  }

  def delete(extensionId: String): Boolean = {
    logger.debug(s"delete($extensionId)")
    dispatch(extensionId)(_.delete)
  }

  def readAll(releaseId: String): Seq[ReleaseExtension] = {
    logger.debug(s"readAll($releaseId)")
    generic.readAll(releaseId) ++
      specialized.values.toSeq.flatMap { repo =>
        val extensionId = releaseId + SEPARATOR + repo.name
        repo.read(extensionId)
      }
  }

  def deleteAll(releaseId: String): Int = {
    logger.debug(s"deleteAll($releaseId)")
    generic.deleteAll(releaseId) +
      specialized.values.toSeq.map { repo =>
        val extensionId = releaseId + SEPARATOR + repo.name
        if (repo.delete(extensionId)) 1 else 0
      }.sum
  }

  def decorate(release: Release): Release = {
    logger.debug(s"decorate(${release.getId})")
    val extensions: Seq[ReleaseExtension] = readAll(release.getId)
    release.setExtensions(extensions.asJava)
    release
  }

  def find(query: ReleaseExtensionQuery): Seq[ReleaseExtension] = {
    generic.find(query)
  }

  def createOrUpdate(extension: ReleaseExtension): Unit = {
    logger.debug(s"createOrUpdate(${extension.getId}, [${extension.getClass.getSimpleName}])")
    dispatch(extension) { repo =>
      if (repo.exists(extension.getId)) {
        repo.update(extension.asInstanceOf[repo.ITEM])
      } else {
        repo.create(extension.asInstanceOf[repo.ITEM])
      }
    }
  }

  protected def repoFor(name: String): Option[ItemRepository[_ <: ReleaseExtension]] =
    specialized.get(name).orElse {
      if (genericExtensions contains name) {
        Some(generic)
      } else {
        None
      }
    }

  protected def dispatch[T](extensionId: String)(f: ItemRepository[_ <: ReleaseExtension] => String => T): T = {
    val name = getName(extensionId)
    repoFor(name).map(f).map(_.apply(extensionId)).getOrElse {
      throw new IllegalArgumentException(s"Unsupported release extension: '$name'")
    }
  }

  protected def dispatch[T](extension: ReleaseExtension)(f: ItemRepository[_ <: ReleaseExtension] => T): T = {
    val name = getName(extension.getId)
    repoFor(name).map(f).getOrElse {
      throw new IllegalArgumentException(s"Unsupported release extension: '$name'")
    }
  }
}

object ReleaseExtensionsRepository {

  abstract class GenericReleaseExtensionRepository extends ItemRepository[ReleaseExtension] {
    def find(query: ReleaseExtensionQuery): Seq[ReleaseExtension]

    def createAll(release: Release, extensionsIds: Set[String]): Int

    def readAll(releaseId: String): Seq[ReleaseExtension]

    def deleteAll(releaseId: String): Int
  }

  abstract class ReleaseExtensionSpecializedRepository[T <: ReleaseExtension] extends ItemRepository[T] {
    def name: String
  }

  sealed trait ReleaseExtensionQuery

  case class FindByNameAndCiUid(extensionId: String, releaseUids: Seq[Int]) extends ReleaseExtensionQuery {
    if (releaseUids.length >= 999) {
      throw new IllegalArgumentException(s"Requested extension $extensionId for too many releases (up to 999 releases will be accepted).")
    }
  }

}
