package com.xebialabs.xlrelease.service;

import java.util.Optional;

import com.xebialabs.xlrelease.config.XlrConfig;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.api.internal.InternalMetadataDecoratorService;
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional;
import com.xebialabs.xlrelease.domain.Phase;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.events.*;
import com.xebialabs.xlrelease.domain.status.PhaseStatus;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.repository.PhaseRepository;
import com.xebialabs.xlrelease.repository.PhaseVersion;
import com.xebialabs.xlrelease.repository.ReleaseRepository;
import com.xebialabs.xlrelease.service.PhaseRestart.RestartPhaseResult;
import com.xebialabs.xlrelease.views.MovementIndexes;

import io.micrometer.core.annotation.Timed;

import static com.google.common.base.Objects.equal;
import static com.google.common.base.Preconditions.checkArgument;
import static com.xebialabs.xlrelease.api.internal.ReleaseGlobalAndFolderVariablesDecorator.GLOBAL_AND_FOLDER_VARIABLES;
import static com.xebialabs.xlrelease.api.internal.ReleaseServerUrlDecorator.SERVER_URL;
import static com.xebialabs.xlrelease.builder.PhaseBuilder.newPhase;
import static com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi;
import static com.xebialabs.xlrelease.utils.CiHelper.eraseTokens;
import static com.xebialabs.xlrelease.utils.CiHelper.rewriteWithNewId;
import static com.xebialabs.xlrelease.variable.VariablePersistenceHelper.scanAndBuildNewVariables;
import static java.util.Arrays.asList;

@Service
public class PhaseService {

    public static final String DEFAULT_RELEASE_PHASE_TITLE = "New Phase";
    public static final String DEFAULT_WORKFLOW_PHASE_TITLE = "Workflow tasks";

    private CiIdService ciIdService;
    private ReleaseRepository releaseRepository;
    private PhaseRepository phaseRepository;
    private XLReleaseEventBus eventBus;
    private PhaseRestart phaseRestart;
    private InternalMetadataDecoratorService decoratorService;
    private ReleaseActorService releaseActorService;

    private XlrConfig xlrConfig;

    @Autowired
    public PhaseService(CiIdService ciIdService,
                        ReleaseRepository releaseRepository,
                        PhaseRepository phaseRepository,
                        XLReleaseEventBus eventBus,
                        PhaseRestart phaseRestart,
                        InternalMetadataDecoratorService decoratorService,
                        ReleaseActorService releaseActorService,
                        XlrConfig xlrConfig) {
        this.ciIdService = ciIdService;
        this.releaseRepository = releaseRepository;
        this.phaseRepository = phaseRepository;
        this.eventBus = eventBus;
        this.phaseRestart = phaseRestart;
        this.decoratorService = decoratorService;
        this.releaseActorService = releaseActorService;
        this.xlrConfig = xlrConfig;
    }

    @Timed
    public Phase create(String parentId) {
        Release release = releaseRepository.findById(parentId);
        Phase newPhase = buildInPosition(release, newPhase().withTitle(getPhaseTitle(release)).build(), null);
        return create(release, newPhase);
    }

    @Timed
    public Phase create(String parentId, Phase phaseTemplate, Integer position) {
        Release release = releaseRepository.findById(parentId);
        checkPosition(release, position);
        Phase newPhase = buildInPosition(release, phaseTemplate, position);
        return create(release, newPhase);
    }

    public Phase build(Release release, Phase phaseTemplate, Integer position) {
        checkPosition(release, position);
        return buildInPosition(release, phaseTemplate, position);
    }

    private void checkPosition(Release release, Integer position) {
        if (position != null) {
            checkArgument(position >= 0 && position <= release.getPhases().size(), "Phase index out of bounds");
            if (position < release.getPhases().size()) {
                Phase phaseAfterPosition = release.getPhases().get(position);
                checkArgument(phaseAfterPosition.isPlanned(), "Can't add a phase before a phase that is active or done");
            }
        }
    }

    private Phase buildInPosition(Release release, Phase phaseTemplate, Integer position) {
        String phaseTitle = getPhaseTitle(release);
        Phase releasePhase = phaseTemplate != null ? phaseTemplate :
                newPhase().withTitle(phaseTitle).build();
        checkArgument(release.isUpdatable(), "You can't add a phase to finished release");
        Phase newPhase = newPhase()
                .withId(getUniqueId(release.getId()))
                .withTitle(Optional.ofNullable(releasePhase.getTitle()).orElse(phaseTitle))
                .withStatus(PhaseStatus.PLANNED)
                .withColor(releasePhase.getColor())
                .withDescription(releasePhase.getDescription())
                .withDueDate(releasePhase.getDueDate())
                .withPlannedDuration(releasePhase.getPlannedDuration())
                .withScheduledStartDate(releasePhase.getScheduledStartDate())
                .withRelease(release)
                .withCiAttributes(releasePhase.get$ciAttributes())
                .build();

        release.addPhase(newPhase, position != null ? position : release.getPhases().size());

        scanAndBuildNewVariables(release, newPhase, ciIdService);

        return newPhase;
    }

    private Phase create(Release release, Phase newPhase) {
        Phase createdPhase = phaseRepository.create(release, newPhase);
        eventBus.publish(new PhaseCreatedEvent(createdPhase));
        return createdPhase;
    }

    private String getUniqueId(String parentId) {
        return ciIdService.getUniqueId(Type.valueOf(Phase.class), parentId);
    }

    @Timed
    public Phase findById(String phaseId) {
        return phaseRepository.findById(phaseId);
    }

    public Phase findByIdWithoutDecorators(String phaseId) {
        return phaseRepository.findByIdWithoutDecorators(phaseId);
    }


    @Timed
    public void delete(String phaseId) {
        Phase phase = findByIdWithoutDecorators(phaseId);
        checkArgument(phase.isPlanned() || (xlrConfig.isDefunctPhaseDeleteAllowed() && phase.isDefunct()), "Only non-active phases can be deleted");

        Release release = phase.getRelease();
        release.deletePhase(phase);
        phaseRepository.delete(release, phase);
        eventBus.publish(new PhaseDeletedEvent(phase));
    }

    @Timed
    public Phase update(String phaseId, Phase toUpdate) {
        Phase updated = findByIdWithoutDecorators(phaseId);
        checkArgument(updated.isUpdatable(),
                "The phase '%s' can't be updated because it has finished.", updated.getTitle());
        checkArgument(isScheduledStartDateUpdatable(updated, toUpdate),
                "The start date of '%s' can't be updated because this phase has already started.", updated.getTitle());

        Phase original = cloneCi(updated);

        if (toUpdate.getTitle() != null) {
            checkArgument(!toUpdate.getTitle().isEmpty(), "Phase title is required.");
            updated.setTitle(toUpdate.getTitle());
        }
        updated.setDescription(toUpdate.getDescription());
        updated.setColor(toUpdate.getColor());

        updated.updateDates(toUpdate.getScheduledStartDate(), toUpdate.getDueDate(), toUpdate.getPlannedDuration());

        scanAndBuildNewVariables(updated.getRelease(), updated, ciIdService);

        Phase savedPhase = phaseRepository.update(updated.getRelease(), original, updated);
        eventBus.publish(new PhaseUpdatedEvent(original, savedPhase));

        return savedPhase;
    }

    @Timed
    public Phase copyPhase(Release release, String originPhaseId, int targetPosition) {
        checkArgument(targetPosition >= 0, "Target position must be greater or equal to 0.");
        checkArgument(targetPosition <= (release.getPhases().size()), "Target position must be between 0 and %s.", release.getPhases().size());
        // release comes from ReleaseExecutionActor and it should always be fresh
        final Phase originPhase = release.getPhase(originPhaseId);
        if (originPhase == null) {
            throw new NotFoundException("Repository entity [%s] not found", originPhaseId);
        }
        checkArgument(originPhase.isUpdatable(), "Cannot copy phase '%s' because it is %s.", originPhase.getTitle(), originPhase.getStatus());

        // when it is not a position after all existing phases, check if the phase we're adding this 'before' is
        // in a planned state. If not, then it is not allowed to copy the phase to this position.
        if (targetPosition < release.getPhases().size()) {
            final Phase phaseAtPosition = release.getPhase(targetPosition); // we can do this because this will be the element shifted to the 'right' (bottom) later
            checkArgument(phaseAtPosition.isUpdatable(), "Can't copy '%s' to targetPosition %s because it will move a phase with status %s to the right. A copied phase must be positioned after an ACTIVE phase.",
                    originPhase.getTitle(), targetPosition, phaseAtPosition.getStatus());
            checkArgument(!phaseAtPosition.isActive(), "Can't copy '%s' to targetPosition %s because it will move an ACTIVE phase to the right.",
                    originPhase.getTitle(), targetPosition);
        }

        String newPhaseId = getUniqueId(release.getId());

        Phase newPhase = cloneCi(originPhase);
        newPhase.resetToPlanned();
        eraseTokens(newPhase);
        rewriteWithNewId(newPhase, newPhaseId);
        newPhase.setTitle(newPhase.getTitle() + " (copy)");
        newPhase.setRelease(release);

        release.getPhases().add(targetPosition, newPhase);

        Phase savedPhase = phaseRepository.create(release, newPhase);
        eventBus.publish(new PhaseCopiedEvent(savedPhase));

        return savedPhase;
    }

    @Timed
    public Phase duplicatePhase(String originPhaseId) {
        Phase originPhase = findByIdWithoutDecorators(originPhaseId);
        Release release = originPhase.getRelease();
        checkArgument(originPhase.isUpdatable(), "Can't duplicate phase '%s' because it is %s.", originPhase.getTitle(), originPhase.getStatus());
        String newPhaseId = getUniqueId(release.getId());

        Phase duplicatedPhase = cloneCi(originPhase);
        duplicatedPhase.resetToPlanned();
        eraseTokens(duplicatedPhase);
        rewriteWithNewId(duplicatedPhase, newPhaseId);
        duplicatedPhase.setOriginId(null);
        release.addBelow(originPhaseId, duplicatedPhase);

        Phase savedDuplicatedPhase = phaseRepository.create(release, duplicatedPhase);

        eventBus.publish(new PhaseDuplicatedEvent(savedDuplicatedPhase));

        return savedDuplicatedPhase;
    }

    @Timed
    public Phase movePhase(Release release, MovementIndexes movementIndexes) {
        Phase movedPhase = release.movePhase(movementIndexes.getOriginIndex(), movementIndexes.getTargetIndex());
        Phase savedPhase = phaseRepository.move(release, movedPhase);
        eventBus.publish(new PhaseMovedEvent(savedPhase, movementIndexes.getOriginIndex(), movementIndexes.getTargetIndex()));
        return savedPhase;
    }

    @Timed
    @IsTransactional
    public Release restartPhases(String releaseId, String phaseId, String taskId, PhaseVersion phaseVersion, boolean resumeRelease, Release release) {
        RestartPhaseResult restartPhaseResult = phaseRestart.restartPhases(releaseId, phaseId, taskId, phaseVersion, resumeRelease, release);
        Release restartedRelease = restartPhaseResult.release();
        phaseRestart.updateGatesReferencingPhases(restartedRelease, restartPhaseResult.phaseIdsToRestore(), restartPhaseResult.restoredPhaseIds());
        decoratorService.decorate(restartedRelease, asList(GLOBAL_AND_FOLDER_VARIABLES(), SERVER_URL()));
        return restartedRelease;
    }

    @Timed
    public String getTitle(String id) {
        return phaseRepository.getTitle(id);
    }

    private boolean isScheduledStartDateUpdatable(Phase original, Phase updated) {
        return original.isPlanned() || equal(original.getScheduledStartDate(), updated.getScheduledStartDate());
    }

    private String getPhaseTitle(Release release) {
        return release.isWorkflow() ? DEFAULT_WORKFLOW_PHASE_TITLE : DEFAULT_RELEASE_PHASE_TITLE;
    }
}
