package com.xebialabs.xlrelease.service;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.domain.Configuration;
import com.xebialabs.xlrelease.domain.events.ConfigurationCreatedEvent;
import com.xebialabs.xlrelease.domain.events.ConfigurationDeletedEvent;
import com.xebialabs.xlrelease.domain.events.ConfigurationUpdatedEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import com.xebialabs.xlrelease.utils.CiHelper;
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils;

import io.micrometer.core.annotation.Timed;

import static com.xebialabs.xlrelease.repository.Ids.CUSTOM_CONFIGURATION_ROOT;
import static java.util.stream.Collectors.toList;

@Service
public class SharedConfigurationService {

    private CiIdService ciIdService;
    private ConfigurationRepository configurationRepository;
    private XLReleaseEventBus eventBus;

    @Autowired
    public SharedConfigurationService(CiIdService ciIdService, ConfigurationRepository configurationRepository, XLReleaseEventBus eventBus) {
        this.ciIdService = ciIdService;
        this.configurationRepository = configurationRepository;
        this.eventBus = eventBus;
    }

    @Timed
    public boolean exists(String configurationInstanceId) {
        return configurationRepository.exists(configurationInstanceId);
    }

    @Timed
    public List<Configuration> findByIds(List<String> configurationIds) {
        return configurationRepository.findByIds(configurationIds);
    }

    @Timed
    public Configuration findById(String configurationInstanceId) {
        return configurationRepository.read(configurationInstanceId);
    }

    @Timed
    public List<Configuration> findAll() {
        return configurationRepository.findAllByTypeAndTitle(Type.valueOf(Configuration.class), null, null, false);
    }

    @Timed
    public List<Configuration> searchByTypeAndTitle(Type configurationType, String title) {
        return configurationRepository.findAllByTypeAndTitle(configurationType, title, null, false);
    }

    @Timed
    public List<Configuration> searchByTypeAndTitle(Type configurationType, String title, String folderId, boolean folderOnly) {
        return configurationRepository.findAllByTypeAndTitle(configurationType, title, folderId, folderOnly);
    }

    @Timed
    public List<Descriptor> findAllConfigurationDescriptors() {
        Type configurationType = Type.valueOf(Configuration.class);
        Collection<Type> allTypes = DescriptorRegistry.getSubtypes(configurationType);
        return allTypes.stream()
                .filter(type -> !type.getDescriptor().isVirtual())
                .map(Type::getDescriptor)
                .collect(toList());
    }

    @Timed
    public Configuration update(String configurationInstanceId, Configuration configuration) {
        Configuration original = findById(configurationInstanceId);

        configuration.setId(configurationInstanceId);
        CiHelper.fixUpInternalReferences(configuration);

        PasswordVerificationUtils.replacePasswordPropertiesInCiIfNeededJava(Optional.of(original), configuration);

        configurationRepository.update(configuration);
        eventBus.publish(new ConfigurationUpdatedEvent(configuration));

        return configuration;
    }


    @Timed
    public Configuration create(Configuration configuration) {
        String id = getUniqueId(CUSTOM_CONFIGURATION_ROOT);
        configuration.setId(id);
        CiHelper.fixUpInternalReferences(configuration);

        PasswordVerificationUtils.replacePasswordPropertiesInCiIfNeededJava(Optional.empty(), configuration);

        configurationRepository.create(configuration);
        eventBus.publish(new ConfigurationCreatedEvent(configuration));

        return configuration;
    }

    @Timed
    public void delete(String configurationInstanceId) {
        Configuration conf = configurationRepository.read(configurationInstanceId);
        configurationRepository.delete(configurationInstanceId);
        eventBus.publish(new ConfigurationDeletedEvent(conf));
    }

    private String getUniqueId(String parentId) {
        return ciIdService.getUniqueId(Type.valueOf(Configuration.class), parentId);
    }

}
