package com.xebialabs.xlrelease.scheduler.workers

import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.scheduler.ScriptTaskJob
import com.xebialabs.xlrelease.scheduler.workers.Worker.{ExecuteJob, ProcessJobResult, ScriptTaskExecutionResult}
import com.xebialabs.xlrelease.script.{FailureScriptTaskResult, RestartScriptTaskResult, ScriptService, SuccessScriptTaskResult}
import com.xebialabs.xlrelease.service.{CommentService, PhaseService}
import com.xebialabs.xlrelease.user.User.{AUTHENTICATED_USER, SYSTEM}
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import scala.util.{Failure, Success, Try}

@Component
class ScriptTaskWorker(val releaseActorService: ReleaseActorService,
                       phaseService: PhaseService,
                       val scriptServicesList: java.util.List[ScriptService],
                       commentService: CommentService)
  extends Worker with ScriptServiceSupport with TaskWorkerFailureLogic with Logging {

  override def execute: ExecuteJob = {
    case ScriptTaskJob(taskRef) =>
      val task = taskRef.get()
      val result = Try {
        commentService.decorate(task)
        scriptServiceFor(task).executeScriptTask(task)
      }
      ScriptTaskExecutionResult(task.getId, task.getExecutionId, result)
  }

  override def processResult: ProcessJobResult = {
    case ScriptTaskExecutionResult(taskId, executionId, executionResult) =>
      executionResult match {
        case Failure(error) => handleFailure(taskId, executionId, error)
        case Success(result) => result match {
          case SuccessScriptTaskResult(taskId, executionId, executionLog, attachmentId, scriptTaskResults, authentication) =>
            withAuthentication(authentication) {
              releaseActorService.finishScriptTask(taskId, executionId, executionLog, attachmentId, Some(scriptTaskResults))
            }
          case RestartScriptTaskResult(taskId, executionId, _, _, scriptTaskResults, exception, authentication) =>
            withAuthentication(authentication) {
              commentService.create(taskId, exception.getMessage, SYSTEM, false)
              releaseActorService.failScriptTask(taskId, exception.getMessage, executionId, Option.empty, Some(scriptTaskResults))
              Try(releaseActorService.restartPhase(exception.getReleaseId,
                exception.getPhaseId,
                exception.getTaskId,
                exception.getPhaseVersion,
                exception.isResumeRelease))
                .recover {
                  case e: Exception =>
                    commentService.create(taskId, e.getMessage, AUTHENTICATED_USER, true);
                }
            }
          case FailureScriptTaskResult(taskId, executionId, executionLog, attachmentId, scriptTaskResults, authentication) =>
            withAuthentication(authentication) {
              releaseActorService.failScriptTask(taskId, executionLog, executionId, attachmentId, Option(scriptTaskResults))
            }
        }
      }
  }

}
