package com.xebialabs.deployit.plumbing;

import java.io.IOException;
import jakarta.servlet.*;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.servlet.http.HttpSession;
import org.springframework.boot.actuate.health.Status;
import org.springframework.http.HttpStatus;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.core.Authentication;

import com.xebialabs.deployit.security.PermissionEnforcer;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.xlrelease.service.XlrServiceManager;

public class XlrAccessControlFilter implements Filter {

    private final PermissionEnforcer permissionEnforcer;

    private final XlrServiceManager serviceManager;

    private final boolean isMaintenanceMode;

    public XlrAccessControlFilter(PermissionEnforcer permissionEnforcer, XlrServiceManager serviceManager, boolean isMaintenanceMode) {
        this.permissionEnforcer = permissionEnforcer;
        this.serviceManager = serviceManager;
        this.isMaintenanceMode = isMaintenanceMode;
    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
        HttpServletRequest httpServletRequest = (HttpServletRequest) servletRequest;
        String requestPath = httpServletRequest.getRequestURI();
        String requestMethod = httpServletRequest.getMethod();
        // This should be treated as a static resource, but it does not fall under the spring security
        // Therefore if user has an active session, it will get treated as an authorized resource, and it will be blocked for non admin users
        if (("/".equals(requestPath)) && "GET".equals(requestMethod)) {
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }
        //Technically we do not need isMaintenanceMode check as the service manager will always be down in the maintenance mode,
        // but I will keep it here in case the behavior in service manager changes in the future.
        boolean isServicesDown = isMaintenanceMode || serviceManager.getHealthStatus() != Status.UP;
        if (isServicesDown) {
            doFilterInternal(servletRequest, servletResponse, filterChain);
        } else {
            filterChain.doFilter(servletRequest, servletResponse);
        }
    }

    public void doFilterInternal(final ServletRequest request, final ServletResponse response, final FilterChain chain) throws IOException, ServletException {
        HttpServletRequest httpServletRequest = (HttpServletRequest) request;
        HttpSession session = httpServletRequest.getSession(false);
        Authentication currentUserAuthentication = Permissions.getAuthentication();

        boolean sessionExists = session != null;
        boolean isAnonymousUser = currentUserAuthentication instanceof AnonymousAuthenticationToken;
        boolean isSessionlessAuthenticatedUser = currentUserAuthentication != null && !isAnonymousUser && !sessionExists;

        // authenticated (api and static resource) calls with JSESSIONID, e.g. calls from GUI
        // Unsecured Resource Request calls without any authentication attached e.g. calls to register license, setting Themes
        if (sessionExists) {
            boolean isStaticResourceRequest = currentUserAuthentication == null;
            boolean isUnsecuredResourceRequest = session.getAttribute("SPRING_SECURITY_CONTEXT") == null;
            if (isUnsecuredResourceRequest || isStaticResourceRequest) {
                chain.doFilter(request, response);
                return;
            }
            isAllowedAccessInMaintenanceMode(request, response, chain, currentUserAuthentication);
            return;
        }

        // api calls from e.g. Postman. They don't have to have a JSESSIONID and thus won't happen inside a session, but they have to be authenticated
        if (isSessionlessAuthenticatedUser) {
            isAllowedAccessInMaintenanceMode(request, response, chain, currentUserAuthentication);
            return;
        }

        chain.doFilter(request, response);
    }


    private void isAllowedAccessInMaintenanceMode(final ServletRequest request, final ServletResponse response, final FilterChain chain, final Authentication currentUserAuthentication) throws IOException, ServletException {
        if (permissionEnforcer.isAdmin(currentUserAuthentication)) {
            chain.doFilter(request, response);
        } else {
            HttpServletResponse httpResponse = (HttpServletResponse) response;
            httpResponse.setStatus(HttpStatus.SERVICE_UNAVAILABLE.value());
        }

    }
}
