package com.xebialabs.deployit.plumbing;

import java.io.Closeable;
import java.io.IOException;
import java.util.TimerTask;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.overthere.OverthereExecutionOutputHandler;

import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;

public class PollingExecutionOutputHandler implements OverthereExecutionOutputHandler, Closeable {

    private final Function<String, Object> onFlushCallback;
    private final boolean resetBufferOnFlush;

    private final StringBuilder buffer;
    private final ScheduledFuture<?> cancelHandle;

    protected PollingExecutionOutputHandler(ScheduledExecutorService scheduledExecutorService, Function<String, Object> onFlushCallback, boolean resetBufferOnFlush, long initialDelay, long delay) {
        this.onFlushCallback = onFlushCallback;
        this.resetBufferOnFlush = resetBufferOnFlush;
        this.buffer = new StringBuilder();
        Runnable flushTimerTask = new TimerTask() {
            @Override
            public void run() {
                flush();
            }
        };
        cancelHandle = scheduledExecutorService.scheduleAtFixedRate(flushTimerTask, initialDelay, delay, TimeUnit.MILLISECONDS);
    }

    public synchronized void flush() {
        String output;
        synchronized (buffer) {
            output = buffer.toString();
            if (resetBufferOnFlush && buffer.length() > 0) {
                buffer.setLength(0);
            }
        }

        if (!isBlank(output)) {
            logger.debug("Flushing buffered content to callback:\n{}", output);
            onFlushCallback.apply(output);
        }
    }

    public void handleChar(final char c) {
        synchronized (buffer) {
            buffer.append(c);
        }
    }

    @Override
    public void handleLine(final String line) {
        synchronized (buffer) {
            buffer.append(line);
        }
    }

    @Override
    public void close() throws IOException {
        logger.debug("Shutdown polling timer thread");
        cancelHandle.cancel(true);
        logger.debug("Flushing content still present in buffer...");
        flush();
    }

    private static final Logger logger = LoggerFactory.getLogger(PollingExecutionOutputHandler.class);

    public static PollingExecutionOutputHandler pollingHandler(ScheduledExecutorService scheduledExecutorService, Function<String, Object> onFlushCallback, boolean resetBufferOnFlush, long initialDelay, long delay) {
        return new PollingExecutionOutputHandler(scheduledExecutorService, onFlushCallback, resetBufferOnFlush, initialDelay, delay);
    }

}
