package com.xebialabs.xlrelease.export;

import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.domain.CustomScriptTask;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.status.FlagStatus;
import com.xebialabs.xlrelease.json.JsonKeys;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static java.util.Collections.unmodifiableSet;

public class TemplateJsonHelper {

    private static final Set<String> DEFAULT_PROPERTIES = unmodifiableSet(newHashSet("id", "type"));

    public static boolean isCustomScriptTask(JSONObject task) throws JSONException {
        return task.getString("type").equals(Type.valueOf(CustomScriptTask.class).toString());
    }

    public static boolean typeNotFound(JSONObject pythonScript) throws JSONException {
        return !Type.valueOf(pythonScript.getString("type")).exists();
    }

    public static boolean typeNotFound(ObjectNode pythonScript) {
        return !Type.valueOf(pythonScript.get("type").asText()).exists();
    }

    public static void resetToManualTask(JSONObject task) throws JSONException {
        Type manualTaskType = Type.valueOf(Task.class);

        task.put("type", manualTaskType.toString());

        Set<String> propertiesOnManualTask = newHashSet();
        for (PropertyDescriptor propertyDescriptor : manualTaskType.getDescriptor().getPropertyDescriptors()) {
            propertiesOnManualTask.add(propertyDescriptor.getName());
        }
        Set<String> propertiesOnUnknownTask = newHashSet();
        for (Iterator i = task.keys(); i.hasNext(); ) {
            propertiesOnUnknownTask.add((String) i.next());
        }

        for (String property : propertiesOnUnknownTask) {
            if (!propertiesOnManualTask.contains(property) && !DEFAULT_PROPERTIES.contains(property)) {
                task.remove(property);
            }
        }
    }

    public static void flagTask(JSONObject task, String warningMessage) throws JSONException {
        task.put("flagStatus", FlagStatus.ATTENTION_NEEDED);
        task.put("flagComment", warningMessage);
    }

    public static void flagTask(ObjectNode task, String warningMessage) {
        task.put("flagStatus", FlagStatus.ATTENTION_NEEDED.toString());
        task.put("flagComment", warningMessage);
    }

    public static List<JSONObject> getTasksFromRelease(JSONObject releaseJson) throws JSONException {
        List<JSONObject> allTasks = newArrayList();

        if (!releaseJson.has("phases")) {
            return allTasks;
        }

        JSONArray phases = releaseJson.getJSONArray("phases");
        for (int i = 0; i < phases.length(); i++) {
            JSONObject phaseJson = phases.getJSONObject(i);
            getAllTasks(phaseJson, allTasks);
        }

        return allTasks;
    }

    public static List<ObjectNode> getTasksFromRelease(ObjectNode releaseJson) {
        List<ObjectNode> allTasks = newArrayList();
        if (!releaseJson.has("phases")) {
            return allTasks;
        }
        Iterator<JsonNode> phases = releaseJson.get("phases").elements();
        while (phases.hasNext()) {
            JsonNode phaseJson = phases.next();
            if (phaseJson instanceof ObjectNode) {
                getAllTasks((ObjectNode) phaseJson, allTasks);
            }
        }

        return allTasks;
    }

    public static List<JSONObject> getDashboardsFromRelease(JSONObject releaseJson) throws JSONException {
        List<JSONObject> allDashboards = newArrayList();

        if (!releaseJson.has(JsonKeys.extensions())) {
            return allDashboards;
        }

        JSONArray extensions = releaseJson.getJSONArray(JsonKeys.extensions());
        for (int i = 0; i < extensions.length(); i++) {
            JSONObject extension = extensions.getJSONObject(i);
            if (extension.has(JsonKeys.tiles()) && extension.has("rows") && extension.has("columns")) {
                allDashboards.add(extension);
            }
        }
        return allDashboards;
    }

    private static void getAllTasks(ObjectNode containerJson, List<ObjectNode> allTasks) {
        if (containerJson.has("tasks")) {
            Iterator<JsonNode> tasks = containerJson.get("tasks").elements();
            while (tasks.hasNext()) {
                JsonNode taskJson = tasks.next();
                if (taskJson instanceof ObjectNode) {
                    allTasks.add((ObjectNode) taskJson);
                    getAllTasks((ObjectNode) taskJson, allTasks);
                }
            }
        }
    }

    private static List<JSONObject> getAllTasks(JSONObject containerJson, List<JSONObject> allTasks) throws JSONException {
        if (!containerJson.has("tasks")) {
            return allTasks;
        }

        JSONArray tasks = containerJson.getJSONArray("tasks");
        for (int j = 0; j < tasks.length(); j++) {
            JSONObject taskJson = tasks.getJSONObject(j);
            allTasks.add(taskJson);
            getAllTasks(taskJson, allTasks);
        }

        return allTasks;
    }

    public static JSONObject getPhase(JSONObject taskJson, JSONObject releaseJson) throws JSONException {
        if (!releaseJson.has("phases")) {
            return null;
        }

        JSONArray phases = releaseJson.getJSONArray("phases");
        for (int i = 0; i < phases.length(); i++) {
            JSONObject phaseJson = phases.getJSONObject(i);
            if (containsTask(phaseJson, taskJson)) {
                return phaseJson;
            }
        }

        return null;
    }

    public static ObjectNode getPhase(ObjectNode taskJson, ObjectNode releaseJson) {
        if (!releaseJson.has("phases")) {
            return null;
        }

        Iterator<JsonNode> phases = releaseJson.get("phases").elements();
        while (phases.hasNext()) {
            JsonNode phaseJson = phases.next();
            if (phaseJson instanceof ObjectNode) {
                if (containsTask((ObjectNode) phaseJson, taskJson)) {
                    return (ObjectNode) phaseJson;
                }
            }
        }

        return null;
    }

    private static boolean containsTask(JSONObject containerJson, JSONObject taskToFind) throws JSONException {
        if (!containerJson.has("tasks")) {
            return false;
        }

        JSONArray tasks = containerJson.getJSONArray("tasks");
        for (int j = 0; j < tasks.length(); j++) {
            JSONObject taskJson = tasks.getJSONObject(j);
            if (taskToFind == taskJson) {
                return true;
            } else {
                if (containsTask(taskJson, taskToFind)) {
                    return true;
                }
            }
        }

        return false;
    }

    private static boolean containsTask(ObjectNode containerJson, ObjectNode taskToFind) {
        if (!containerJson.has("tasks")) {
            return false;
        }

        Iterator<JsonNode> tasks = containerJson.get("tasks").elements();
        while (tasks.hasNext()) {
            JsonNode taskJson = tasks.next();
            if (taskJson instanceof ObjectNode) {
                if (taskToFind == taskJson) {
                    return true;
                } else {
                    if (containsTask((ObjectNode) taskJson, taskToFind)) {
                        return true;
                    }
                }
            }
        }

        return false;
    }
}
