package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.GateTask
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.domain.status.TaskStatus.COMPLETED
import com.xebialabs.xlrelease.repository.{Ids, TaskRepository}
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.user.User
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.jmx.`export`.annotation.{ManagedOperation, ManagedResource}
import org.springframework.stereotype.{Component, Service}

import java.time.Instant
import java.time.temporal.ChronoUnit
import java.util.Date
import scala.jdk.CollectionConverters._


@Service
class StuckGateDetectorService @Autowired()(
                                             val xlrConfig: XlrConfig,
                                             val releaseService: ReleaseService,
                                             val releaseActorService: ReleaseActorService,
                                             val taskRepository: TaskRepository
                                           ) extends Logging {

  private val completionComment: String = "Completed by StuckGateDetectorService"

  def run(): Unit = {
    unstuckGates(findInProgressGateTasks())
  }

  private def findInProgressGateTasks(): Seq[String] = taskRepository.findTaskIdsByTaskTypeStatusAndStartDate(
    Type.valueOf(classOf[GateTask]),
    TaskStatus.IN_PROGRESS,
    Date.from(Instant.now().minus(xlrConfig.features.gateActor.graceDuration.toMillis, ChronoUnit.MILLIS)))

  private def unstuckGates(gates: Seq[String]): Unit = {
    logger.debug(s"Monitoring gates.size gate tasks")
    val resolveOptions = ResolveOptions.WITHOUT_DECORATORS.withReferences

    val gatesByRelease = gates.groupBy(gateId => Ids.releaseIdFrom(gateId))
    logger.debug(s"Processing ${gatesByRelease.size} releases with gate tasks")
    gatesByRelease.foreach {
      case (releaseId, gateTaskIds) =>
        try {
          logger.debug(s"Loading ${releaseId}")
          // do we need to load more than just DEPENDENCIES?
          val release = releaseService.findById(releaseId, resolveOptions)

          gateTaskIds.foreach { gateId =>
            try {
              val gateTask = release.getTask(gateId).asInstanceOf[GateTask]
              logger.debug("       status: " + gateTask.getStatus)
              logger.debug("   conditions: " + gateTask.getConditions.asScala.forall(_.isChecked))
              logger.debug(" dependencies: " + gateTask.getDependencies.asScala.forall(_.isDone))

              if ((gateTask.getConditions.asScala.nonEmpty || gateTask.getDependencies.asScala.nonEmpty)
                && gateTask.getConditions.asScala.forall(_.isChecked)
                && gateTask.getDependencies.asScala.forall(_.isDone)) {
                logger.info(s"Completing stuck gate ${gateId}")
                releaseActorService.markTaskAsDone(COMPLETED, gateTask.getId, completionComment, User.SYSTEM)
              } else {
                logger.debug(s"Conditions not met for ${gateId} (in progress)")
              }
            } catch {
              case e: Exception => logger.error(s"Unable to process task ${gateId}", e)
            }
          }
        } catch {
          case e: Exception => logger.error(s"Unable to porcess release ${releaseId}", e)
        }
    }

  }
}

@Component
@ManagedResource(objectName = "com.xebialabs.xlrelease.settings:name=StuckGateDetector", description = "MBean to configure the stuck gate detector")
class StuckGateDetectorMBean(stuckGateDetectorService: StuckGateDetectorService) {
  @ManagedOperation(description = "Run stuck gate detector")
  def run(): Unit = {
    stuckGateDetectorService.run()
  }
}
