package com.xebialabs.xlrelease.metrics

import com.codahale.metrics.jmx.ObjectNameFactory
import com.xebialabs.xlrelease.metrics.MetricsConfiguration.METRICS_DOMAIN

import javax.management.{MalformedObjectNameException, ObjectName}

object XlrObjectNameFactory extends ObjectNameFactory {

  override def createName(`type`: String, originalDomain: String, originalName: String): ObjectName = {
    val (domain, name) = customDomainAndName(originalName)
    try {
      val objectName = new ObjectName(domain, "name", name)
      if (objectName.isPattern) {
        new ObjectName(domain, "name", ObjectName.quote(name))
      } else {
        objectName
      }
    } catch {
      case _: MalformedObjectNameException =>
        try {
          new ObjectName(domain, "name", ObjectName.quote(name))
        } catch {
          case e: MalformedObjectNameException =>
            throw new RuntimeException(e)
        }
    }
  }

  //noinspection ScalaStyle
  private[metrics] def customDomainAndName(name: String): (String, String) = {
    val xlrClass = """method\.timed\.class\.com\.xebialabs\.xlrelease\.(?:([\w\.]+)\.)?([A-Z]\w+(?:\.\w+)?)\.exception\.([^\.]+)\.method\.([^\.]+)""".r
    val (domainSuffix, customName) = name match {
      case n if n.startsWith("executor.") => ("executors", name.stripPrefix("executor."))
      case n if n.startsWith("jetty.connections.") => ("jetty", name.stripPrefix("jetty.connections."))
      case n if n.startsWith("jetty.thread-pool.") => ("jetty", name.stripPrefix("jetty.thread-pool."))
      case n if n.startsWith("jetty.") => ("jetty", name.stripPrefix("jetty."))
      case n if n.startsWith("logback.events.") => ("logback", name.stripPrefix("logback.events."))
      case n if n.startsWith("jvm.") => ("jvm", name.stripPrefix("jvm."))
      case n if n.startsWith("system.") => ("system", name.stripPrefix("system."))
      case n if n.startsWith("process.") => ("system", name.stripPrefix("process."))
      case n if n.startsWith("org.eclipse.jetty.webapp.WebAppContext") => ("handler", name.split('.').last)
      case n if n.contains("RepositoryPool") => ("pool.repository", name.stripPrefix("RepositoryPool.pool.").stripSuffix(".pool.RepositoryPool"))
      case n if n.contains("ReportingPool") => ("pool.reporting", name.stripPrefix("ReportingPool.pool.").stripSuffix(".pool.ReportingPool"))
      case n if n.startsWith("ReleaseExecutionActor") => ("ReleaseExecutionActor.counts", name.stripPrefix("ReleaseExecutionActor:"))
      case xlrClass(suffix, className, exception, methodName) if suffix != null && suffix.toLowerCase().contains("service") =>
        extractDomainSuffixAndCustomName("service", className, exception, methodName)
      case xlrClass(suffix, className, exception, methodName) if suffix != null && suffix.toLowerCase().contains("repository") =>
        extractDomainSuffixAndCustomName("repository", className, exception, methodName)
      case xlrClass(suffix, className, exception, methodName) =>
        if (className.toLowerCase().endsWith("service")) {
          extractDomainSuffixAndCustomName("service", className, exception, methodName)
        } else if (className.toLowerCase().endsWith("repository")) {
          extractDomainSuffixAndCustomName("repository", className, exception, methodName)
        } else {
          extractDomainSuffixAndCustomName(suffix, className, exception, methodName)
        }
      case _ => (null, name)
    }
    val customDomain = if (domainSuffix == null) METRICS_DOMAIN else s"$METRICS_DOMAIN.$domainSuffix"
    (customDomain, customName)
  }

  private def extractDomainSuffixAndCustomName(suffix: String, className: String, exception: String, methodName: String): (String, String) = {
    if (exception != null && exception == "none") {
      (suffix, s"$className.$methodName")
    } else {
      (suffix, s"$className.$methodName.exception.$exception")
    }
  }
}
