package com.xebialabs.xlrelease.scheduler.converters

import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.scheduler._
import com.xebialabs.xlrelease.script.{TaskSoftReference, TaskSoftReferenceFactory}
import com.xebialabs.xlrelease.service.ReleaseService
import org.springframework.stereotype.Component

@Component
class TaskJobConverter(releaseService: ReleaseService) {

  // scalastyle:off cyclomatic.complexity
  def fromJobRow(jobRow: JobRow, taskSoftReference: Option[TaskSoftReference[_ <: Task]] = None): TaskJob[_] = {
    val taskRef = taskSoftReference.getOrElse(TaskSoftReferenceFactory.taskRef(releaseService, jobRow.taskId, jobRow.executionId, jobRow.releaseUid))

    val job = jobRow.jobType match {
      case JobType.PRECONDITION =>
        PreconditionJob(taskRef.asInstanceOf[TaskSoftReference[Task]])
      case JobType.FACET_CHECK =>
        FacetCheckJob(taskRef.asInstanceOf[TaskSoftReference[Task]])
      case JobType.FAILURE_HANDLER =>
        FailureHandlerJob(taskRef.asInstanceOf[TaskSoftReference[Task]])
      case JobType.CUSTOM_SCRIPT_TASK =>
        CustomScriptTaskJob(taskRef.asInstanceOf[TaskSoftReference[CustomScriptTask]])
      case JobType.NEXT_CUSTOM_SCRIPT_TASK =>
        NextCustomScriptTaskJob(taskRef.asInstanceOf[TaskSoftReference[CustomScriptTask]], jobRow.scheduledStartTime)
      case JobType.SCRIPT_TASK =>
        ScriptTaskJob(taskRef.asInstanceOf[TaskSoftReference[ResolvableScriptTask]])
      case JobType.CREATE_RELEASE_TASK =>
        CreateReleaseTaskJob(taskRef.asInstanceOf[TaskSoftReference[CreateReleaseTask]])
      case JobType.NOTIFICATION_TASK =>
        NotificationTaskJob(taskRef.asInstanceOf[TaskSoftReference[NotificationTask]])
      case JobType.CONTAINER_TASK =>
        ContainerTaskJob(taskRef.asInstanceOf[TaskSoftReference[ContainerTask]])
      case JobType.FAIL_JOB | JobType.STOP_WORKER_THREAD =>
        throw new IllegalStateException(s"'${jobRow.jobType}' is not a valid job type for TaskJob")
    }

    job.id = jobRow.id
    job.node = jobRow.node
    job.runnerId = jobRow.runnerId
    job.status = jobRow.status
    job.version = jobRow.version
    job.submitTime = jobRow.submitTime
    job.reservationTime = jobRow.reservationTime
    job.startTime = jobRow.startTime
    job
  }

}
