package com.xebialabs.xlrelease.security.authentication;

import java.util.HashMap;
import java.util.HashSet;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.lang.NonNull;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.security.authentication.PersonalAuthenticationToken;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.variables.reference.VariableMappingUsagePoint;
import com.xebialabs.xlrelease.repository.CiProperty;
import com.xebialabs.xlrelease.script.VariablesHolderForScriptContext;
import com.xebialabs.xlrelease.security.UsernamePassword;
import com.xebialabs.xlrelease.service.ConfigurationVariableService;
import com.xebialabs.xlrelease.service.UserLastActiveActorService;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.xebialabs.xlrelease.variable.VariableHelper.replaceAll;
import static scala.jdk.javaapi.CollectionConverters.asScala;

@Component
public class AuthenticationService {

    private AuthenticationManager authenticationManager;
    private UserLastActiveActorService userLastActiveActorService;

    @Autowired
    public AuthenticationService(
            @Qualifier("authenticationManager")
            final AuthenticationManager authenticationManager,
            final UserLastActiveActorService userLastActiveActorService) {
        this.authenticationManager = authenticationManager;
        this.userLastActiveActorService = userLastActiveActorService;
    }

    private ThreadLocal<Authentication> authenticationThreadLocal = new ThreadLocal<>();
    private ThreadLocal<Task> currentScriptTask = new ThreadLocal<>();
    private ThreadLocal<Release> currentScriptRelease = new ThreadLocal<>();
    private ThreadLocal<VariablesHolderForScriptContext> currentVariables = new ThreadLocal<>();

    public UsernamePassword loginScriptUser(Task task) {
        Release release = task.getRelease();
        currentScriptTask.set(task);
        return loginScriptUser(release);
    }

    public UsernamePassword loginScriptUser(Task task, VariablesHolderForScriptContext variablesHolderForScriptContext) {
        UsernamePassword usernamePass = loginScriptUser(task);
        currentVariables.set(variablesHolderForScriptContext);
        return usernamePass;
    }

    public UsernamePassword loginScriptUser(Release release) {
        currentScriptRelease.set(release);
        String scriptUser = replaceAll(release.getScriptUsername(), release.getAllStringVariableValues(), new HashSet<>(), false);
        String scriptUserPassword = resolveUserScriptPassword(release);

        // Save authentication
        authenticationThreadLocal.set(SecurityContextHolder.getContext().getAuthentication());

        if (release.isWorkflow() && isNullOrEmpty(scriptUser) && isNullOrEmpty(scriptUserPassword)) {
            //attempt to login with workflow owner
            authenticateScriptUser(new OwnerAuthenticationToken(release.getOwner(), null));
        } else if (isNullOrEmpty(scriptUser) && !isNullOrEmpty(scriptUserPassword)) {
            //attempt to login with personal access token
            authenticateScriptUser(new PersonalAuthenticationToken(scriptUserPassword));
        } else if (!isNullOrEmpty(scriptUser)) {
            if (isNullOrEmpty(scriptUserPassword)) {
                throw new IllegalArgumentException(String.format("'Password' property has to be set to run automated tasks as '%s'", scriptUser));
            }
            //attempt to login with username and password
            authenticateScriptUser(new UsernamePasswordAuthenticationToken(scriptUser, scriptUserPassword));
        }

        currentVariables.remove();

        return UsernamePassword.apply(scriptUser, scriptUserPassword);
    }

    private void authenticateScriptUser(final Authentication token) {
        Authentication authentication = authenticationManager.authenticate(token);
        userLastActiveActorService.updateLastActive(authentication.getName());
        var newCtxt = SecurityContextHolder.createEmptyContext();
        newCtxt.setAuthentication(authentication);
        SecurityContextHolder.setContext(newCtxt);
    }

    public void logoutScriptUser() {
        // Restore authentication
        var newCtxt = SecurityContextHolder.createEmptyContext();
        newCtxt.setAuthentication(authenticationThreadLocal.get());
        SecurityContextHolder.setContext(newCtxt);
        authenticationThreadLocal.remove();
        currentScriptRelease.remove();
        currentScriptTask.remove();
        currentVariables.remove();
    }

    public boolean isInScriptTask() {
        return currentScriptRelease.get() != null;
    }

    public boolean hasScriptTask() {
        return currentScriptTask.get() != null;
    }

    @NonNull
    public Task getCurrentScriptTask() {
        if (!isInScriptTask()) {
            throw new IllegalStateException("Current authentication context is not a script task");
        } else if (currentScriptTask.get() == null) {
            throw new IllegalStateException("Script task not available in this context");
        } else {
            return currentScriptTask.get();
        }
    }

    @NonNull
    public Release getCurrentScriptRelease() {
        if (!isInScriptTask()) {
            throw new IllegalStateException("Current authentication context is not a script task");
        } else {
            return currentScriptRelease.get();
        }
    }

    public boolean hasVariablesHolderForScriptContext() {
        return currentVariables.get() != null;
    }

    @NonNull
    public VariablesHolderForScriptContext getCurrentVariablesHolderForScriptContext() {
        if (!hasVariablesHolderForScriptContext()) {
            throw new IllegalStateException("Current VariablesHolderForScriptContext is null");
        } else {
            return currentVariables.get();
        }
    }

    private String resolveUserScriptPassword(Release release) {
        final var variableMapping = release.getVariableMapping();
        var propName = Release.SCRIPT_USER_PASSWORD_VARIABLE_MAPPING_KEY;

        if (variableMapping != null && variableMapping.containsKey(propName)) {
            // copying variableMapping and scriptUserPassword in case something persists the release downstream
            final var scriptUserPass = release.getScriptUserPassword();
            final var variableMappingCopy = new HashMap<>(variableMapping);
            release.setVariableMapping(variableMappingCopy);

            var varKey = variableMappingCopy.get(propName);
            var usagePoint = CiProperty.of(release, propName)
                    .map(ciProperty -> new VariableMappingUsagePoint(release, propName, ciProperty))
                    .orElseThrow();
            // mutates variableMapping and properties
            ConfigurationVariableService.replaceFromUsagePoint(
                    VariableHelper.withoutVariableSyntax(varKey),
                    usagePoint,
                    asScala(VariableHelper.getAllReleaseVariablesByKeys(release).values()).toSeq()
            );

            var resolvedPass = PasswordEncrypter.getInstance().ensureDecrypted(release.getScriptUserPassword());

            // set to old release state
            release.setScriptUserPassword(scriptUserPass);
            release.setVariableMapping(variableMapping);

            return resolvedPass;
        } else {
            return PasswordEncrypter.getInstance().ensureDecrypted(release.getScriptUserPassword());
        }
    }
}
